/* 
   Unix SMB/CIFS implementation.
   Net_sam_logon info3 helpers
   Copyright (C) Alexander Bokovoy              2002.
   Copyright (C) Andrew Bartlett                2002.
   Copyright (C) Gerald Carter			2003.
   Copyright (C) Tim Potter			2003.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "includes.h"

#define NETSAMLOGON_TDB	"netsamlogon_cache.tdb"

static TDB_CONTEXT *netsamlogon_tdb = NULL;

/***********************************************************************
 open the tdb
 ***********************************************************************/
 
bool netsamlogon_cache_init(void)
{
	if (!netsamlogon_tdb) {
		netsamlogon_tdb = tdb_open_log(lock_path(NETSAMLOGON_TDB), 0,
						   TDB_DEFAULT, O_RDWR | O_CREAT, 0600);
	}

	return (netsamlogon_tdb != NULL);
}


/***********************************************************************
 Shutdown samlogon_cache database
***********************************************************************/

bool netsamlogon_cache_shutdown(void)
{
	if(netsamlogon_tdb)
		return (tdb_close(netsamlogon_tdb) == 0);
		
	return True;
}

/***********************************************************************
 Clear cache getpwnam and getgroups entries from the winbindd cache
***********************************************************************/
void netsamlogon_clear_cached_user(TDB_CONTEXT *tdb, NET_USER_INFO_3 *user)
{
	bool got_tdb = False;
	DOM_SID sid;
	fstring key_str, sid_string;

	/* We may need to call this function from smbd which will not have
           winbindd_cache.tdb open.  Open the tdb if a NULL is passed. */

	if (!tdb) {
		tdb = tdb_open_log(lock_path("winbindd_cache.tdb"), 
				   WINBINDD_CACHE_TDB_DEFAULT_HASH_SIZE,
				   TDB_DEFAULT, O_RDWR, 0600);
		if (!tdb) {
			DEBUG(5, ("netsamlogon_clear_cached_user: failed to open cache\n"));
			return;
		}
		got_tdb = True;
	}

	sid_copy(&sid, &user->dom_sid.sid);
	sid_append_rid(&sid, user->user_rid);

	/* Clear U/SID cache entry */

	fstr_sprintf(key_str, "U/%s", sid_to_fstring(sid_string, &sid));

	DEBUG(10, ("netsamlogon_clear_cached_user: clearing %s\n", key_str));

	tdb_delete(tdb, string_tdb_data(key_str));

	/* Clear UG/SID cache entry */

	fstr_sprintf(key_str, "UG/%s", sid_to_fstring(sid_string, &sid));

	DEBUG(10, ("netsamlogon_clear_cached_user: clearing %s\n", key_str));

	tdb_delete(tdb, string_tdb_data(key_str));

	if (got_tdb)
		tdb_close(tdb);
}

/***********************************************************************
 Store a NET_USER_INFO_3 structure in a tdb for later user 
 username should be in UTF-8 format
***********************************************************************/

bool netsamlogon_cache_store( const char *username, NET_USER_INFO_3 *user )
{
	TDB_DATA 	data;
        fstring 	keystr, tmp;
	prs_struct 	ps;
	bool 		result = False;
	DOM_SID		user_sid;
	time_t		t = time(NULL);
	TALLOC_CTX 	*mem_ctx;
	

	if (!netsamlogon_cache_init()) {
		DEBUG(0,("netsamlogon_cache_store: cannot open %s for write!\n", NETSAMLOGON_TDB));
		return False;
	}

	sid_copy( &user_sid, &user->dom_sid.sid );
	sid_append_rid( &user_sid, user->user_rid );

	/* Prepare key as DOMAIN-SID/USER-RID string */
	slprintf(keystr, sizeof(keystr), "%s", sid_to_fstring(tmp, &user_sid));

	DEBUG(10,("netsamlogon_cache_store: SID [%s]\n", keystr));
	
	/* only Samba fills in the username, not sure why NT doesn't */
	/* so we fill it in since winbindd_getpwnam() makes use of it */
	
	if ( !user->uni_user_name.buffer ) {
		init_unistr2( &user->uni_user_name, username, UNI_STR_TERMINATE );
		init_uni_hdr( &user->hdr_user_name, &user->uni_user_name );
	}
		
	/* Prepare data */
	
	if ( !(mem_ctx = TALLOC_P( NULL, int )) ) {
		DEBUG(0,("netsamlogon_cache_store: talloc() failed!\n"));
		return False;
	}

	prs_init( &ps, RPC_MAX_PDU_FRAG_LEN, mem_ctx, MARSHALL);
	
	{
		uint32 ts = (uint32)t;
		if ( !prs_uint32( "timestamp", &ps, 0, &ts ) )
			return False;
	}
	
	if ( net_io_user_info3("", user, &ps, 0, 3, 0) ) 
	{
		data.dsize = prs_offset( &ps );
		data.dptr = (uint8 *)prs_data_p( &ps );

		if (tdb_store_bystring(netsamlogon_tdb, keystr, data, TDB_REPLACE) != -1)
			result = True;
		
		prs_mem_free( &ps );
	}

	TALLOC_FREE( mem_ctx );
		
	return result;
}

/***********************************************************************
 Retrieves a NET_USER_INFO_3 structure from a tdb.  Caller must 
 free the user_info struct (malloc()'d memory)
***********************************************************************/

NET_USER_INFO_3* netsamlogon_cache_get( TALLOC_CTX *mem_ctx, const DOM_SID *user_sid)
{
	NET_USER_INFO_3	*user = NULL;
	TDB_DATA 	data;
	prs_struct	ps;
        fstring 	keystr, tmp;
	uint32		t;
	
	if (!netsamlogon_cache_init()) {
		DEBUG(0,("netsamlogon_cache_get: cannot open %s for write!\n", NETSAMLOGON_TDB));
		return False;
	}

	/* Prepare key as DOMAIN-SID/USER-RID string */
	slprintf(keystr, sizeof(keystr), "%s", sid_to_fstring(tmp, user_sid));
	DEBUG(10,("netsamlogon_cache_get: SID [%s]\n", keystr));
	data = tdb_fetch_bystring( netsamlogon_tdb, keystr );
	
	if ( data.dptr ) {

		user = TALLOC_ZERO_P(mem_ctx, NET_USER_INFO_3);
		if (user == NULL) {
			return NULL;
		}

		prs_init( &ps, 0, mem_ctx, UNMARSHALL );
		prs_give_memory( &ps, (char *)data.dptr, data.dsize, True );
		
		if ( !prs_uint32( "timestamp", &ps, 0, &t ) ) {
			prs_mem_free( &ps );
			TALLOC_FREE(user);
			return False;
		}
		
		if ( !net_io_user_info3("", user, &ps, 0, 3, 0) ) {
			TALLOC_FREE( user );
		}
			
		prs_mem_free( &ps );

#if 0	/* The netsamlogon cache needs to hang around.  Something about 
	   this feels wrong, but it is the only way we can get all of the
	   groups.  The old universal groups cache didn't expire either.
	   --jerry */
	{
		time_t		now = time(NULL);
		uint32		time_diff;
	   
		/* is the entry expired? */
		time_diff = now - t;
		
		if ( (time_diff < 0 ) || (time_diff > lp_winbind_cache_time()) ) {
			DEBUG(10,("netsamlogon_cache_get: cache entry expired \n"));
			tdb_delete( netsamlogon_tdb, key );
			TALLOC_FREE( user );
		}
#endif
	}
	
	return user;
}

bool netsamlogon_cache_have(const DOM_SID *user_sid)
{
	TALLOC_CTX *mem_ctx = talloc_init("netsamlogon_cache_have");
	NET_USER_INFO_3 *user = NULL;
	bool result;

	if (!mem_ctx)
		return False;

	user = netsamlogon_cache_get(mem_ctx, user_sid);

	result = (user != NULL);

	talloc_destroy(mem_ctx);

	return result;
}
