/*
 * get_ipaddr()
 * Copyright James A. McQuillan, 2000

 * Author: James A. McQuillan <jam@McQuil.com>

 * 2007, Oliver Grawert <ogra@canonical.com>
 * 2008, Ryan Niebur <RyanRyan52@gmail.com>
 *       Scott Balneaves <sbalneav@ltsp.org>
 *       Warren Togami <wtogami@redhat.com>
 *       Toshio Kuratomi
 * 2009, Ryan Niebur <RyanRyan52@gmail.com>

 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, you can find it on the World Wide
 * Web at http://www.gnu.org/copyleft/gpl.html, or write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.

 */

#include <sys/utsname.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <glib/gi18n.h>

#include "ldm.h"

void
get_ipaddr(void)
{
    int numreqs = 10;
    struct ifconf ifc;
    struct ifreq *ifr;	/* netdevice(7) */
    struct ifreq info;
    struct sockaddr_in *sa;

    int skfd, n;

    skfd = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
    if (skfd < 0)
        die("socket");

    /*
     * Get a list of all the interfaces.
     */

    ifc.ifc_buf = NULL;

    while (TRUE) {
        ifc.ifc_len = sizeof(struct ifreq) * numreqs;
	    ifc.ifc_buf = (char *)realloc(ifc.ifc_buf, ifc.ifc_len);
	    if (ifc.ifc_buf == NULL)
            die(_("Out of memory"));

	    if (ioctl(skfd, SIOCGIFCONF, &ifc) < 0) {
            logerr("SIOCGIFCONF");
	        goto out;
	    }

	    if (ifc.ifc_len == (int)sizeof(struct ifreq) * numreqs){
	        /* assume it overflowed and try again */
	        numreqs += 10;
	        continue;
	    }

	    break;
    }

    /*
     * Look for the first interface that has an IP address, is not
     * loopback, and is up.
     */

    ifr = ifc.ifc_req;
    for (n = 0; n < ifc.ifc_len; n += sizeof(struct ifreq)) {
	    if (ifr->ifr_addr.sa_family != AF_INET)
	        continue;

	    strcpy(info.ifr_name, ifr->ifr_name);
	    if (ioctl(skfd, SIOCGIFFLAGS, &info) < 0) {
            logerr("SIOCGIFFLAGS");
	        goto out;
	    }

	    if (!(info.ifr_flags & IFF_LOOPBACK) && (info.ifr_flags & IFF_UP)) {
	        sa = (struct sockaddr_in *) &ifr->ifr_addr;
	        ldm.ipaddr = g_strdup(inet_ntoa(sa->sin_addr));
	        break;
	    }

	    ifr++;
    }

out:
    if (ifc.ifc_buf)
        free(ifc.ifc_buf);

    if (n == ifc.ifc_len)
	    die(_("No configured interface found\n"));
}
