/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-pixbuf-loader.h"

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>

G_DEFINE_TYPE (NlPixbufLoader, nl_pixbuf_loader, G_TYPE_OUTPUT_STREAM);

#define NL_PIXBUF_LOADER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
  NL_TYPE_PIXBUF_LOADER, \
  NlPixbufLoaderPrivate))

struct _NlPixbufLoaderPrivate
{
  GdkPixbufLoader *loader;
};

enum
{
  COMPLETED,

  LAST_SIGNAL
};

static guint32 _loader_signals[LAST_SIGNAL] = { 0 };


static gssize
nl_pixbuf_loader_write_fn (GOutputStream *stream,
                           const void    *buffer,
                           gsize          count,
                           GCancellable  *cancellable,
                           GError       **error)
{
  NlPixbufLoader *self = NL_PIXBUF_LOADER (stream);

  if (gdk_pixbuf_loader_write (self->priv->loader, buffer, count, error))
    return count;
  else
    return -1;
}

static gboolean
nl_pixbuf_loader_flush (GOutputStream *stream,
                        GCancellable  *cancellable,
                        GError       **error)
{
  return TRUE;
}

static gboolean
nl_pixbuf_loader_close_fn (GOutputStream  *stream,
                           GCancellable   *cancellable,
                           GError        **error)
{
  NlPixbufLoader *self = NL_PIXBUF_LOADER (stream);

  return gdk_pixbuf_loader_close (self->priv->loader, error);
}

static void
nl_pixbuf_loader_finalize (GObject *object)
{
  NlPixbufLoaderPrivate *priv;

  priv = NL_PIXBUF_LOADER_GET_PRIVATE (object);

  if (priv->loader)
    {
      g_object_unref (priv->loader);
      priv->loader = NULL;
    }

  G_OBJECT_CLASS (nl_pixbuf_loader_parent_class)->finalize (object);
}


static void
nl_pixbuf_loader_class_init (NlPixbufLoaderClass *klass)
{
  GObjectClass       *obj_class = G_OBJECT_CLASS (klass);
  GOutputStreamClass *out_class = G_OUTPUT_STREAM_CLASS (klass);

  obj_class->finalize = nl_pixbuf_loader_finalize;

  out_class->write_fn = nl_pixbuf_loader_write_fn;
  out_class->flush    = nl_pixbuf_loader_flush;
  out_class->close_fn = nl_pixbuf_loader_close_fn;

  _loader_signals[COMPLETED] =
    g_signal_new ("completed",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlPixbufLoaderClass, completed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  g_type_class_add_private (obj_class, sizeof (NlPixbufLoaderPrivate));
}

static void
nl_pixbuf_loader_init (NlPixbufLoader *loader)
{
  NlPixbufLoaderPrivate *priv;

  priv = loader->priv = NL_PIXBUF_LOADER_GET_PRIVATE (loader);

  priv->loader = gdk_pixbuf_loader_new ();
}

/*
 * Private Methods
 */
static void
on_async_ready (NlPixbufLoader *self,
                GAsyncResult   *res,
                gpointer        null)
{
  NlPixbufLoaderPrivate *priv = self->priv;
  GdkPixbuf *pixbuf;
  GError *error = NULL;
  
  g_output_stream_splice_finish (G_OUTPUT_STREAM (self), res, &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_signal_emit (self, _loader_signals[COMPLETED], 0, NULL);
      g_error_free (error);
    }
  else
    {
      pixbuf = gdk_pixbuf_loader_get_pixbuf (priv->loader);

      g_signal_emit (self, _loader_signals[COMPLETED], 0, pixbuf);

      g_object_unref (pixbuf);
    }
}

/*
 * Public methods
 */
NlPixbufLoader *
nl_pixbuf_loader_new (const gchar *uri)

{
  GFile            *file;
  GFileInputStream *in;
  NlPixbufLoader   *pixbuf_loader;
  
  pixbuf_loader = g_object_new (NL_TYPE_PIXBUF_LOADER,
                                NULL);

  file = g_file_new_for_uri (uri);
  in = g_file_read (file, NULL, NULL);

  g_output_stream_splice_async (G_OUTPUT_STREAM (pixbuf_loader),
                                G_INPUT_STREAM (in),
                                G_OUTPUT_STREAM_SPLICE_CLOSE_TARGET |
                                G_OUTPUT_STREAM_SPLICE_CLOSE_SOURCE,
                                0,
                                NULL,
                                (GAsyncReadyCallback)on_async_ready,
                                NULL);

  g_object_unref (in);
  g_object_unref (file);

  return pixbuf_loader;
}


/*
 * Private methods
 */


/*
 * Public Methods
 */
