/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *             Cody Russell <cody.russell@canonical.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-favorite-gadget.h"

#include <glib.h>
#include <gio/gio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>
#include <libgadget/libgadget.h>

G_DEFINE_TYPE (NlFavoriteGadget, nl_favorite_gadget,
               NL_TYPE_FAVORITE);

#define NL_FAVORITE_GADGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_FAVORITE_GADGET, NlFavoriteGadgetPrivate))

struct _NlFavoriteGadgetPrivate
{
  NlShell           *shell;
  LauncherFavorites *favorites;

  GadgetManager *manager;
  GadgetProxy   *proxy;
  gchar         *uid;
  gchar         *path;

  ClutterActor  *texture;

  GThread *thread;

  gint last_x;
  gint last_y;
};

enum
{
  PROP_0,
  PROP_UID,
  PROP_SHELL
};

/* Forwards */
static void on_queue_redraw (GadgetProxy            *proxy,
                             NlFavoriteGadget *self);
static void on_queue_resize (GadgetProxy            *proxy,
                             NlFavoriteGadget *self);
static void on_open_url     (GadgetProxy            *proxy,
                             const gchar            *url,
                             NlFavoriteGadget *self);
static void on_begin_resize (GadgetProxy            *proxy,
                             GadgetResizePoint       point,
                             NlFavoriteGadget *self);
static void on_begin_drag_move (GadgetProxy            *proxy,
                                NlFavoriteGadget *self);
static void on_remove_me       (GadgetProxy           *proxy,
                                NlFavoriteGadget      *self);

static gboolean on_button_press (ClutterActor           *actor,
                                 ClutterButtonEvent     *event,
                                 NlFavoriteGadget *self);
static gboolean on_button_release (ClutterActor           *actor,
                                   ClutterButtonEvent     *event,
                                   NlFavoriteGadget *self);
static gboolean on_enter_event (ClutterActor           *actor,
                                ClutterCrossingEvent   *event,
                                NlFavoriteGadget *self);
static gboolean on_leave_event (ClutterActor           *actor,
                                ClutterCrossingEvent   *event,
                                NlFavoriteGadget *self);
static gboolean on_motion_event (ClutterActor           *actor,
                                 ClutterMotionEvent     *event,
                                 NlFavoriteGadget *self);
static gboolean on_scroll_event (ClutterActor           *actor,
                                 ClutterScrollEvent     *event,
                                 NlFavoriteGadget *self);
static gboolean on_key_press    (ClutterActor           *actor,
                                 ClutterKeyEvent        *event,
                                 NlFavoriteGadget *self);
static gboolean on_key_release  (ClutterActor           *actor,
                                 ClutterKeyEvent        *event,
                                 NlFavoriteGadget *self);

static void     on_focus_in     (ClutterActor           *actor,
                                 NlFavoriteGadget       *self);
static void     on_focus_out    (ClutterActor           *actor,
                                 NlFavoriteGadget       *self);

static const gchar * nl_favorite_gadget_get_name (NlFavorite *self);
static GdkPixbuf   * nl_favorite_gadget_get_icon (NlFavorite *self);
static void          nl_favorite_gadget_removed   (NlFavorite *self);
static NlFavoriteViewType nl_favorite_gadget_get_view_type (NlFavorite *self);


/* static stuff */
static GMutex *mutex = NULL;

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFavoriteGadgetPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_GADGET (object));
  priv = NL_FAVORITE_GADGET_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      if (priv->uid)
        g_free (priv->uid);
      priv->uid = g_value_dup_string (value);
      break;

    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFavoriteGadgetPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_GADGET (object));
  priv = NL_FAVORITE_GADGET_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      g_value_set_string (value, priv->uid);
      break;

    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_favorite_gadget_finalize (GObject *object)
{
  NlFavoriteGadgetPrivate *priv;

  priv = NL_FAVORITE_GADGET_GET_PRIVATE (object);

  if (priv->proxy)
    {
      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }
  if (priv->uid)
    {
      g_free (priv->uid);
      priv->uid = NULL;
    }
  if (priv->path)
    {
      g_free (priv->path);
      priv->path = NULL;
    }
  if (priv->manager)
    {
      g_object_unref (priv->manager);
      priv->manager = NULL;
    }

  G_OBJECT_CLASS (nl_favorite_gadget_parent_class)->finalize (object);
}

static gboolean
setup_gadget (NlFavoriteGadget *gadget)
{
  NlFavoriteGadgetPrivate *priv = NL_FAVORITE_GADGET_GET_PRIVATE (gadget);

  g_thread_join (priv->thread);

  g_signal_connect (priv->proxy, "queue-redraw",
                    G_CALLBACK (on_queue_redraw), gadget);
  g_signal_connect (priv->proxy, "queue-resize",
                    G_CALLBACK (on_queue_resize), gadget);
  g_signal_connect (priv->proxy, "open-url",
                    G_CALLBACK (on_open_url), gadget);
  g_signal_connect (priv->proxy, "begin-resize",
                    G_CALLBACK (on_begin_resize), gadget);
  g_signal_connect (priv->proxy, "begin-drag-move",
                    G_CALLBACK (on_begin_drag_move), gadget);
  g_signal_connect (priv->proxy, "remove-me",
                    G_CALLBACK (on_remove_me), gadget);

  priv->texture = clutter_cairo_texture_new (200, 200);
  clutter_container_add_actor (CLUTTER_CONTAINER (gadget), priv->texture);
  clutter_actor_set_reactive (priv->texture, TRUE);
  clutter_actor_show (priv->texture);

  g_signal_connect (priv->texture, "button-press-event",
                    G_CALLBACK (on_button_press), gadget);
  g_signal_connect (priv->texture, "button-release-event",
                    G_CALLBACK (on_button_release), gadget);
  g_signal_connect (priv->texture, "enter-event",
                    G_CALLBACK (on_enter_event), gadget);
  g_signal_connect (priv->texture, "leave-event",
                    G_CALLBACK (on_leave_event), gadget);
  g_signal_connect (priv->texture, "motion-event",
                    G_CALLBACK (on_motion_event), gadget);
  g_signal_connect (priv->texture, "scroll-event",
                    G_CALLBACK (on_scroll_event), gadget);
  g_signal_connect (priv->texture, "key-press-event",
                    G_CALLBACK (on_key_press), gadget);
  g_signal_connect (priv->texture, "key-release-event",
                    G_CALLBACK (on_key_release), gadget);
  g_signal_connect (priv->texture, "key-focus-in",
                    G_CALLBACK (on_focus_in), gadget);
  g_signal_connect (priv->texture, "key-focus-out",
                    G_CALLBACK (on_focus_out), gadget);

  on_queue_resize (priv->proxy, (NlFavoriteGadget*)gadget);
  on_queue_redraw (priv->proxy, (NlFavoriteGadget*)gadget);

  return FALSE;
}

static gpointer
create_gadget (NlFavoriteGadget *gadget)
{
  NlFavoriteGadgetPrivate *priv = NL_FAVORITE_GADGET_GET_PRIVATE (gadget);

  g_mutex_lock (mutex);

  priv->proxy = gadget_manager_load_gadget (priv->manager,
                                            priv->path,
                                            priv->uid);
  if (!priv->proxy)
    {
      g_debug ("Unable to load gadget %s path %s", priv->uid, priv->path);
    }

  g_idle_add ((GSourceFunc)setup_gadget, gadget);

  g_mutex_unlock (mutex);

  return NULL;
}

static void
nl_favorite_gadget_constructed (GObject *object)
{
  NlFavoriteGadgetPrivate *priv;

  if (!mutex)
    mutex = g_mutex_new ();

  priv = NL_FAVORITE_GADGET_GET_PRIVATE (object);
  priv->favorites = launcher_favorites_get_default ();

  priv->manager = gadget_manager_get_default ();
  gadget_manager_load_sources (priv->manager);

  priv->path = launcher_favorites_get_string (priv->favorites,
               priv->uid,
               "path");
  if (!gadget_manager_can_handle_path (priv->manager, priv->path))
    {
      g_warning ("Unable to load gadget %s, cannot handle type", priv->path);
      return;
    }

  priv->thread = g_thread_create ((GThreadFunc)create_gadget,
                                  object,
                                  TRUE,
                                  NULL);
}

static void
get_preferred_width (ClutterActor *self,
                     gfloat   for_height,
                     gfloat  *min_width,
                     gfloat  *nat_width)
{
  NlFavoriteGadgetPrivate *priv = NL_FAVORITE_GADGET (self)->priv;
  gint width;

  if (!priv->proxy)
    return;

  width = gadget_proxy_get_width (priv->proxy);

  if (width < 1)
    width = 1;

  if (min_width)
    *min_width = (gfloat) (width);
  if (nat_width)
    *nat_width = (gfloat) (width);
}

static void
get_preferred_height (ClutterActor *self,
                      gfloat   for_width,
                      gfloat  *min_height,
                      gfloat  *nat_height)
{
  NlFavoriteGadgetPrivate *priv = NL_FAVORITE_GADGET (self)->priv;
  gint height;

  if (!priv->proxy)
    return;

  height = gadget_proxy_get_height (priv->proxy);

  if (height < 1)
    height = 1;

  if (min_height)
    *min_height = (gfloat) (height);
  if (nat_height)
    *nat_height = (gfloat) (height);
}

static void
allocate (ClutterActor          *self,
          const ClutterActorBox *box,
          ClutterAllocationFlags flags)
{
  NlFavoriteGadgetPrivate *priv;
  CtkPadding      padding;
  ClutterActorBox child_box;

  priv = NL_FAVORITE_GADGET (self)->priv;

  CLUTTER_ACTOR_CLASS (nl_favorite_gadget_parent_class)->allocate (self,
      box,
      flags);

  if (!CLUTTER_IS_ACTOR (ctk_bin_get_child (CTK_BIN (self))))
    return;

  ctk_actor_get_padding (CTK_ACTOR (self), &padding);

  child_box.x1 = padding.left;
  child_box.x2 = (box->x2 - box->x1) - padding.left - padding.right;
  child_box.y1 = padding.top;
  child_box.y2 = (box->y2 - box->y1) - padding.top - padding.bottom;

  clutter_actor_allocate (priv->texture, &child_box, flags);
}


static void
nl_favorite_gadget_class_init (NlFavoriteGadgetClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  NlFavoriteClass   *fav_class = NL_FAVORITE_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_favorite_gadget_finalize;
  obj_class->constructed  = nl_favorite_gadget_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  act_class->get_preferred_width  = get_preferred_width;
  act_class->get_preferred_height = get_preferred_height;
  act_class->allocate             = allocate;

  fav_class->get_name      = nl_favorite_gadget_get_name;
  fav_class->get_icon      = nl_favorite_gadget_get_icon;
  fav_class->removed       = nl_favorite_gadget_removed;
  fav_class->get_view_type = nl_favorite_gadget_get_view_type;

  pspec = g_param_spec_string ("uid", "uid", "uid",  "",
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_UID, pspec);

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFavoriteGadgetPrivate));
}

static void
nl_favorite_gadget_init (NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;

  priv = self->priv = NL_FAVORITE_GADGET_GET_PRIVATE (self);
}

/*
 * Public methods
 */
NlFavorite *
nl_favorite_gadget_new (const gchar *uid, NlShell *shell)

{
  NlFavorite *favorite_gadget;

  favorite_gadget = g_object_new (NL_TYPE_FAVORITE_GADGET,
                                  "uid", uid,
                                  "shell", shell,
                                  NULL);

  return favorite_gadget;
}

/*
 * Private methods
 */
static void
on_queue_redraw (GadgetProxy            *proxy,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  cairo_t *cr;

  g_return_if_fail (NL_IS_FAVORITE_GADGET (self));
  priv = self->priv;

  cr = clutter_cairo_texture_create (CLUTTER_CAIRO_TEXTURE (priv->texture));

  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

  gadget_cairo_draw (GADGET_CAIRO (priv->proxy), cr);

  cairo_destroy (cr);
}

static void
on_queue_resize (GadgetProxy            *proxy,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  gint width;
  gint height;

  g_return_if_fail (NL_IS_FAVORITE_GADGET (self));
  priv = self->priv;

  width = gadget_proxy_get_width (proxy);
  height = gadget_proxy_get_height (proxy);

  if (width == 0)
    width = 1;
  if (height == 0)
    height = 1;

  clutter_cairo_texture_set_surface_size (CLUTTER_CAIRO_TEXTURE(priv->texture),
                                          width, height);
  clutter_actor_set_size (priv->texture, width, height);
  clutter_actor_set_size (CLUTTER_ACTOR (self), width, height);

  clutter_actor_queue_relayout (CLUTTER_ACTOR (self));
  on_queue_redraw (proxy, self);
}

static void
on_open_url (GadgetProxy            *proxy,
             const gchar            *url,
             NlFavoriteGadget *self)
{
  g_debug ("open url %s", url);
}

static void
on_begin_resize (GadgetProxy            *proxy,
                 GadgetResizePoint       point,
                 NlFavoriteGadget *self)
{
  g_debug ("RESIZE: %d", point);
}

static void
on_begin_drag_move (GadgetProxy            *proxy,
                    NlFavoriteGadget *self)
{
  ClutterMotionEvent event;

  event.x = self->priv->last_x;
  event.y = self->priv->last_y;

  g_signal_emit_by_name (self, "begin-drag-move", &event);
}

static void
on_remove_me (GadgetProxy *proxy, NlFavoriteGadget *self)
{
  g_signal_emit_by_name (self, "remove-me");
}

/*
 * EVENTS
 */
static gboolean
on_button_press (ClutterActor           *actor,
                 ClutterButtonEvent     *event,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventButton e;
  gfloat         realx = 0;
  gfloat         realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);
  priv->last_x = event->x;
  priv->last_y = event->y;

  e.type = event->click_count == 1 ? GDK_BUTTON_PRESS :
           event->click_count == 2 ? GDK_2BUTTON_PRESS :
           GDK_3BUTTON_PRESS;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = event->modifier_state;
  e.button = event->button;
  e.device = NULL;
  e.x_root = event->x;
  e.y_root = event->y;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  clutter_actor_grab_key_focus (actor);

  g_signal_emit_by_name (self, "active");

  return TRUE;
}

static gboolean
on_button_release (ClutterActor           *actor,
                   ClutterButtonEvent     *event,
                   NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventButton e;
  gfloat         realx = 0;
  gfloat         realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);

  e.type = GDK_BUTTON_RELEASE;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = event->modifier_state;
  e.button = event->button;
  e.device = NULL;
  e.x_root = event->x;
  e.y_root = event->y;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}


static gboolean
on_enter_event (ClutterActor           *actor,
                ClutterCrossingEvent   *event,
                NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventCrossing e;
  gfloat           realx = 0;
  gfloat           realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);

  e.type = GDK_ENTER_NOTIFY;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = 0;
  e.x_root = event->x;
  e.y_root = event->y;
  e.mode = GDK_CROSSING_NORMAL;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static gboolean
on_leave_event (ClutterActor           *actor,
                ClutterCrossingEvent   *event,
                NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventCrossing e;
  gfloat           realx = 0;
  gfloat           realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);

  e.type = GDK_LEAVE_NOTIFY;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = 0;
  e.x_root = event->x;
  e.y_root = event->y;
  e.mode = GDK_CROSSING_NORMAL;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static gboolean
on_motion_event (ClutterActor           *actor,
                 ClutterMotionEvent     *event,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventMotion e;
  gfloat         realx = 0;
  gfloat         realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);

  e.type = GDK_MOTION_NOTIFY;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = event->modifier_state;
  e.x_root = event->x;
  e.y_root = event->y;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static gboolean
on_scroll_event (ClutterActor           *actor,
                 ClutterScrollEvent     *event,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventScroll e;
  gfloat         realx = 0;
  gfloat         realy = 0;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  clutter_actor_get_transformed_position (actor, &realx, &realy);

  e.type = GDK_SCROLL;
  e.time = event->time;
  e.x = event->x - realx;
  e.y = event->y - realy;
  e.state = event->modifier_state;
  e.x_root = event->x;
  e.y_root = event->y;
  e.direction = event->direction == CLUTTER_SCROLL_UP ? GDK_SCROLL_UP :
                event->direction == CLUTTER_SCROLL_DOWN ? GDK_SCROLL_DOWN :
                event->direction == CLUTTER_SCROLL_LEFT ? GDK_SCROLL_LEFT :
                GDK_SCROLL_RIGHT;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static gboolean
on_key_press    (ClutterActor           *actor,
                 ClutterKeyEvent        *event,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventKey e;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  e.window = nl_shell_get_toplevel (priv->shell)->window;
  e.type = GDK_KEY_PRESS;
  e.time = event->time;
  e.state = event->modifier_state;
  e.keyval = event->keyval;
  e.hardware_keycode = event->hardware_keycode;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static gboolean
on_key_release  (ClutterActor           *actor,
                 ClutterKeyEvent        *event,
                 NlFavoriteGadget *self)
{
  NlFavoriteGadgetPrivate *priv;
  GdkEventKey e;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), FALSE);
  priv = self->priv;

  e.window = nl_shell_get_toplevel (priv->shell)->window;
  e.type = GDK_KEY_RELEASE;
  e.time = event->time;
  e.state = event->modifier_state;
  e.keyval = event->keyval;
  e.hardware_keycode = event->hardware_keycode;

  gadget_proxy_event (priv->proxy, (GdkEvent *)&e);

  return TRUE;
}

static void
on_focus_in (ClutterActor           *actor,
             NlFavoriteGadget       *self)
{
  g_return_if_fail (NL_IS_FAVORITE_GADGET (self));

  gadget_proxy_set_focused (self->priv->proxy, TRUE);
}

static void
on_focus_out (ClutterActor           *actor,
              NlFavoriteGadget       *self)
{
  g_return_if_fail (NL_IS_FAVORITE_GADGET (self));

  gadget_proxy_set_focused (self->priv->proxy, FALSE);
}

static const gchar *
nl_favorite_gadget_get_name (NlFavorite *self)
{
  NlFavoriteGadgetPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), NULL);
  priv = NL_FAVORITE_GADGET (self)->priv;

  return gadget_proxy_get_name (priv->proxy);
}

static GdkPixbuf *
nl_favorite_gadget_get_icon (NlFavorite *self)
{
  NlFavoriteGadgetPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_GADGET (self), NULL);
  priv = NL_FAVORITE_GADGET (self)->priv;

  return gadget_proxy_get_icon (priv->proxy);
}

static NlFavoriteViewType
nl_favorite_gadget_get_view_type (NlFavorite *self)
{
  return NL_FAVORITE_VIEW_FLOATING;
}

static void
nl_favorite_gadget_removed (NlFavorite *self)
{
  NlFavoriteGadgetPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_GADGET (self));
  priv = NL_FAVORITE_GADGET (self)->priv;

  gadget_proxy_removed (priv->proxy);

  launcher_favorites_remove_favorite (priv->favorites, priv->uid);
}

/*
 * Public Methods
 */

