/*  This file is part of the KDE project
    Copyright (C) 2007 Kevin Ottens <ervin@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include "kfileplacesitem_p.h"
#include "kfileplacesmodel.h"

#include <QtCore/QDateTime>

#include <kbookmarkmanager.h>
#include <solid/storageaccess.h>
#include <solid/storagevolume.h>



KFilePlacesItem::KFilePlacesItem(KBookmarkManager *manager,
                                 const QString &address,
                                 const QString &udi)
    : m_manager(manager)
{
    m_bookmark = m_manager->findByAddress(address);

    if (udi.isEmpty() && m_bookmark.metaDataItem("ID").isEmpty()) {
        m_bookmark.setMetaDataItem("ID", generateNewId());
    } else if (!udi.isEmpty()) {
        Solid::Device dev(udi);
        connect(dev.as<Solid::StorageAccess>(), SIGNAL(accessibilityChanged(bool, const QString &)),
                this, SLOT(onAccessibilityChanged()));
    }
}

KFilePlacesItem::~KFilePlacesItem()
{
}

QString KFilePlacesItem::id() const
{
    if (isDevice()) {
        return bookmark().metaDataItem("UDI");
    } else {
        return bookmark().metaDataItem("ID");
    }
}

bool KFilePlacesItem::isDevice() const
{
    return !bookmark().metaDataItem("UDI").isEmpty();
}

KBookmark KFilePlacesItem::bookmark() const
{
    return m_bookmark;
}

Solid::Device KFilePlacesItem::device() const
{
    return Solid::Device(bookmark().metaDataItem("UDI"));
}

QVariant KFilePlacesItem::data(int role) const
{
    QVariant returnData;

    if (isDevice() && role!=KFilePlacesModel::HiddenRole && role!=Qt::BackgroundRole) {
        returnData = deviceData(role);
    } else {
        returnData = bookmarkData(role);
    }

    return returnData;
}

QVariant KFilePlacesItem::bookmarkData(int role) const
{
    KBookmark b = bookmark();

    if (b.isNull()) return QVariant();

    switch (role)
    {
    case Qt::DisplayRole:
        return b.text();
    case Qt::DecorationRole:
        return KIcon(b.icon());
    case Qt::BackgroundRole:
        if (b.metaDataItem("IsHidden")=="true") {
            return Qt::lightGray;
        } else {
            return QVariant();
        }
    case KFilePlacesModel::UrlRole:
        return QUrl(b.url());
    case KFilePlacesModel::SetupNeededRole:
        return false;
    case KFilePlacesModel::HiddenRole:
        return b.metaDataItem("IsHidden")=="true";
    default:
        return QVariant();
    }
}

QVariant KFilePlacesItem::deviceData(int role) const
{
    Solid::Device d = device();

    if (d.isValid()) {
        const Solid::StorageAccess *access = d.as<Solid::StorageAccess>();
        const Solid::StorageVolume *volume = d.as<Solid::StorageVolume>();
        QStringList overlays;

        switch (role)
        {
        case Qt::DisplayRole:
            return d.product();
        case Qt::DecorationRole:
            if (access->isAccessible()) {
                overlays << "emblem-mounted";
            } else {
                overlays << QString(); // We have to guarantee the placement of the next emblem
            }
            if (volume && volume->usage()==Solid::StorageVolume::Encrypted) {
                overlays << "security-high";
            }
            return KIcon(d.icon(), 0, overlays);
        case KFilePlacesModel::UrlRole:
            if (access) {
                return QUrl(KUrl(access->filePath()));
            } else {
                return QVariant();
            }
        case KFilePlacesModel::SetupNeededRole:
            if (access) {
                return !access->isAccessible();
            } else {
                return QVariant();
            }
        default:
            return QVariant();
        }
    } else {
        return QVariant();
    }
}

KBookmark KFilePlacesItem::createBookmark(KBookmarkManager *manager,
                                          const QString &label,
                                          const KUrl &url,
                                          const QString &iconName)
{
    KBookmarkGroup root = manager->root();
    KBookmark bookmark = root.addBookmark(label, url, iconName);
    bookmark.setMetaDataItem("ID", generateNewId());

    return bookmark;
}

KBookmark KFilePlacesItem::createDeviceBookmark(KBookmarkManager *manager,
                                                const QString &udi)
{
    KBookmarkGroup root = manager->root();
    KBookmark bookmark = root.createNewSeparator();
    bookmark.setMetaDataItem("UDI", udi);

    return bookmark;
}

QString KFilePlacesItem::generateNewId()
{
    static int count = 0;

    return QString::number(count++);

//    return QString::number(QDateTime::currentDateTime().toTime_t())
//         + '/' + QString::number(qrand());
}

void KFilePlacesItem::onAccessibilityChanged()
{
    emit deviceChanged(id());
}

#include "kfileplacesitem_p.moc"
