/*  This file is part of the KDE project
    Copyright (C) 2000 Matej Koss <koss@miesto.sk>
    Copyright (C) 2007 Kevin Ottens <ervin@kde.org>
    Copyright (C) 2007 Rafael Fernández López <ereslibre@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include "kwidgetjobtracker.h"
#include "kwidgetjobtracker_p.h"

#include <QProcess>
#include <QTimer>
#include <QLabel>
#include <QCheckBox>
#include <QProgressBar>
#include <QVBoxLayout>
#include <QGridLayout>

#include <kurl.h>
#include <kpushbutton.h>
#include <klineedit.h>
#include <kguiitem.h>
#include <kiconloader.h>
#include <kdialog.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <klocale.h>
#include <kwindowsystem.h>

void KWidgetJobTracker::Private::_k_slotShowProgressWidget()
{
    if (progressWidgetsToBeShown.isEmpty()) {
        return;
    }

    KJob *job = progressWidgetsToBeShown.dequeue();

    // If the job has been unregistered before reaching this point, widget will
    // return 0.
    QWidget *widget = q->widget(job);

    if (widget) {
        widget->show();
    }
}

KWidgetJobTracker::KWidgetJobTracker(QWidget *parent)
    : KAbstractWidgetJobTracker(parent), d(new Private(parent, this))
{
}

KWidgetJobTracker::~KWidgetJobTracker()
{
    delete d;
}

QWidget *KWidgetJobTracker::widget(KJob *job)
{
    if (!d->progressWidget.contains(job)) {
        return 0;
    }

    return d->progressWidget[job];
}

void KWidgetJobTracker::registerJob(KJob *job)
{
    KAbstractWidgetJobTracker::registerJob(job);

    if (d->progressWidget.contains(job)) {
        return;
    }

    Private::ProgressWidget *vi = new Private::ProgressWidget(job, this, d->parent);
    d->progressWidget.insert(job, vi);
    d->progressWidgetsToBeShown.enqueue(job);

    QTimer::singleShot(500, this, SLOT(_k_slotShowProgressWidget()));
}

void KWidgetJobTracker::unregisterJob(KJob *job)
{
    KAbstractWidgetJobTracker::unregisterJob(job);

    if (!d->progressWidget.contains(job)) {
        return;
    }

    if (!d->progressWidget[job]->keepOpenChecked) {
        delete d->progressWidget[job];
    }

    d->progressWidget.remove(job);
}

bool KWidgetJobTracker::keepOpen(KJob *job) const
{
    if (!d->progressWidget.contains(job)) {
        return false;
    }

    return d->progressWidget[job]->keepOpen();
}


void KWidgetJobTracker::infoMessage(KJob *job, const QString &plain, const QString &rich)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->infoMessage(plain, rich);
}

void KWidgetJobTracker::description(KJob *job, const QString &title,
                                    const QPair<QString, QString> &field1,
                                    const QPair<QString, QString> &field2)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->description(title, field1, field2);
}

void KWidgetJobTracker::totalAmount(KJob *job, KJob::Unit unit, qulonglong amount)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->totalAmount(unit, amount);
}

void KWidgetJobTracker::processedAmount(KJob *job, KJob::Unit unit, qulonglong amount)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->processedAmount(unit, amount);
}

void KWidgetJobTracker::percent(KJob *job, unsigned long percent)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->percent(percent);
}

void KWidgetJobTracker::speed(KJob *job, unsigned long value)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->speed(value);
}

void KWidgetJobTracker::slotClean(KJob *job)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->slotClean();
}

void KWidgetJobTracker::suspended(KJob *job)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->suspended();
}

void KWidgetJobTracker::resumed(KJob *job)
{
    if (!d->progressWidget.contains(job)) {
        return;
    }

    d->progressWidget[job]->resumed();
}

bool KWidgetJobTracker::Private::ProgressWidget::keepOpen() const
{
    return keepOpenChecked;
}


void KWidgetJobTracker::Private::ProgressWidget::infoMessage(const QString &plain, const QString &/*rich*/)
{
    speedLabel->setText(plain);
    speedLabel->setAlignment(speedLabel->alignment() & ~Qt::TextWordWrap);
}

void KWidgetJobTracker::Private::ProgressWidget::description(const QString &title,
                                                                const QPair<QString, QString> &field1,
                                                                const QPair<QString, QString> &field2)
{
    if (caption.isEmpty() ) {
        setWindowTitle(title);
        caption = title;
    }

    sourceInvite->setText(field1.first);
    sourceEdit->setText(field1.second);

    if (field2.first.isEmpty()) {
        setDestVisible(false);
    } else {
        setDestVisible(true);
        checkDestination(KUrl(field2.second));
        destInvite->setText(field2.first);
        destEdit->setText(field2.second);
    }
}

void KWidgetJobTracker::Private::ProgressWidget::totalAmount(KJob::Unit unit, qulonglong amount)
{
    switch(unit)
    {
    case KJob::Bytes:
        totalSizeKnown = true;
        // size is measured in bytes
        if (totalSize == amount)
            return;
        totalSize = amount;
        if (startTime.isNull())
            startTime.start();
        break;

    case KJob::Files:
        if (totalFiles == amount)
            return;
        totalFiles = amount;
        showTotals();
        break;

    case KJob::Directories:
        if (totalDirs == amount)
            return;
        totalDirs = amount;
        showTotals();
        break;
    }
}

void KWidgetJobTracker::Private::ProgressWidget::processedAmount(KJob::Unit unit, qulonglong amount)
{
    QString tmp;

    switch(unit)
    {
    case KJob::Bytes:
        if (processedSize == amount)
            return;
        processedSize = amount;

        if (totalSizeKnown) {
            tmp = i18n( "%1 of %2 complete",
                        KGlobal::locale()->formatByteSize(amount),
                        KGlobal::locale()->formatByteSize(totalSize));
        } else {
            tmp = KGlobal::locale()->formatByteSize(amount);
        }
        sizeLabel->setText(tmp);
        if (!totalSizeKnown) // update jumping progressbar
            progressBar->setValue(amount);
        break;

    case KJob::Directories:
        if (processedDirs == amount)
            return;
        processedDirs = amount;

        tmp = i18np("%2 / %1 folder", "%2 / %1 folders", totalDirs,  processedDirs);
        tmp += "   ";
        tmp += i18np("%2 / %1 file", "%2 / %1 files", totalFiles,  processedFiles);
        progressLabel->setText(tmp);
        break;

    case KJob::Files:
        if (processedFiles == amount)
            return;
        processedFiles = amount;

        if (totalDirs > 1) {
            tmp = i18np("%2 / %1 folder", "%2 / %1 folders", totalDirs,  processedDirs);
            tmp += "   ";
        }
        tmp += i18np("%2 / %1 file", "%2 / %1 files", totalFiles,  processedFiles);
        progressLabel->setText(tmp);
    }
}

void KWidgetJobTracker::Private::ProgressWidget::percent(unsigned long percent)
{
    QString title = caption + " (";

    if (totalSizeKnown) {
        title += i18n("%1 % of %2 ", percent,
                      KGlobal::locale()->formatByteSize(totalSize));
    } else if (totalFiles) {
        title += i18np("%2 % of 1 file", "%2 % of %1 files", totalFiles, percent);
    } else {
        title += i18n("%1 %", percent);
    }

    title += ')';

    progressBar->setMaximum(100);
    progressBar->setValue(percent);
    setWindowTitle(title);
}

void KWidgetJobTracker::Private::ProgressWidget::speed(unsigned long value)
{
    if (value == 0) {
        speedLabel->setText(i18n("Stalled"));
    } else {
        const QString speedStr = KGlobal::locale()->formatByteSize(value);
        if (totalSizeKnown) {
            const int remaining = 1000*(totalSize - processedSize)/value;
            speedLabel->setText(i18n("%1/s ( %2 remaining )", speedStr,
                                     KGlobal::locale()->formatDuration(remaining)));
        } else { // total size is not known (#24228)
            speedLabel->setText(i18nc("speed in bytes per second", "%1/s", speedStr));
        }
    }
}

void KWidgetJobTracker::Private::ProgressWidget::slotClean()
{
    if (!keepOpenChecked) {
        hide();
    } else {
        percent(100);
        cancelClose->setGuiItem(KStandardGuiItem::close());
        openFile->setEnabled(true);
        if (!totalSizeKnown || totalSize < processedSize)
            totalSize = processedSize;
        processedAmount(KJob::Bytes, totalSize);
        keepOpenCheck->setEnabled(false);
        if (!startTime.isNull()) {
            int s = startTime.elapsed();
            if (!s)
                s = 1;
            speedLabel->setText(i18n("%1/s (done)",
                                        KGlobal::locale()->formatByteSize(1000 * totalSize / s)));
        }
        tracker->setAutoDelete(job, true);
    }
}

void KWidgetJobTracker::Private::ProgressWidget::suspended()
{
    pauseButton->setText(i18n("Resume"));
    suspendedProperty = true;
}

void KWidgetJobTracker::Private::ProgressWidget::resumed()
{
    pauseButton->setText(i18n("Pause"));
    suspendedProperty = false;
}

void KWidgetJobTracker::Private::ProgressWidget::closeEvent(QCloseEvent *event)
{
    if (tracker->stopOnClose(job)) {
        tracker->slotStop(job);
    } else if (tracker->autoDelete(job)) {
        deleteLater();
    } else {
        tracker->slotClean(job);
    }

    QWidget::closeEvent(event);
}

void KWidgetJobTracker::Private::ProgressWidget::init()
{
    // Set a useful icon for this window!
    KWindowSystem::setIcons( winId(),
                             KIconLoader::global()->loadIcon( "document-save", KIconLoader::NoGroup, 32 ),
                             KIconLoader::global()->loadIcon( "document-save", KIconLoader::NoGroup, 16 ) );

    QVBoxLayout *topLayout = new QVBoxLayout(this);
    topLayout->setMargin(KDialog::marginHint());
    topLayout->setSpacing(KDialog::spacingHint() );
    topLayout->addStrut( 360 );   // makes dlg at least that wide

    QGridLayout *grid = new QGridLayout();
    topLayout->addLayout(grid);
    grid->addItem(new QSpacerItem(KDialog::spacingHint(),0),0,1); //addColSpacing(1, KDialog::spacingHint());
    // filenames or action name
    sourceInvite = new QLabel(i18n("Source:"), this);
    grid->addWidget(sourceInvite, 0, 0);

    sourceEdit = new KLineEdit(this);
    sourceEdit->setReadOnly(true);
    sourceEdit->setSqueezedTextEnabled(true);
    grid->addWidget(sourceEdit, 0, 2);

    destInvite = new QLabel(i18n("Destination:"), this);
    grid->addWidget(destInvite, 1, 0);

    destEdit = new KLineEdit(this);
    destEdit->setReadOnly (true);
    destEdit->setSqueezedTextEnabled(true);
    grid->addWidget(destEdit, 1, 2);

    QHBoxLayout *progressHBox = new QHBoxLayout();
    topLayout->addLayout(progressHBox);

    progressBar = new QProgressBar(this);
    progressBar->setMaximum(0); // want a jumping progress bar if percent is not emitted
    progressHBox->addWidget(progressBar);

    suspendedProperty = false;
    pauseButton = new KPushButton(i18n("Pause"), this);
    QObject::connect(pauseButton, SIGNAL(clicked()),
                     this, SLOT(_k_pauseResumeClicked()));
    progressHBox->addWidget(pauseButton);

    // processed info
    QHBoxLayout *hBox = new QHBoxLayout();
    topLayout->addLayout(hBox);

    sizeLabel = new QLabel(this);
    hBox->addWidget(sizeLabel);

    resumeLabel = new QLabel(this);
    hBox->addWidget(resumeLabel);

    progressLabel = new QLabel(this);
/*    progressLabel->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                            QSizePolicy::Preferred));*/
    progressLabel->setAlignment(Qt::AlignRight);
    hBox->addWidget(progressLabel);

    hBox = new QHBoxLayout();
    topLayout->addLayout(hBox);

    speedLabel = new QLabel(this);
    hBox->addWidget(speedLabel, 1);

    QFrame *line = new QFrame(this);
    line->setFrameShape(QFrame::HLine);
    line->setFrameShadow(QFrame::Sunken);
    topLayout->addWidget(line);

    keepOpenCheck = new QCheckBox(i18n("&Keep this window open after transfer is complete"), this);
    QObject::connect(keepOpenCheck, SIGNAL(toggled(bool)),
                     this, SLOT(_k_keepOpenToggled(bool)));
    topLayout->addWidget(keepOpenCheck);
    keepOpenCheck->hide();

    hBox = new QHBoxLayout();
    topLayout->addLayout(hBox);

    openFile = new KPushButton(i18n("Open &File"), this);
    QObject::connect(openFile, SIGNAL(clicked()),
                     this, SLOT(_k_openFile()));
    hBox->addWidget(openFile);
    openFile->setEnabled(false);
    openFile->hide();

    openLocation = new KPushButton(i18n("Open &Destination"), this);
    QObject::connect(openLocation, SIGNAL(clicked()),
                     this, SLOT(_k_openLocation()));
    hBox->addWidget(openLocation);
    openLocation->hide();

    hBox->addStretch(1);

    cancelClose = new KPushButton(KStandardGuiItem::cancel(), this);
    QObject::connect(cancelClose, SIGNAL(clicked()),
                     this, SLOT(_k_stop()));
    hBox->addWidget(cancelClose);

    resize(sizeHint());
    setMaximumHeight(sizeHint().height());

    keepOpenChecked = false;
    setWindowTitle(i18n("Progress Dialog")); // show something better than kuiserver
}

void KWidgetJobTracker::Private::ProgressWidget::showTotals()
{
    // Show the totals in the progress label, if we still haven't
    // processed anything. This is useful when the stat'ing phase
    // of CopyJob takes a long time (e.g. over networks).
    if (processedFiles == 0 && processedDirs == 0)
    {
        QString tmps;
        if (totalDirs > 1)
            // that we have a singular to translate looks weired but is only logical
            tmps = i18np("%1 folder", "%1 folders", totalDirs) + "   ";
        tmps += i18np("%1 file", "%1 files", totalFiles);
        progressLabel->setText( tmps );
    }
}

void KWidgetJobTracker::Private::ProgressWidget::setDestVisible(bool visible)
{
    // We can't hide the destInvite/destEdit labels,
    // because it screws up the QGridLayout.
    if (visible)
    {
        destInvite->show();
        destEdit->show();
    }
    else
    {
        destInvite->hide();
        destEdit->hide();
        destInvite->setText( QString() );
        destEdit->setText( QString() );
    }
}

void KWidgetJobTracker::Private::ProgressWidget::checkDestination(const KUrl &dest)
{
    bool ok = true;

    if (dest.isLocalFile()) {
        QString path = dest.path( KUrl::RemoveTrailingSlash );
        QStringList tmpDirs = KGlobal::dirs()->resourceDirs( "tmp" );
        for (QStringList::Iterator it = tmpDirs.begin() ; ok && it != tmpDirs.end() ; ++it)
            if (path.contains(*it))
                ok = false; // it's in the tmp resource
    }

    if (ok) {
        openFile->show();
        openLocation->show();
        keepOpenCheck->show();
        location=dest;
    }
}

void KWidgetJobTracker::Private::ProgressWidget::_k_keepOpenToggled(bool keepOpen)
{
    keepOpenChecked = keepOpen;
}

void KWidgetJobTracker::Private::ProgressWidget::_k_openFile()
{
    QProcess::startDetached("konqueror", QStringList() << location.prettyUrl());
}

void KWidgetJobTracker::Private::ProgressWidget::_k_openLocation()
{
    location.setFileName("");
    _k_openFile();
}

void KWidgetJobTracker::Private::ProgressWidget::_k_pauseResumeClicked()
{
    if ( !suspendedProperty ) {
        tracker->slotSuspend(job);
    } else {
        tracker->slotResume(job);
    }
}

void KWidgetJobTracker::Private::ProgressWidget::_k_stop()
{
    tracker->slotStop(job);
    tracker->widget(job)->deleteLater();
}

#include "kwidgetjobtracker.moc"
#include "kwidgetjobtracker_p.moc"
