#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2011-2012 Canonical, Ltd.
#
# Authors:
#  Ugo Riboni <ugo.riboni@canonical.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import json
from twisted.internet import defer, protocol

from dbus_proxy import DBUSProxy
from client_tracker import tracker
from whitelist_manager import whitelist

import dbus

class Message(object):
   _msg_types = {
      # registration messages (always coming from JS only)
      'register_on_bus': [('bus_name', ''), ('object_name', '')],
      'register_method': [('interface', ''), ('name', ''), ('in_sig', ''), ('out_sig', '')],

      # objects in JS want to call objects on the dbus
      'js2dbus_call': [('call_id', -1), ('bus_name', ''), ('object_name', ''),
                       ('interface', ''), ('method', ''), ('args', [])],
      'js2dbus_return': [('call_id', -1), ('result', None)],
      'js2dbus_error': [('call_id', -1), ('error', '')],

      # objects on the dbus want to call objects in JS
      'dbus2js_call':  [('call_id', -1), ('interface', ''), ('method', ''),
                        ('args', [])],
      'dbus2js_return': [('call_id', -1), ('result', None)]
      }

   def __init__(self):
      self.command = ''
      self.data = None

   def parse(self, json_data):
      try:
         self.data = json.loads(json_data)
      except:
         print "Error parsing json data:"
         print json_data
         return ''

      self.command = self.data.get('cmd', '')
      return self.command

   def extract(self):
      description = self.__class__._msg_types.get(self.command, None)
      return [self.data.get(field, default) for field, default in description]

   @classmethod
   def emit(klass, command, **kwargs):
      description = klass._msg_types.get(command, None)
      if description is None:
         print "Can not create message of type %s as it's not defined." % command
         return None
      data = { 'cmd' : command }
      for field, default in description:
         data[field] = kwargs.get(field, default)
      return json.dumps(data)

class DBUSServerProtocol(protocol.Protocol):
   def __init__(self):
      self.client_id = None

   def dataReceived(self, msg):
      print msg

      packet = Message()
      command = packet.parse(msg)
      if len(command) == 0:
         print "No command in message, discarding."
         return

      if command == "register_on_bus":
         bus_name, object_name = packet.extract()
         print "** Registering on bus as %s %s" % (bus_name, object_name)
         if not whitelist.check_registrable(bus_name):
            print "Bus name %s is not authorized for registration." % bus_name
         else:
            tracker.register_on_bus(self.client_id,
                                    lambda: DBUSProxy(bus_name, object_name))

      elif command == "register_method":
         interface, method, in_sig, out_sig = packet.extract()
         if len(interface) == 0 or len(method) == 0:
            print "No interface and/or method name specified. Can't register."
            return

         print "** Registering method"
         tracker.register_method(self.client_id, interface, method, in_sig, out_sig)

      elif command == "dbus2js_return":
         call_id, result = packet.extract()
         callback, errback = tracker.unqueue_deferred_call(call_id)
         try:
            if callback is not None:
               callback(result) 
         except TypeError as error:
            if errback is not None:
               errback(Exception("Method return value doesn't match method signature: %s" % error))

      elif command == "js2dbus_call":
         call_id, bus_name, object_name, interface, method, args = packet.extract()
         if not whitelist.check_callable(bus_name, object_name, interface, method):
            print "Calling forbidden method: %s %s %s.%s" % (
                  bus_name, object_name, interface, method)
            self.send_message('js2dbus_error', call_id=call_id,
                              error='Method forbidden')
            return

         try:
            obj = dbus.SessionBus().get_object(bus_name, object_name)
            method = obj.get_dbus_method(method, dbus_interface=interface)

            def on_success(ret):
               self.send_message('js2dbus_return', call_id=call_id, result=ret)

            def on_failure(e):
               # FIXME: return an error to the caller
               print "Error in calling from JS to DBUS: %s" % e

            method(*args, reply_handler=on_success, error_handler=on_failure)

         except dbus.exceptions.DBusException as e:
            # FIXME: return an error to the caller
            print "Error in calling from JS to DBUS: %s" % e
            return

      else:
         print 'Unknown command "%s" in message, discarding.' % command

   def connectionMade(self):
      self.client_id = tracker.register_client(self)
      print "** Connection made. Id: %d" % self.client_id 

   def connectionLost(self, reason):
      print "** Connection lost. Id: %d" % self.client_id
      tracker.unregister_client(self.client_id)

   def forward_call(self, **kwargs):
      self.send_message('dbus2js_call', **kwargs)

   def send_message(self, command, **message_args):
      self.transport.write(Message.emit(command, **message_args))

class DBUSFactory(protocol.Factory):
   def buildProtocol(self, addr):
      return DBUSServerProtocol()