#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2011-2012 Canonical, Ltd.
#
# Authors:
#  Ugo Riboni <ugo.riboni@canonical.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dbus
import dbus.service
import types

from client_tracker import tracker

class DBUSProxy(dbus.service.Object):

   DBUS_DEFAULT_FLAGS = {
      '_dbus_is_method': True, '_dbus_sender_keyword': None,
      '_dbus_get_args_options': {'byte_arrays': False, 'utf8_strings': False},
      '_dbus_rel_path_keyword': None, '_dbus_connection_keyword': None,
      '_dbus_message_keyword': None, '_dbus_destination_keyword': None, '_dbus_path_keyword': None
      }

   # This method gets called by the proxy whenever we receive a call from DBUS that has to be
   # dispatched to one of the connected clients.
   @classmethod
   def dispatch_method(klass, interface, method, in_sig, out_sig, *args, **kwargs):
      # Remove the first element from the list of args, since it's always a reference to the proxy
      actual_args = list(args[1:])
      print "** Dispatching method call: %s(%s)" % (method, actual_args)
      client_id, client = tracker.client_with_method(interface, method)
      if not client is None:
         call_id = tracker.defer_call_return(client_id,
                                             kwargs.get('_dbus_callback'),
                                             kwargs.get('_dbus_errback')) if len(out_sig) > 0 else -1
         client.forward_call(interface=interface, method=method,
                             args=actual_args, call_id=call_id)

   def __init__(self, bus_name, object_name):
      self.bus_name = bus_name
      self.object_name = object_name

      # FIXME: use this code to check if the bus name is already taken
      #answer = dbus.SessionBus().request_name('org.documentroot.Calculator')
      #if answer == dbus.bus.REQUEST_NAME_REPLY_PRIMARY_OWNER: print 'got bus name'
      #if answer == dbus.bus.REQUEST_NAME_REPLY_ALREADY_OWNER: print 'already had bus name'
      #if answer == dbus.bus.REQUEST_NAME_REPLY_IN_QUEUE: print 'queued'
      #if answer == dbus.bus.REQUEST_NAME_REPLY_EXISTS: print 'could not get bus name'

      registeredBusName = dbus.service.BusName(bus_name, bus = dbus.SessionBus())
      dbus.service.Object.__init__(self, registeredBusName, object_name)

   # Register a new method at runtime on an interface on this instance of a DBUS object.
   # It will appear in DBUS introspection and when called the call will be proxied to the actual
   # connected client that registered it.
   def register_method(self, interface, method, in_sig, out_sig):
      def wrapper(klass, *args, **kwargs):
         return DBUSProxy.dispatch_method(interface, method, in_sig, out_sig, *args, **kwargs)

      # Retrieve the actual number of dbus args from the in_sig then create
      # fake argument names for each of them, so they can be sent out from DBUS introspection.
      num_args = len([arg for arg in dbus.Signature(in_sig)])
      arg_names = ['arg_%d' % n for n in xrange(0, num_args)]

      # Add a series of _dbus flags to the function. The dbus.service.Object system uses these
      # to decide what methods are exported over DBUS, their parameters and so on.
      # These are generally set by the dbus.service.method decorator, but since we need to add
      # them at runtime we have to set them manually.
      wrapper.__dict__.update(self.DBUS_DEFAULT_FLAGS)
      wrapper.__dict__.update({
         '_dbus_out_signature': out_sig,
         '_dbus_in_signature': in_sig,
         '_dbus_interface': interface,
         '_dbus_args': arg_names,
         })

      # Rename the wrapper to the wanted method name, so that reflection will pick it up correctly
      # and add it to the class, so that enumerating the class methods will return it (methods are
      # enumerated during DBUS introspection and when they are called as well)
      wrapper.__name__ = str(method)
      setattr(self.__class__, method, types.MethodType(wrapper, self.__class__))

      # If the DBUS signature expects a return value pass two "names". When the function is called
      # from the bus, the dispatcher will be passed two kwargs with these names. The first kwarg
      # will contain a function to be called in case of success, the other in case of error.
      wrapper.__dict__['_dbus_async_callbacks'] = ('_dbus_callback', '_dbus_errback') if len(out_sig) > 0 else None

      # Finally add this method to an internal table on the dbus.service.Object class. This is
      # normally done at __init__ time and picks up all methods decorated by dbus.service.method.
      # However since we're adding methods at runtime, we need to add them to the table as well.
      qualified_classname = self.__class__.__module__ + "." + self.__class__.__name__
      table = self._dbus_class_table[qualified_classname]
      if interface not in table: table[interface] = {}
      table[interface][method] = wrapper
