/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmaki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libdbusmenu-gtk/menu.h>
#include <libindicator/indicator.h>
#include <libindicator/indicator-object.h>
#include <libindicator/indicator-service-manager.h>
#include <libindicator/indicator-image-helper.h>

#include "dbus-shared-names.h"
#include "marshal.h"
#include "service-menuitem.h"
#include "tech-menuitem.h"
#include "indicator-network-service-xml.h"

#define NETWORK_DEFAULT_ICON		"nm-no-connection"
#define MOBILE_DEFAULT_ICON		"gsm-signal-none"

#define INDICATOR_NETWORK_TYPE            (indicator_network_get_type())
#define INDICATOR_NETWORK(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), INDICATOR_NETWORK_TYPE, IndicatorNetwork))
#define INDICATOR_NETWORK_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), INDICATOR_NETWORK_TYPE, IndicatorNetworkClass))
#define IS_INDICATOR_NETWORK(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), INDICATOR_NETWORK_TYPE))
#define IS_INDICATOR_NETWORK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), INDICATOR_NETWORK_TYPE))
#define INDICATOR_NETWORK_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), INDICATOR_NETWORK_TYPE, IndicatorNetworkClass))

typedef struct _IndicatorNetwork      IndicatorNetwork;
typedef struct _IndicatorNetworkClass IndicatorNetworkClass;

struct _IndicatorNetworkClass
{
  IndicatorObjectClass parent_class;
};

struct _IndicatorNetwork
{
  IndicatorObject parent;

  /* FIXME: move to a private struct */
  IndicatorServiceManager *service;
  GDBusProxy *backend_proxy;
  GtkImage *indicator_image;
  const gchar *accessible_desc;
  guint watch_id;
  
  GSList *custom_items;

  gboolean connected;
};

/* needed to to workaround a warning in INDICATOR_SET_TYPE() */
GType get_type(void);

GType indicator_network_get_type(void);
INDICATOR_SET_VERSION
INDICATOR_SET_TYPE(INDICATOR_NETWORK_TYPE)

static void indicator_network_dispose(GObject *object);
static void indicator_network_finalize(GObject *object);
G_DEFINE_TYPE(IndicatorNetwork, indicator_network, INDICATOR_OBJECT_TYPE);

static void
update_icons(IndicatorNetwork *self, const gchar *name)
{
  GtkImage *mobile_image  = NULL;
  GtkImage *network_image = NULL;

  GdkPixbuf *lbuf = NULL;
  GdkPixbuf *rbuf = NULL;
  GdkPixbuf *tbuf = NULL;

  gint lwidth  = 0;
  gint rwidth  = 0;
  gint twidth  = 0;

  gint lheight = 0;
  gint rheight = 0;
  gint theight = 0;

  gchar **names = NULL;
  
  if (name == NULL || g_strcmp0(name, "") == 0) {
    mobile_image  = indicator_image_helper(MOBILE_DEFAULT_ICON);
    network_image = indicator_image_helper(NETWORK_DEFAULT_ICON);
  } else {
    names = g_strsplit(name, " ", 0);

    if (g_strv_length(names) != 2) {
      g_warning("%s: invalid name: '%s'", __func__, name);
      g_strfreev(names);
      return;
    }
    
    mobile_image  = indicator_image_helper(names[0]);
    network_image = indicator_image_helper(names[1]);
        
    g_strfreev(names);
  }

  g_assert(network_image != NULL &&
	   mobile_image  != NULL);

  lbuf = gtk_image_get_pixbuf(mobile_image);
  rbuf = gtk_image_get_pixbuf(network_image);

  g_assert(lbuf != NULL &&
	   rbuf != NULL);

  lheight = gdk_pixbuf_get_height(lbuf);
  rheight = gdk_pixbuf_get_height(rbuf);

  lwidth = gdk_pixbuf_get_width(lbuf);
  rwidth = gdk_pixbuf_get_width(rbuf);

  if (lheight > rheight)
    theight = lheight;
  else
    theight = rheight;

  if (lwidth > rwidth)
    twidth = lwidth;
  else
    twidth = rwidth;

  tbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
			TRUE,
			8,
			twidth * 2 + 3, /* make room for two icons
					   and add some px between them */
			theight);
  gdk_pixbuf_fill(tbuf, 0x00000000); // make completely transparent

  gdk_pixbuf_copy_area(lbuf, 0, 0, lwidth, lheight,
		       tbuf, 0, 0);
  gdk_pixbuf_copy_area(rbuf, 0, 0, rwidth, rheight,
		       tbuf, twidth + 3, 0);

  gtk_image_set_from_pixbuf(self->indicator_image, tbuf);

  gtk_widget_destroy(GTK_WIDGET(network_image));
  gtk_widget_destroy(GTK_WIDGET(mobile_image));
  g_object_unref(lbuf);
  g_object_unref(rbuf);
  g_object_unref(tbuf);
}

static void set_icon(IndicatorNetwork *self, const gchar *name)
{
  if (name == NULL || strlen(name) == 0)
    return;

  g_debug("%s(%s)", __func__, name);

  update_icons(self, name);
}

static void set_accessible_desc(IndicatorNetwork *self, const gchar *desc)
{
  if (desc == NULL || strlen(desc) == 0)
    return;

  g_debug("%s(%s)", __func__, desc);

  self->accessible_desc = g_strdup(desc);

  g_signal_emit(G_OBJECT(self),
                INDICATOR_OBJECT_SIGNAL_ACCESSIBLE_DESC_UPDATE_ID,
                0,
                (IndicatorObjectEntry *)indicator_object_get_entries(INDICATOR_OBJECT(self))->data,
                TRUE);
}

static gboolean confirm_menu_open (IndicatorObject *io)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(io);

  g_return_val_if_fail(IS_INDICATOR_NETWORK (io), FALSE);
  g_return_val_if_fail(self->backend_proxy != NULL, FALSE);

  /// @todo decide what to do with scan requests..
  //g_dbus_proxy_call(self->backend_proxy, "RequestScan", NULL, G_DBUS_CALL_FLAGS_NONE,
  //		    -1, NULL, NULL, NULL);

  return FALSE;
}

static void menu_visibility_changed (GtkWidget *widget, gpointer *user_data)
{
	static guint id = 0;

	g_return_if_fail (GTK_IS_WIDGET (widget));

	/* confirm the menu is open after 2s, or cancel the timer
	   if its closed in the meantime */
	if (gtk_widget_get_visible (widget))
		id = g_timeout_add_seconds (2, (GSourceFunc) confirm_menu_open,
									user_data);
	else if (id > 0)
		g_source_remove (id);
}

static gboolean new_service_menuitem(DbusmenuMenuitem *newitem,
				     DbusmenuMenuitem *parent,
				     DbusmenuClient   *client,
				     gpointer user_data)
{
  GtkWidget *smi;

  IndicatorNetwork *self = user_data;

  g_debug("%s()", __func__);

  g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
  g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
  /* Note: not checking parent, it's reasonable for it to be NULL */

  smi = service_menuitem_new();

  dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem,
				  GTK_MENU_ITEM(smi), parent);

  service_menuitem_set_dbusmenu(SERVICE_MENUITEM(smi), newitem);

  self->custom_items = g_slist_append(self->custom_items, smi);

  return TRUE;
}


static gboolean new_tech_menuitem(DbusmenuMenuitem *newitem,
				  DbusmenuMenuitem *parent,
				  DbusmenuClient   *client,
				  gpointer user_data)
{
  GtkWidget *smi;
  IndicatorNetwork *self = user_data;

  g_debug("%s()", __func__);

  g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
  g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
  /* Note: not checking parent, it's reasonable for it to be NULL */

  smi = tech_menuitem_new();

  dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem,
				  GTK_MENU_ITEM(smi), parent);

  tech_menuitem_set_dbusmenu(TECH_MENUITEM(smi), newitem);

  self->custom_items = g_slist_append(self->custom_items, smi);

  return TRUE;
}

static GtkLabel *get_label(IndicatorObject * io)
{
  return NULL;
}

static GtkImage *get_icon(IndicatorObject * io)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(io);

  self->indicator_image = GTK_IMAGE(gtk_image_new());

  update_icons(self, NULL);

  gtk_widget_show(GTK_WIDGET(self->indicator_image));

  return self->indicator_image;
}

static const gchar *get_accessible_desc(IndicatorObject * io)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(io);

  return self->accessible_desc;
}

void remove_cb(GtkContainer *container,
	       GtkWidget    *widget,
	       gpointer      user_data)
{
  IndicatorNetwork *self = user_data;

  // if disconnected, connection_changed() handles the remove
  if (self->connected) {

    if (IS_SERVICE_MENUITEM(widget) ||
	IS_TECH_MENUITEM(widget)) {
      g_debug("REMOVED %p", widget);
      self->custom_items = g_slist_remove(self->custom_items, widget);
      gtk_widget_destroy(widget);
    }
  }
}


/* Indicator based function to get the menu for the whole
   applet.  This starts up asking for the parts of the menu
   from the various services. */
static GtkMenu *get_menu(IndicatorObject *io)
{
  IndicatorNetwork *self;

  self = INDICATOR_NETWORK(io);
  g_assert(self != NULL);

  DbusmenuGtkMenu *menu = dbusmenu_gtkmenu_new(INDICATOR_NETWORK_DBUS_NAME,
					       INDICATOR_NETWORK_DBUS_OBJECT);
  DbusmenuGtkClient * client = dbusmenu_gtkmenu_get_client(menu);

  g_signal_connect (GTK_MENU (menu),
		    "map", G_CALLBACK (menu_visibility_changed), io);
  g_signal_connect (GTK_MENU (menu),
		    "hide", G_CALLBACK (menu_visibility_changed), io);

  g_signal_connect(menu,
		   "remove",
		   G_CALLBACK(remove_cb),
		   self);

  dbusmenu_client_add_type_handler_full(DBUSMENU_CLIENT(client),
					SERVICE_MENUITEM_NAME,
					new_service_menuitem,
					self,
					NULL);

  dbusmenu_client_add_type_handler_full(DBUSMENU_CLIENT(client),
					TECH_MENUITEM_NAME,
					new_tech_menuitem,
					self,
					NULL);

  g_debug("%s()", __func__);

  return GTK_MENU(menu);
}

static void icon_changed(IndicatorNetwork *self, GVariant *parameters)
{
  const gchar *name;
  GVariant *value;

  value = g_variant_get_child_value(parameters, 0);
  name = g_variant_get_string(value, NULL);

  set_icon(self, name);

  g_variant_unref(value);
}

static void accessible_desc_changed(IndicatorNetwork *self, GVariant *parameters)
{
  const gchar *desc;
  GVariant *value;

  value = g_variant_get_child_value(parameters, 0);
  desc = g_variant_get_string(value, NULL);

  set_accessible_desc(self, desc);

  g_variant_unref(value);
}

static void backend_signal_cb(GDBusProxy *proxy,
			      const gchar *sender_name,
			      const gchar *signal_name,
			      GVariant *parameters,
			      gpointer user_data)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(user_data);

  /*
   * gdbus documentation is not clear who owns the variant so take it, just
   * in case
   */
  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, INDICATOR_NETWORK_SIGNAL_ICON_CHANGED) == 0) {
    icon_changed(self, parameters);
  } else if (g_strcmp0(signal_name, INDICATOR_NETWORK_SIGNAL_ACCESSIBLE_DESC_CHANGED) == 0) {
    accessible_desc_changed(self, parameters);
  }

  g_variant_unref(parameters);
}

static void get_icon_cb(GObject *object, GAsyncResult *res,
		       gpointer user_data)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(user_data);
  GVariant *result, *value;
  GError *error = NULL;
  const gchar *name;

  g_return_if_fail(self != NULL);

  result = g_dbus_proxy_call_finish(self->backend_proxy, res, &error);

  if (error != NULL) {
    g_debug("GetIcon call failed: %s", error->message);
    g_error_free(error);
    return;
  }

  if (result == NULL)
    return;

  value = g_variant_get_child_value(result, 0);
  name = g_variant_get_string(value, NULL);

  set_icon(self, name);

  g_variant_unref(value);
  g_variant_unref(result);
}

static void backend_appeared(GDBusConnection *connection, const gchar *name,
			     const gchar *name_owner, gpointer user_data)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(user_data);

  g_dbus_proxy_call(self->backend_proxy, "GetIcon", NULL,
		    G_DBUS_CALL_FLAGS_NONE, -1, NULL, get_icon_cb, self);
}

static void backend_vanished(GDBusConnection *connection, const gchar *name,
			     gpointer user_data)
{
  /* FIXME: change icon to show that we are not connected to backend */
}

static void create_backend_proxy_cb(GObject *source_object, GAsyncResult *res,
				    gpointer user_data)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(user_data);
  GDBusInterfaceInfo *info;
  GError *error = NULL;

  g_return_if_fail(self != NULL);

  self->backend_proxy = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get backend proxy: %s", error->message);
    g_error_free(error);
    return;
  }

  if (self->backend_proxy == NULL) {
    g_warning("Failed to get backend proxy, but no errors");
    return;
  }

  g_signal_connect(self->backend_proxy, "g-signal",
		   G_CALLBACK(backend_signal_cb), self);

  self->watch_id = g_bus_watch_name(G_BUS_TYPE_SESSION, 
				    INDICATOR_NETWORK_DBUS_NAME,
				    G_BUS_NAME_WATCHER_FLAGS_NONE,
				    backend_appeared,
				    backend_vanished,
				    self,
				    NULL);
}

static void create_backend_proxy (IndicatorNetwork *self)
{
  g_return_if_fail (IS_INDICATOR_NETWORK (self));

  if (self->backend_proxy != NULL)
    return;

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SESSION,
			   G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   INDICATOR_NETWORK_BACKEND_NAME,
			   INDICATOR_NETWORK_BACKEND_MANAGER_PATH,
			   INDICATOR_NETWORK_BACKEND_MANAGER_INTERFACE,
			   NULL,
			   create_backend_proxy_cb,
			   self);
}

static void indicator_connected(IndicatorNetwork *self)
{
  g_return_if_fail (IS_INDICATOR_NETWORK (self));

  create_backend_proxy(self);
}

static void connection_changed(IndicatorServiceManager *sm,
			       gboolean connected,
			       gpointer user_data)
{
  IndicatorNetwork *self = user_data;
  GSList *iter;

  self->connected = connected;

  if (self->connected) {
    indicator_connected(self);
  } else {
    g_debug("DISCONNECTED: %d", g_slist_length(self->custom_items));

    for (iter = self->custom_items; iter != NULL; iter = iter->next)
      g_debug("freeing: %p", iter->data);

    g_slist_free_full(self->custom_items, (GDestroyNotify)gtk_widget_destroy);
    self->custom_items = NULL;

    if (self->watch_id != 0) {
      g_bus_unwatch_name(self->watch_id);
      self->watch_id = 0;
    }

    if (self->backend_proxy != NULL) {
      g_object_unref(self->backend_proxy);
      self->backend_proxy = NULL;
    }
  }


  return;
}

static void indicator_network_class_init(IndicatorNetworkClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);
  IndicatorObjectClass *io_class;

  object_class->dispose = indicator_network_dispose;
  object_class->finalize = indicator_network_finalize;

  io_class = INDICATOR_OBJECT_CLASS(klass);
  io_class->get_label = get_label;
  io_class->get_image = get_icon;
  io_class->get_menu = get_menu;
  io_class->get_accessible_desc = get_accessible_desc;
}

static void indicator_network_init(IndicatorNetwork *self)
{
  self->service = NULL;

  self->service = indicator_service_manager_new_version(INDICATOR_NETWORK_DBUS_NAME,
							INDICATOR_NETWORK_DBUS_VERSION);
  g_signal_connect(G_OBJECT(self->service),
		   INDICATOR_SERVICE_MANAGER_SIGNAL_CONNECTION_CHANGE,
		   G_CALLBACK(connection_changed),
		   self);

  self->custom_items = NULL;
  self->connected = FALSE;

  return;
}

static void indicator_network_dispose(GObject *object)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(object);

  if (self->service != NULL) {
    g_object_unref(G_OBJECT(self->service));
    self->service = NULL;
  }

  if (self->watch_id != 0) {
    g_bus_unwatch_name(self->watch_id);
    self->watch_id = 0;
  }

  if (self->backend_proxy != NULL) {
    g_object_unref(self->backend_proxy);
    self->backend_proxy = NULL;
  }

  G_OBJECT_CLASS(indicator_network_parent_class)->dispose(object);

  return;
}

static void indicator_network_finalize(GObject *object)
{
  G_OBJECT_CLASS(indicator_network_parent_class)->finalize(object);
  return;
}
