#include "android-account-sync.h"

#include <glib.h>
#include <string.h>
#include <libedataserver/e-source.h>
#include <libedataserver/e-source-list.h>
#include <libedataserver/e-source-group.h>
#include <libedataserver/e-client.h>
#include <libedataserverui/e-client-utils.h>

#define SOURCE_LIST_PATH                        "/apps/evolution/calendar/sources"
#define CALENDAR_COLOR                          "#E2D4B7"

#define CALENDAR_CONFIG_PREFIX                  "/apps/evolution/calendar"
#define SELECTED_CALENDARS_KEY                  CALENDAR_CONFIG_PREFIX"/display/selected_calendars"
#define PRIMARY_CALENDAR_KEY                    CALENDAR_CONFIG_PREFIX"/display/primary_calendar"

#define UBUNTU_ANDROID_ACCOUNT_SERVICE_NAME     "com.canonical.Android"
#define UBUNTU_ANDROID_ACCOUNT_OBJECT_PATH      "/com/canonical/android/accounts/Accounts"
#define UBUNTU_ANDROID_ACCOUNT_INTERFACE_NAME   "com.canonical.android.accounts.Accounts"

#define GOOGLE_CALENDAR_ADAPTER                 "com.android.calendar"
#define GOOGLE_ACCOUNT_TYPE                     "com.google"

#define ANDROID_ACCOUNT_SYNC_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), ANDROID_ACCOUNT_SYNC_TYPE, AndroidAccountSyncPrivate))

G_DEFINE_TYPE (AndroidAccountSync, android_account_sync, G_TYPE_OBJECT)

typedef struct _AndroidAccountSyncPrivate        AndroidAccountSyncPrivate;
struct _AndroidAccountSyncPrivate
{
    guint dbus_watch_id;
    GDBusProxy *accounts_proxy;
    GList *pending_accounts;
    GList *accounts;
};

typedef struct _AndroidAccountSyncQueryAccount        AndroidAccountSyncQueryAccount;
struct _AndroidAccountSyncQueryAccount
{
    AndroidAccountSync *self;
    gchar* account;
};


static void
android_account_sync_dispose (GObject *gobject)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (gobject);

    if (priv->dbus_watch_id > 0) {
        g_bus_unwatch_name (priv->dbus_watch_id);
        priv->dbus_watch_id = 0;
    }
    g_clear_object (&priv->accounts_proxy);

    G_OBJECT_CLASS (android_account_sync_parent_class)->dispose (gobject);
}

static void
android_account_sync_class_init (AndroidAccountSyncClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = android_account_sync_dispose;

    g_type_class_add_private (klass, sizeof (AndroidAccountSyncPrivate));
}

static void
android_account_sync_init (AndroidAccountSync *self)
{
}

AndroidAccountSync*
android_account_sync_new (void)
{
    return (AndroidAccountSync*) g_object_new (ANDROID_ACCOUNT_SYNC_TYPE, NULL);
}

static ESource*
_android_account_sync_eds_register_source(const gchar* username)
{
    gchar *email = g_uri_escape_string (username, NULL, FALSE);

    // google calendar url (https://www.google.com/calendar/feeds/<username>%40gmail.com/private/full)
    gchar *aux = g_strdup_printf ("https://www.google.com/calendar/feeds/%s/private/full", email);
    ESource *cSource = e_source_new ("Android", aux);
    g_free (aux);

    // Caldav url (caldav://<username>%40gmail.com@www.google.com/calendar/dav/<username>@gmail.com/events)
    aux = g_strdup_printf ("caldav://%s@www.google.com/calendar/dav/%s/events", email, username);
    e_source_set_absolute_uri (cSource, aux);
    g_free (aux);

    // Calendar color on evolution application
    e_source_set_color_spec (cSource, CALENDAR_COLOR);

    // refresh timeout
    e_source_set_property (cSource, "refresh-type", "0");
    e_source_set_property (cSource, "refresh", "30");

    // Google username
    e_source_set_property (cSource, "setup-username", username);
    e_source_set_property (cSource, "username", username);
    e_source_set_property (cSource, "remember_password", "true");

    // Google calendar uses ssl authentication
    e_source_set_property (cSource, "ssl", "1");

    // Google calendar requiere authentication
    e_source_set_property (cSource, "auth", "1");

    // Others
    e_source_set_property (cSource, "alarm", "true");
    e_source_set_property (cSource, "default", "true");
    e_source_set_property (cSource, "offline_sync", "true");

    // Google calendar name, keep default value for now
    // EDataServer.ESource.set_property(cSource, "googlename", "Birthdays");

    return cSource;
}

static void
_android_account_sync_eds_append_source (const gchar *key,
                                         const gchar *defaultCalendarKey,
                                         const gchar* value)
{
    GConfClient *config = gconf_client_get_default ();
    g_return_if_fail (config != NULL);
    GSList *valueLst;
    GError *error = NULL;

    valueLst = gconf_client_get_list (config, key, GCONF_VALUE_STRING, &error);
    if (error != NULL) {
        g_warning ("Fail to get active calendar list: %s", error->message);
        g_object_unref (config);
        g_error_free (error);
        return;
    }

    GSList *found = g_slist_find_custom (valueLst, value, (GCompareFunc) g_strcmp0);
    if (!found) {
        valueLst = g_slist_append (valueLst, (gpointer) g_strdup ((gchar*) value));
        gconf_client_set_list (config, key, GCONF_VALUE_STRING, valueLst, NULL);
    }

    gconf_client_set_string (config, defaultCalendarKey, value, NULL);

    if (valueLst) {
        g_slist_free_full (valueLst, g_free);
    }
    g_object_unref (config);
}

static void
_android_account_sync_eds_remove_source (const gchar *key,
                                         const gchar *defaultCalendarKey,
                                         const gchar* value)
{
    GConfClient *config = gconf_client_get_default ();
    g_return_if_fail (config != NULL);
    GSList *valueLst;
    GError *error = NULL;

    valueLst = gconf_client_get_list (config, key, GCONF_VALUE_STRING, &error);
    if (error != NULL) {
        g_warning ("Fail to get active calendar list: %s", error->message);
        g_object_unref (config);
        g_error_free (error);
        return;
    }

    GSList *found = g_slist_find_custom (valueLst, value, (GCompareFunc) g_strcmp0);
    if (found) {
        g_free (found->data);
        valueLst = g_slist_delete_link (valueLst, found);
        gconf_client_set_list (config, key, GCONF_VALUE_STRING, valueLst, NULL);
    }

    gchar *defaultCalendar = gconf_client_get_string (config, defaultCalendarKey, NULL);
    if (g_strcmp0 (defaultCalendar, value) == 0) {
        gconf_client_set_string (config, defaultCalendarKey, "", NULL);
    }
    g_free (defaultCalendar);

    if (valueLst) {
        g_slist_free_full (valueLst, g_free);
    }

    g_object_unref (config);
}

static void
_android_account_sync_eds_active_source (ESource *source)
{
    const gchar *sUID = e_source_peek_uid (source);
    g_assert (sUID);
    _android_account_sync_eds_append_source (SELECTED_CALENDARS_KEY, PRIMARY_CALENDAR_KEY, sUID);
}

static void
_android_account_sync_eds_deactive_source (ESource *source)
{
    const gchar *sUID = e_source_peek_uid (source);
    g_assert (sUID);
    _android_account_sync_eds_remove_source (SELECTED_CALENDARS_KEY, PRIMARY_CALENDAR_KEY, sUID);
}

static void
_android_account_sync_eds_register_users (GList *users)
{
    ESourceList *calendarList;
    GSList *calendarSources;
    GList *newUsers = users;

    calendarList = e_source_list_new_for_gconf_default (SOURCE_LIST_PATH);
    if (calendarList) {
        e_source_list_ensure_group (calendarList, "Google", "google://", FALSE);
        ESourceGroup  *calendarGroup;

        calendarGroup = e_source_list_peek_group_by_base_uri (calendarList,
                                                              "google://");

        calendarSources = e_source_group_peek_sources (calendarGroup);

        // Check for resgistered accounts
        GSList *sources = calendarSources;
        for(; sources; sources = sources->next) {
            const gchar* username = e_source_get_property ((ESource*) sources->data, "username");
            GList *found = g_list_find_custom (newUsers, username, (GCompareFunc ) g_strcmp0);
            if (found) {
                newUsers = g_list_remove_link (newUsers, found);
            } else {
                _android_account_sync_eds_deactive_source ((ESource*) sources->data);
                e_source_group_remove_source (calendarGroup, (ESource*) sources->data);
            }
        }

        // Register new accounts
        GList *walk;
        for (walk = newUsers; walk; walk = walk->next) {
            ESource *cSource;
            cSource = _android_account_sync_eds_register_source ((const gchar*) walk->data);
            if (cSource) {
                if (e_source_group_add_source (calendarGroup, cSource, -1)) {
                    e_source_list_sync (calendarList, NULL);

                    _android_account_sync_eds_active_source (cSource);
                    g_object_unref (cSource);
                }
            }
        }
    }
}


static void
_android_account_sync_get_auto_enabled_sync_adapters_finish (GObject *source_object,
                                                              GAsyncResult *res,
                                                              AndroidAccountSyncQueryAccount *q)
{
    AndroidAccountSync *self = q->self;
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);
    gboolean valid_account = FALSE;
    GError *error = NULL;
    GVariant *result = g_dbus_proxy_call_finish (priv->accounts_proxy, res, &error);
    if (error != NULL) {
        g_warning ("Fail to call dbus method: %s", error->message);
        g_error_free (error);
        goto finish;
    }

    GVariantIter *iter;
    g_variant_get (result, "(as)", &iter);

    gchar *adapter;
    while (g_variant_iter_loop (iter, "s", &adapter)) {
        if (g_str_equal (adapter, GOOGLE_CALENDAR_ADAPTER)) {
            valid_account = TRUE;
            priv->accounts = g_list_append (priv->accounts, q->account);
        }
    }

    g_variant_unref (result);

finish:
    priv->pending_accounts = g_list_remove (priv->pending_accounts, q->account);
    if (!valid_account) {
        g_free (q->account);
    }
    g_free (q);

    // Wait for all pending accounts be verified
    if (g_list_length (priv->pending_accounts) == 0) {
        _android_account_sync_eds_register_users (priv->accounts);
        g_list_free_full (priv->accounts, (GDestroyNotify) g_free);
        priv->accounts = NULL;
    }
}

static void
_android_account_sync_get_accounts_adapters (AndroidAccountSync *self,
                                              GList *accounts)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);
    priv->pending_accounts = accounts;

    for(; accounts; accounts = accounts->next) {
        AndroidAccountSyncQueryAccount *q = g_new0(AndroidAccountSyncQueryAccount, 1);
        q->self = self;
        q->account = (gchar*) accounts->data;
        // Request account adapters
        g_dbus_proxy_call (priv->accounts_proxy,
                           "getAutoEnabledSyncAdapters",
                           g_variant_new ("((ss))",
                                          accounts->data,
                                          GOOGLE_ACCOUNT_TYPE),
                           G_DBUS_CALL_FLAGS_NONE,
                           -1,
                           NULL,
                           (GAsyncReadyCallback) _android_account_sync_get_auto_enabled_sync_adapters_finish,
                           q);
    }
}

static void
_android_account_sync_get_all_accounts_finish  (GObject *source_object,
                                                GAsyncResult *res,
                                                AndroidAccountSync *self)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);

    GError *error = NULL;
    GVariant *result = g_dbus_proxy_call_finish (priv->accounts_proxy, res, &error);
    if (error != NULL) {
        g_warning ("Fail to call dbus method: %s", error->message);
        g_error_free (error);
        return;
    }

    GVariantIter *iter;
    g_variant_get (result, "(a(ss))", &iter);

    GList *users = NULL;
    gchar *account;
    gchar *type;
    while (g_variant_iter_loop (iter, "(ss)", &account, &type)) {
        if (g_str_equal (type, "com.google")) {
            users = g_list_append (users, g_strdup (account));
        }
    }

    g_variant_iter_free (iter);
    g_variant_unref (result);

    if (users != NULL) {
        _android_account_sync_get_accounts_adapters (self, users);
    } else {
        // Remove all accounts
        _android_account_sync_eds_register_users (NULL);
    }
}

static void
_android_account_sync_update_accounts (AndroidAccountSync *self)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);
    // Request accounts
    g_dbus_proxy_call (priv->accounts_proxy,
                       "getAllAccounts",
                       NULL,
                       G_DBUS_CALL_FLAGS_NONE,
                       -1,
                       NULL,
                       (GAsyncReadyCallback) _android_account_sync_get_all_accounts_finish,
                       self);
}


static void
_android_account_sync_dbus_signal_cb (GDBusProxy * proxy,
                                      gchar * sender_name,
                                      gchar * signal_name,
                                      GVariant * parameters,
                                      AndroidAccountSync *self)
{
    if (g_str_equal (signal_name, "SyncSettingsUpdated") ||
        g_str_equal (signal_name, "AccountsUpdated")) {
        _android_account_sync_update_accounts (self);
    }
}

static void
_android_account_sync_start_finished (GObject *source_object,
                                      GAsyncResult *res,
                                      AndroidAccountSync *self)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);
    GError *error = NULL;
    priv->accounts_proxy = g_dbus_proxy_new_for_bus_finish (res, &error);
    if (error != NULL) {
        g_warning ("Fail to create dbus proxy object: %s", error->message);
        g_error_free (error);
        priv->accounts_proxy = NULL;
        return;
    }

    g_signal_connect(priv->accounts_proxy,
                     "g-signal",
                     G_CALLBACK(_android_account_sync_dbus_signal_cb),
                     self);

    _android_account_sync_update_accounts (self);
}

static void
_android_account_sync_service_appear (GDBusConnection *connection,
                                      const gchar *name,
                                      const gchar *new_owner,
                                      AndroidAccountSync *self)
{
    g_dbus_proxy_new_for_bus (G_BUS_TYPE_SESSION,
                              G_DBUS_PROXY_FLAGS_NONE,
                              NULL,
                              UBUNTU_ANDROID_ACCOUNT_SERVICE_NAME,
                              UBUNTU_ANDROID_ACCOUNT_OBJECT_PATH,
                              UBUNTU_ANDROID_ACCOUNT_INTERFACE_NAME,
                              NULL,
                              (GAsyncReadyCallback) _android_account_sync_start_finished,
                              self);
}

static void
_android_account_sync_service_vanished (GDBusConnection *connection,
                                        const gchar *name,
                                        AndroidAccountSync *self)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);

    g_clear_object (&priv->accounts_proxy);
}


void
android_account_sync_start (AndroidAccountSync *self)
{
    AndroidAccountSyncPrivate *priv = ANDROID_ACCOUNT_SYNC_GET_PRIVATE (self);
    if (priv->dbus_watch_id > 0) {
        return;
    }

    priv->dbus_watch_id = g_bus_watch_name (G_BUS_TYPE_SESSION,
                                            UBUNTU_ANDROID_ACCOUNT_SERVICE_NAME,
                                            G_BUS_NAME_WATCHER_FLAGS_NONE,
                                            (GBusNameAppearedCallback) _android_account_sync_service_appear,
                                            (GBusNameVanishedCallback) _android_account_sync_service_vanished,
                                            self,
                                            NULL);

}
