#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.tests import SQLiteTestBase

from imagestore.sqlitestorage import SQLiteStorage
from imagestore.model import Image, ImageRegistration, ImageState
from imagestore.tests.helpers import ImageTestCase


class SQLiteStorageTest(ImageTestCase, SQLiteTestBase):

    def setUp(self):
        ImageTestCase.setUp(self)
        SQLiteTestBase.setUp(self)
        self.storage = SQLiteStorage(self.filename)
        self.image1 = self.createImage(1)
        self.image2 = self.createImage(2)

    def tearDown(self):
        ImageTestCase.tearDown(self)
        SQLiteTestBase.tearDown(self)
        self.storage.close()

    def testPatchSchemaOnCreation(self):
        cursor = self.execute("SELECT version FROM patch")
        self.assertEquals(type(cursor.fetchall()), list)

    def testCommitAndClose(self):
        storage_mock = self.mocker.patch(self.storage)
        storage_mock.commit()
        storage_mock.close()
        self.mocker.replay()
        self.storage.commitAndClose()

    def testAddImage(self):
        self.storage.addImage(self.image1)
        self.storage.commitAndClose()
        cursor = self.execute("SELECT uri, json FROM image")
        self.assertEquals(list(cursor.fetchall()),
                          [(self.image1["uri"], str(self.image1))])

    def testReplaceImage(self):
        self.storage.addImage(self.image1)
        self.image1["title"] = "New title"
        self.storage.addImage(self.image1)
        image = self.storage.getImage(self.image1["uri"])
        self.assertEquals(image["title"], "New title")

    def testRollback(self):
        self.storage.addImage(self.image1)
        self.storage.rollback()
        self.storage.commitAndClose()
        cursor = self.execute("SELECT uri, json FROM image")
        self.assertEquals(list(cursor.fetchall()), [])

    def testGetImage(self):
        self.execute("INSERT INTO image (uri, json) VALUES (?, ?)",
                     ((self.image1["uri"], str(self.image1))))
        self.execute("INSERT INTO image (uri, json) VALUES (?, ?)",
                     ((self.image2["uri"], str(self.image2))))

        self.close()

        image1 = self.storage.getImage(self.image1["uri"])

        self.storage.commitAndClose()

        self.assertEquals(image1, self.image1)

    def testGetNonExistentImage(self):
        image = self.storage.getImage("http://example.com")
        self.assertEquals(image, None)

    def testSetAndGetUpgrades(self):
        uri1 = "http://uri/1"
        uri2 = "http://uri/2"
        self.storage.setUpgrades([uri1])
        self.assertEquals(self.storage.getUpgrades(), [uri1])
        self.storage.setUpgrades([uri2])
        self.assertEquals(self.storage.getUpgrades(), [uri2])
        self.storage.setUpgrades([uri1])
        self.assertEquals(self.storage.getUpgrades(), [uri1])
        self.storage.setUpgrades([uri1, uri2])
        self.assertEquals(self.storage.getUpgrades(), [uri1, uri2])

    def testAddImageDoesntResetUpgrade(self):
        self.storage.setUpgrades([self.image1["uri"]])
        self.storage.addImage(self.image1)
        self.assertEquals(self.storage.getUpgrades(), [self.image1["uri"]])

    def testGetImageStateWithoutImage(self):
        state = self.storage.getImageState("http://non/existent")
        self.assertEquals(state, ImageState({
                            "image-uri": "http://non/existent",
                            "status": "uninstalled",
                          }))
        self.assertEquals(type(state), ImageState)

    def testGetImageStateUninstalled(self):
        image = self.createImage(1)
        self.storage.addImage(image)
        state = self.storage.getImageState(image["uri"])
        self.assertEquals(state, ImageState({
                            "image-uri": image["uri"],
                            "status": "uninstalled",
                          }))

    def testSetRegistrationAndGetImageStateInstalled(self):
        imageRegistration1 = self.createImageRegistration(1)
        image = imageRegistration1.image
        self.storage.setImageRegistration(imageRegistration1)
        state2 = self.storage.getImageState(image["uri"])

        self.assertEquals(state2, ImageState({
                              "image-uri": image["uri"],
                              "status": "installed",
                              "emi": imageRegistration1.emi.eid,
                              "eki": imageRegistration1.eki.eid,
                              "eri": imageRegistration1.eri.eid,
                          }))

    def testSetRegistrationAndGetUnexistingImageState(self):
        imageRegistration1 = self.createImageRegistration(1)
        image = imageRegistration1.image
        self.storage.setImageRegistration(imageRegistration1)
        state = self.storage.getImageState("http://another/url")
        self.assertEquals(state, ImageState({
                              "image-uri": "http://another/url",
                              "status": "uninstalled",
                          }))

    def testSetRegistrationTwice(self):
        imageRegistration1 = self.createImageRegistration(1)
        image = imageRegistration1.image
        self.storage.setImageRegistration(imageRegistration1)
        self.storage.setImageRegistration(imageRegistration1)
        state2 = self.storage.getImageState(image["uri"])

        self.assertEquals(state2, ImageState({
                              "image-uri": image["uri"],
                              "status": "installed",
                              "emi": imageRegistration1.emi.eid,
                              "eki": imageRegistration1.eki.eid,
                              "eri": imageRegistration1.eri.eid,
                          }))

    def testSetRegistrationAddsImage(self):
        imageRegistration = self.createImageRegistration(1)
        self.storage.setImageRegistration(imageRegistration)
        image = self.storage.getImage(imageRegistration.image["uri"])
        self.assertEquals(image, imageRegistration.image)

    def testSetErrorMessage(self):
        self.storage.setErrorMessage("http://uri", "Some error!")
        state = self.storage.getImageState("http://uri")
        self.assertEquals(state, ImageState({
                              "image-uri": "http://uri",
                              "status": "uninstalled",
                              "error-message": "Some error!",
                          }))

    def testSetErrorMessageTwice(self):
        self.storage.setErrorMessage("http://uri", "Some error!")
        self.storage.setErrorMessage("http://uri", "Another error!")
        state = self.storage.getImageState("http://uri")
        self.assertEquals(state, ImageState({
                              "image-uri": "http://uri",
                              "status": "uninstalled",
                              "error-message": "Another error!",
                          }))

    def testClearErrorMessage(self):
        self.storage.setErrorMessage("http://uri", "Some error!")
        self.storage.clearErrorMessage("http://uri")
        state = self.storage.getImageState("http://uri")
        self.assertEquals(state, ImageState({
                              "image-uri": "http://uri",
                              "status": "uninstalled",
                          }))


    def testGetStateWithUpgrade(self):
        self.storage.setUpgrades(["http://uri"])
        state = self.storage.getImageState("http://uri")
        self.assertEquals(state, ImageState({
                              "image-uri": "http://uri",
                              "status": "uninstalled",
                              "is-upgrade": True,
                          }))

    def testSetUpgradesWithRegisteredImage(self):
        """If an image is installed, it can't be an upgrade."""
        imageRegistration = self.createImageRegistration(1)
        image = imageRegistration.image
        self.storage.addImage(image)
        self.storage.setImageRegistration(imageRegistration)
        self.storage.setUpgrades([image["uri"]])
        state = self.storage.getImageState(image["uri"])
        self.assertEquals(state, ImageState({
                              "image-uri": image["uri"],
                              "status": "installed",
                              "emi": imageRegistration.emi.eid,
                              "eki": imageRegistration.eki.eid,
                              "eri": imageRegistration.eri.eid,
                              #"is-upgrade": True,  <== That's the test.
                          }))

    def testGetErrorMessage(self):
        errorMessage = self.storage.getErrorMessage("http://uri")
        self.assertEquals(errorMessage, None)
        self.storage.setErrorMessage("http://uri", "Some error!")
        errorMessage = self.storage.getErrorMessage("http://uri")
        self.assertEquals(errorMessage, "Some error!")

    def testGetInstalledImageUris(self):
        imageRegistration1 = self.createImageRegistration(1)
        imageRegistration2 = self.createImageRegistration(2)
        image3 = self.createImage(3)
        image4 = self.createImage(4)

        self.storage.setImageRegistration(imageRegistration1)
        self.storage.setImageRegistration(imageRegistration2)
        self.storage.addImage(image3)
        self.storage.addImage(image4)
        self.assertEquals(set(self.storage.getInstalledImageUris()),
                          set([imageRegistration1.image["uri"],
                               imageRegistration2.image["uri"]]))
