#include "hrn-spinner.h"

enum {
    PROP_0,
};

struct _HrnSpinnerPrivate {
    ClutterTimeline *timeline;
    int frame;

    CoglHandle material;
    CoglHandle texture;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_SPINNER, HrnSpinnerPrivate))
G_DEFINE_TYPE (HrnSpinner, hrn_spinner, NBTK_TYPE_WIDGET);

static void
hrn_spinner_finalize (GObject *object)
{
    HrnSpinner *self = (HrnSpinner *) object;
    HrnSpinnerPrivate *priv = self->priv;

    cogl_material_unref (priv->material);
    cogl_texture_unref (priv->texture);

    G_OBJECT_CLASS (hrn_spinner_parent_class)->finalize (object);
}

static void
hrn_spinner_dispose (GObject *object)
{
    HrnSpinner *self = (HrnSpinner *) object;
    HrnSpinnerPrivate *priv = self->priv;

    if (priv->timeline) {
        clutter_timeline_stop (priv->timeline);
        g_object_unref (priv->timeline);
        priv->timeline = NULL;
    }

    G_OBJECT_CLASS (hrn_spinner_parent_class)->dispose (object);
}

static void
hrn_spinner_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_spinner_get_property (GObject    *object,
                          guint       prop_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

#define TILE_WIDTH (float) 0.125
#define TILE_HEIGHT (float) 0.3333333 /* *Sigh* */

static void
hrn_spinner_paint (ClutterActor *actor)
{
    HrnSpinner *spinner = (HrnSpinner *) actor;
    HrnSpinnerPrivate *priv = spinner->priv;
    NbtkPadding padding;
    int ox, oy;
    float tx, ty;

    CLUTTER_ACTOR_CLASS (hrn_spinner_parent_class)->paint (actor);

    nbtk_widget_get_padding ((NbtkWidget *) actor, &padding);

    cogl_push_matrix ();
    cogl_translate (padding.left, padding.top, 0);

    ox = priv->frame % 8;
    oy = priv->frame / 8;

    tx = (float) ox * TILE_WIDTH;
    ty = (float) oy * TILE_HEIGHT;

    cogl_set_source (priv->material);
    cogl_rectangle_with_texture_coords (0.0, 0.0, 16.0, 16.0,
                                        tx, ty,
                                        tx + TILE_WIDTH, ty + TILE_HEIGHT);
    cogl_pop_matrix ();
}

static void
hrn_spinner_class_init (HrnSpinnerClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_spinner_dispose;
    o_class->finalize = hrn_spinner_finalize;
    o_class->set_property = hrn_spinner_set_property;
    o_class->get_property = hrn_spinner_get_property;

    a_class->paint = hrn_spinner_paint;

    g_type_class_add_private (klass, sizeof (HrnSpinnerPrivate));
}

static void
new_frame_cb (ClutterTimeline *timeline,
              int              msecs,
              HrnSpinner      *self)
{
    HrnSpinnerPrivate *priv = self->priv;
    int old_frame;

    old_frame = priv->frame;

    /* We go from 1->23 frames */
    priv->frame = (22 * ((float) msecs / 1000.0)) + 1;

    if (priv->frame != old_frame) {
        clutter_actor_queue_redraw ((ClutterActor *) self);
    }
}

static void
hrn_spinner_init (HrnSpinner *self)
{
    HrnSpinnerPrivate *priv;

    self->priv = GET_PRIVATE (self);
    priv = self->priv;

    priv->timeline = clutter_timeline_new (1000);
    clutter_timeline_set_loop (priv->timeline, TRUE);
    g_signal_connect (priv->timeline, "new-frame",
                      G_CALLBACK (new_frame_cb), self);
    priv->frame = 1;

    priv->material = cogl_material_new ();
}

HrnSpinner *
hrn_spinner_new (const char *filename)
{
    HrnSpinner *spinner;
    HrnSpinnerPrivate *priv;
    GError *error = NULL;

    spinner = g_object_new (HRN_TYPE_SPINNER, NULL);
    priv = spinner->priv;

    priv->texture = cogl_texture_new_from_file (filename,
                                                COGL_TEXTURE_NONE,
                                                COGL_PIXEL_FORMAT_ANY, &error);
    if (error != NULL) {
        g_warning ("Error loading %s: %s", filename, error->message);
        g_error_free (error);

        g_object_unref (spinner);
        return NULL;
    }

    cogl_material_set_layer (priv->material, 0, priv->texture);

    return spinner;
}

void
hrn_spinner_start (HrnSpinner *spinner)
{
    HrnSpinnerPrivate *priv = spinner->priv;

    clutter_timeline_start (priv->timeline);
}

void
hrn_spinner_stop (HrnSpinner *spinner)
{
    HrnSpinnerPrivate *priv = spinner->priv;

    clutter_timeline_stop (priv->timeline);
}
