/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <clutter/clutter.h>
#include <cogl/cogl.h>

#include <nbtk/nbtk.h>

#include "hrn.h" /* FIXME */
#include "hrn-marshal.h"
#include "hrn-sidebar.h"
#include "hrn-sidebar-item.h"
#include "hrn-sidebar-item-factory.h"
#include "hrn-sidebar-subitem.h"

enum {
  PROP_0,
  PROP_PIN_MANAGER,
};

enum {
  SOURCE_CHANGED,
  LAST_SIGNAL
};

G_DEFINE_TYPE (HrnSidebar, hrn_sidebar, NBTK_TYPE_BIN);

#define HRN_SIDEBAR_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_SIDEBAR, HrnSidebarPrivate))

struct _HrnSidebarPrivate
{
  NbtkWidget *scroll;
  NbtkWidget *listview;

  ClutterModel *sources;

  HrnSidebarItemFactory *factory;
  HrnPinManager *pin_manager;
};

static guint32 signals[LAST_SIGNAL] = {0, };

static GObject *
hrn_sidebar_constructor (GType type, guint n_params,
                         GObjectConstructParam *params);
static void
hrn_sidebar_dispose (GObject *object);
static void
hrn_sidebar_allocate (ClutterActor *self, const ClutterActorBox *box,
                      ClutterAllocationFlags flags);
static void
hrn_sidebar_paint (ClutterActor *actor);
static void
hrn_sidebar_pick (ClutterActor *actor, const ClutterColor   *color);

static void
hrn_sidebar_map (ClutterActor *self)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->map (self);

  if (priv->scroll)
    clutter_actor_map (CLUTTER_ACTOR (priv->scroll));
}

static void
hrn_sidebar_unmap (ClutterActor *self)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->unmap (self);

  if (priv->scroll)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->scroll));
}

static void
hrn_sidebar_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
  HrnSidebar *sidebar = (HrnSidebar *) object;
  HrnSidebarPrivate *priv = sidebar->priv;

  switch (prop_id) {
  case PROP_PIN_MANAGER:
    priv->pin_manager = g_value_get_object (value);
    break;

  default:
    break;
  }
}

static void
hrn_sidebar_get_property (GObject *object,
                          guint prop_id,
                          GValue *value,
                          GParamSpec *pspec)
{
}

static void
hrn_sidebar_class_init (HrnSidebarClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);


  gobject_class->dispose     = hrn_sidebar_dispose;
  gobject_class->constructor = hrn_sidebar_constructor;
  gobject_class->set_property = hrn_sidebar_set_property;
  gobject_class->get_property = hrn_sidebar_get_property;

  actor_class->allocate      = hrn_sidebar_allocate;
  actor_class->paint         = hrn_sidebar_paint;
  actor_class->pick          = hrn_sidebar_pick;
  actor_class->map           = hrn_sidebar_map;
  actor_class->unmap         = hrn_sidebar_unmap;

  g_type_class_add_private (gobject_class, sizeof (HrnSidebarPrivate));

  g_object_class_install_property (gobject_class, PROP_PIN_MANAGER,
                                   g_param_spec_object ("pin-manager", "", "",
                                                        HRN_TYPE_PIN_MANAGER,
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_CONSTRUCT_ONLY |
                                                        G_PARAM_STATIC_STRINGS));
  signals[SOURCE_CHANGED] = g_signal_new ("source-changed",
                                          G_TYPE_FROM_CLASS (klass),
                                          G_SIGNAL_RUN_FIRST |
                                          G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                          hrn_marshal_VOID__POINTER_INT_POINTER,
                                          G_TYPE_NONE, 3, G_TYPE_POINTER,
                                          G_TYPE_INT, G_TYPE_POINTER);
}

static void
hrn_sidebar_init (HrnSidebar *self)
{
  self->priv = HRN_SIDEBAR_GET_PRIVATE (self);
}

#define SIDEBAR_WIDTH           170
#define SIDEBAR_WIDTH_PADDED    200

void
hrn_sidebar_add_source (HrnSidebar *sidebar,
                        HrnSource  *source,
                        gboolean is_local)
{
  HrnSidebarPrivate *priv = sidebar->priv;
  const char *name;

  if (is_local) {
    name = _("Your local media");
  } else {
    name = hrn_source_get_name (source);
  }

  clutter_model_append (priv->sources,
                        0, source,
                        1, name, -1);
}

void
hrn_sidebar_remove_source (HrnSidebar *sidebar,
                           HrnSource  *source)
{
  HrnSidebarPrivate *priv = sidebar->priv;
  ClutterModelIter *iter;

  iter = clutter_model_get_first_iter (priv->sources);
  while (iter) {
    HrnSource *model_source;
    clutter_model_iter_get (iter, 0, &model_source, -1);
    if (model_source == source) {
      guint row = clutter_model_iter_get_row (iter);

      clutter_model_remove (priv->sources, row);
      g_object_unref (model_source);
      return;
    }

    g_object_unref (model_source);
    iter = clutter_model_iter_next (iter);
  }
}

static GObject *
hrn_sidebar_constructor (GType type, guint n_params,
                         GObjectConstructParam *params)
{
  HrnSidebarPrivate *priv;
  GObject           *object;
  HrnSidebar        *sidebar;

  object = G_OBJECT_CLASS (hrn_sidebar_parent_class)->constructor (type,
                                                                   n_params,
                                                                   params);

  sidebar = HRN_SIDEBAR (object);
  priv    = HRN_SIDEBAR_GET_PRIVATE (sidebar);

  nbtk_widget_set_style_class_name (NBTK_WIDGET (sidebar), "HrnSidebar");
  clutter_actor_set_name (CLUTTER_ACTOR (sidebar), "HrnSidebar");
  clutter_actor_set_reactive ((ClutterActor *) sidebar, TRUE);

  priv->scroll = (NbtkWidget*) nbtk_scroll_view_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->scroll),
                            CLUTTER_ACTOR (sidebar));
  clutter_actor_set_width (CLUTTER_ACTOR (priv->scroll), SIDEBAR_WIDTH);

  priv->sources = clutter_list_model_new (2, HRN_TYPE_SOURCE, "source",
                                          G_TYPE_STRING, "name");

  priv->listview = nbtk_list_view_new ();
  clutter_actor_set_width (CLUTTER_ACTOR (priv->listview), SIDEBAR_WIDTH);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll),
                               CLUTTER_ACTOR (priv->listview));

  nbtk_list_view_set_model (NBTK_LIST_VIEW (priv->listview), priv->sources);

  priv->factory = hrn_sidebar_item_factory_new (sidebar, priv->pin_manager);
  nbtk_list_view_set_factory (NBTK_LIST_VIEW (priv->listview),
                              NBTK_ITEM_FACTORY (priv->factory));

  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->listview),
                                "source", 0);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->listview),
                                "name", 1);

  g_object_set_data (object, "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_SIDEBAR));
  g_object_set_data (G_OBJECT (priv->listview), "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_SIDEBAR));

#if 0
  /* hack to sync up size */
  {
    gfloat width  = SIDEBAR_WIDTH_PADDED;
    int  height;

    gtk_window_get_size (GTK_WINDOW (window), NULL, &height);
    height -= 61;

    clutter_actor_set_size (CLUTTER_ACTOR (priv->listview), width - 40, height);
    clutter_actor_set_size (CLUTTER_ACTOR (priv->scroll), width, height);
    clutter_actor_set_size (CLUTTER_ACTOR (sidebar), width, height);
  }
#endif

  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);

  return object;
}


static void
hrn_sidebar_allocate (ClutterActor          *self,
                      const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (self);
  ClutterActorClass *parent_class;
  ClutterActorBox child_box;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class);
  parent_class->allocate (self, box, flags);

  child_box.x1 = 0;
  child_box.x2 = box->x2 - box->x1;
  child_box.y1 = 0;
  child_box.y2 = (box->y2 - box->y1) - 170;

  clutter_actor_allocate (CLUTTER_ACTOR (priv->scroll), &child_box, flags);
}

static void
hrn_sidebar_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_sidebar_parent_class)->dispose (object);
}

static void
hrn_sidebar_paint (ClutterActor *actor)
{
  HrnSidebar        *sidebar = HRN_SIDEBAR (actor);
  HrnSidebarPrivate *priv    = HRN_SIDEBAR_GET_PRIVATE (sidebar);

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll));
}

static void
hrn_sidebar_pick (ClutterActor *actor, const ClutterColor   *color)
{
  gfloat             width;
  gfloat             height;
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (actor);

  clutter_actor_get_size (CLUTTER_ACTOR (actor), &width, &height);

  cogl_set_source_color ((void*) color);
  cogl_rectangle (0, 0, width, height);

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll));
}

HrnSidebar *
hrn_sidebar_new (HrnPinManager *pin_manager)
{
  HrnSidebar *sidebar = g_object_new (HRN_TYPE_SIDEBAR,
                                      "pin-manager", pin_manager,
                                      NULL);

  return sidebar;
}

void
hrn_sidebar_change_source (HrnSidebar *sidebar,
                           HrnSource  *source,
                           int         filter,
                           const char *query_name)
{
  HrnSidebarPrivate *priv = sidebar->priv;
  HrnPin *pin;

  pin = hrn_pin_manager_get_pin (priv->pin_manager, query_name);
  g_signal_emit (sidebar, signals[SOURCE_CHANGED], 0, source, filter, pin);
}
