/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <glib/gprintf.h>
#include "hrn.h"
#include "hrn-item.h"

G_DEFINE_TYPE (HrnItem, hrn_item, HRN_TYPE_SWITCHER);

#define HRN_ITEM_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_ITEM, HrnItemPrivate))

enum
{
  PROP_0,
  PROP_ITEM
};

static ClutterTexture *frame_tex (void);
static ClutterTexture *frame_tex_zoom_image (void);
static ClutterTexture *frame_tex_zoom_video (void);
static ClutterTexture *frame_tex_zoom_audio (void);
static ClutterTexture *frame_tex3 (void);
static ClutterTexture *frame_tex_play_image (void);
static ClutterTexture *frame_tex_play_video (void);
static ClutterTexture *frame_tex_play_audio (void);
static ClutterTexture *frame_tex5 (void);

/* XXX: these files are directly referenced when they shouldn't */
gboolean removed_items (gpointer data);

static void
delete_cb (gpointer userdata)
{
  BklItem *item = userdata;
  GFile   *file = g_file_new_for_uri (bkl_item_get_uri (item));

  /* FIXME: When the file is deleted, bickley should notice that it is gone
     and inform us */
  /* hrn_uri_removed (current_source, bkl_item_get_uri (item), NULL); */

  g_file_trash (file, NULL, NULL);
  g_object_unref (file);
  hrn_popup_close ();
  removed_items (NULL);  /* XXX: we will reposition twice when removing,
                          * first when removing the item itself, then
                          * when bickley tells ut it has been removed
                          */
}

static gpointer actions[] = { N_ ("Delete"), delete_cb, NULL };

HrnView *
hrn_item_get_view (ClutterActor *actor)
{
  while (actor)
    {
      if (HRN_IS_VIEW (actor))
        return HRN_VIEW (actor);
      actor = clutter_actor_get_parent (actor);
    }
  return NULL;
}

#if HRN_USE_NORMAL_LABELS
static void ensure_labels (HrnItem *item, gint group);
#endif

struct _HrnItemPrivate
{
  void         *unused;
  gboolean      selected;
  ClutterActor *frame;
};

ClutterColor white = { 0xff, 0xff, 0xff, 0xff };

static GObject *
hrn_item_constructor (GType type, guint n_params, GObjectConstructParam *params);
static void
hrn_item_dispose (GObject *object);


static void
hrn_item_allocate (ClutterActor *self, const ClutterActorBox *box,
                   ClutterAllocationFlags flags)
{
  ClutterActorClass *parent_class;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_item_parent_class);

  parent_class->allocate (self, box, flags);



#if HRN_USE_NORMAL_LABELS
    {
      switch (hrn_view_labels_visible (hrn_item_get_view (self)))
        {
          case ZOOM_ITEMS_8:
            ensure_labels (HRN_ITEM (self), 1);
            break;

          case ZOOM_ITEMS_4:
          case ZOOM_ITEMS_3:
            ensure_labels (HRN_ITEM (self), 2);
            ensure_labels (HRN_ITEM (self), 2);
            break;
        }
    }


  if (HRN_ITEM (self)->title)
    {
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (self)->
                                                            title), flags);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (self)->
                                                            meta), flags);
    }

  if (HRN_ITEM (self)->title2)
    {
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (self)->
                                                            title2), flags);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (self)->
                                                            meta2), flags);
    }
#endif


  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (self)->priv->frame),
                                         flags);
}


gfloat
compute_static_x (gint cluster_no, gint accumpos, gint power_of_two);
gfloat
compute_static_y (gint cluster_no, gint accumpos, gint power_of_two);


static void
hrn_item_paint (ClutterActor *actor)
{
  ClutterActorClass *parent_class;
  HrnItem           *item = HRN_ITEM (actor);

  parent_class = CLUTTER_ACTOR_CLASS (hrn_item_parent_class);

  hrn_switcher_adapt (HRN_SWITCHER (actor));
  parent_class->paint (actor);

  /* these checks are probably a bit too expensive to be done in paint */
  if (hrn_item_get_selected (HRN_ITEM (actor)) == FALSE)
    {
      if (hrn_view_is_selected (bkl_item_get_uri (HRN_ITEM (actor)->item)))
        hrn_item_set_selected (HRN_ITEM (actor), TRUE);
    }
  else
    {
      if (!hrn_view_is_selected (bkl_item_get_uri (HRN_ITEM (actor)->item)))
        hrn_item_set_selected (HRN_ITEM (actor), FALSE);
    }

  if (hrn_switcher_get_state (HRN_SWITCHER (actor)) > 0)
    {
      if (!hrn_item_get_view (actor))
        return;
      switch (hrn_view_labels_visible (hrn_item_get_view (actor)))
        {
          case ZOOM_ITEMS_8:
            if (item->title)
              clutter_actor_paint (CLUTTER_ACTOR (item->title));
            if (item->meta)
              clutter_actor_paint (CLUTTER_ACTOR (item->meta));
            break;

          case ZOOM_ITEMS_4:
            if (item->title2)
              clutter_actor_paint (CLUTTER_ACTOR (item->title2));
            if (item->meta2)
              clutter_actor_paint (CLUTTER_ACTOR (item->meta2));
            break;

          case ZOOM_ITEMS_3:
            if (item->title2)
              clutter_actor_paint (CLUTTER_ACTOR (item->title2));
            if (item->meta2)
              clutter_actor_paint (CLUTTER_ACTOR (item->meta2));
        }
    }
  clutter_actor_paint (CLUTTER_ACTOR (item->priv->frame));
}

static void
hrn_item_pick (ClutterActor *actor, const ClutterColor *color)
{
  return hrn_item_paint (actor);
}

static void hrn_item_set_property (GObject *gobject, guint prop_id,
                                   const GValue *value,
                                   GParamSpec   *pspec);

static void hrn_item_get_property (GObject *gobject, guint prop_id,
                                   GValue     *value,
                                   GParamSpec *pspec);

static void
hrn_item_map (ClutterActor *self)
{
  HrnItem *item = HRN_ITEM (self);

  CLUTTER_ACTOR_CLASS (hrn_item_parent_class)->map (self);

  if (item->title)
    clutter_actor_map (CLUTTER_ACTOR (item->title));
  if (item->meta)
    clutter_actor_map (CLUTTER_ACTOR (item->meta));
  if (item->title2)
    clutter_actor_map (CLUTTER_ACTOR (item->title2));
  if (item->meta2)
    clutter_actor_map (CLUTTER_ACTOR (item->meta2));
  if (item->priv->frame)
    clutter_actor_map (CLUTTER_ACTOR (item->priv->frame));
}

static void
hrn_item_unmap (ClutterActor *self)
{
  HrnItem *item = HRN_ITEM (self);

  CLUTTER_ACTOR_CLASS (hrn_item_parent_class)->unmap (self);

  if (item->title)
    clutter_actor_unmap (CLUTTER_ACTOR (item->title));
  if (item->meta)
    clutter_actor_unmap (CLUTTER_ACTOR (item->meta));
  if (item->title2)
    clutter_actor_unmap (CLUTTER_ACTOR (item->title2));
  if (item->meta2)
    clutter_actor_unmap (CLUTTER_ACTOR (item->meta2));
  if (item->priv->frame)
    clutter_actor_unmap (CLUTTER_ACTOR (item->priv->frame));
}



static void
hrn_item_class_init (HrnItemClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec        *pspec;

  gobject_class->dispose      = hrn_item_dispose;
  gobject_class->constructor  = hrn_item_constructor;
  actor_class->paint          = hrn_item_paint;
  actor_class->pick           = hrn_item_pick;
  actor_class->map            = hrn_item_map;
  actor_class->unmap          = hrn_item_unmap;
  actor_class->allocate       = hrn_item_allocate;
  gobject_class->set_property = hrn_item_set_property;
  gobject_class->get_property = hrn_item_get_property;

  pspec = g_param_spec_object ("item",
                               "BklItem",
                               "",
                               BKL_TYPE_ITEM,
                               G_PARAM_READWRITE |
                               G_PARAM_CONSTRUCT_ONLY);
  g_object_class_install_property (gobject_class, PROP_ITEM, pspec);

  g_type_class_add_private (gobject_class, sizeof (HrnItemPrivate));
}

static void
hrn_item_init (HrnItem *self)
{
  HrnItemPrivate *priv = HRN_ITEM_GET_PRIVATE (self);

  memset (priv, 0, sizeof (priv));
  self->priv = priv;
  priv->frame = hrn_texture_frame_new (frame_tex (), 8, 64, 8, 64);
  clutter_actor_set_parent (priv->frame, CLUTTER_ACTOR (self));
  clutter_actor_set_size (priv->frame, DIM/2, DIM/2);
  clutter_actor_set_scale (priv->frame, 2.0, 2.0);
  hrn_texture_frame_set_draw_middle (HRN_TEXTURE_FRAME (priv->frame), TRUE);
  hrn_texture_frame_set_can_pinch (HRN_TEXTURE_FRAME (priv->frame), TRUE);
  hrn_texture_frame_set_scale_invariant (HRN_TEXTURE_FRAME (priv->frame), TRUE);
}

#if HRN_USE_NORMAL_LABELS

static gboolean
entered (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  return FALSE;
}

static void
quicksearch_cb (NbtkButton  *button,
                gpointer     data)
{
#if 0
   hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar,
                          nbtk_button_get_label (button));
#else
   /* FIXME: Ideally we'd pass some signals up through the hierarchy */
   hrn_set_query (nbtk_button_get_label (button));
#endif
}

static void
ensure_labels (HrnItem *item, gint group)
{
  if (item->title && group == 1)
    return;
  if (item->title2 && group > 1)
    return;

  if (hrn_switcher_get_state (HRN_SWITCHER (item)) < 1)
    return;

  if (group == 1)
    {
      item->title = hrn_button_new ();
      item->meta  = hrn_button_new ();

      g_signal_connect (item->title, "clicked",
                        G_CALLBACK (quicksearch_cb), item);
      g_signal_connect (item->meta, "clicked",
                        G_CALLBACK (quicksearch_cb), item);

      if (item->item)
        {
          gchar *buf = hrn_item_get_title (item->item);
          nbtk_button_set_label (NBTK_BUTTON (HRN_ITEM (item)->title), buf);
          g_free (buf);

          buf = hrn_item_get_meta (item->item);
          nbtk_button_set_label (NBTK_BUTTON (HRN_ITEM (item)->meta), buf);
          g_free (buf);

          nbtk_bin_set_alignment (NBTK_BIN (HRN_ITEM (item)->title), 0.0, 0.0);
          nbtk_bin_set_alignment (NBTK_BIN (HRN_ITEM (item)->meta), 0.0, 0.0);
        }

      clutter_actor_set_scale (CLUTTER_ACTOR (item->title), 1 / 0.46, 1 / 0.46);
      clutter_actor_set_scale (CLUTTER_ACTOR (item->meta), 1 / 0.46, 1 / 0.46);

      nbtk_widget_set_style_class_name (item->title, "HrnItemTitle");
      nbtk_widget_set_style_class_name (item->meta, "HrnItemMeta");

      clutter_actor_set_width (CLUTTER_ACTOR (item->title), DIM / 2 * 0.9);
      clutter_actor_set_width (CLUTTER_ACTOR (item->meta), DIM / 2 * 0.9);

      clutter_actor_set_position (CLUTTER_ACTOR (item->title), 2, (gint) DIM);
      clutter_actor_set_position (CLUTTER_ACTOR (item->meta), 2, (gint) DIM +
                                  clutter_actor_get_height (CLUTTER_ACTOR (
                                                              item
                                                              ->
                                                              title))
                                  * 2.2);

      clutter_actor_set_parent (CLUTTER_ACTOR (item->title),
                                CLUTTER_ACTOR (item));
      clutter_actor_set_parent (CLUTTER_ACTOR (item->meta), CLUTTER_ACTOR (item));


      g_signal_connect (HRN_ITEM (item)->meta, "enter-event",
                        G_CALLBACK (entered), NULL);
      clutter_actor_set_reactive (CLUTTER_ACTOR (HRN_ITEM (item)->meta), TRUE);

      /* we allocate here as well, since these actors might have been added on
       * demand for the paint */
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (item)->
                                                            title), TRUE);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (item)->
                                                            meta), TRUE);
    }
  else
    {
      item->title2 = hrn_button_new ();
      item->meta2  = hrn_button_new ();

      g_signal_connect (item->title2, "clicked",
                        G_CALLBACK (quicksearch_cb), item);
      g_signal_connect (item->meta2, "clicked",
                        G_CALLBACK (quicksearch_cb), item);

      if (item->item)
        {
          gchar *buf = hrn_item_get_title (item->item);
          nbtk_button_set_label (NBTK_BUTTON (HRN_ITEM (item)->title2), buf);
          g_free (buf);

          buf = hrn_item_get_meta (item->item);
          nbtk_button_set_label (NBTK_BUTTON (HRN_ITEM (item)->meta2), buf);
          g_free (buf);

          nbtk_bin_set_alignment (NBTK_BIN (HRN_ITEM (item)->title2), 0.0, 0.0);
          nbtk_bin_set_alignment (NBTK_BIN (HRN_ITEM (item)->meta2), 0.0, 0.0);
        }

      clutter_actor_set_scale (CLUTTER_ACTOR (item->title2), 1.0, 1.0);
      clutter_actor_set_scale (CLUTTER_ACTOR (item->meta2), 1.0, 1.0);

      nbtk_widget_set_style_class_name (item->title2, "HrnItemTitle2");
      nbtk_widget_set_style_class_name (item->meta2, "HrnItemMeta2");

      clutter_actor_set_width (CLUTTER_ACTOR (item->title2), DIM * 0.9);
      clutter_actor_set_width (CLUTTER_ACTOR (item->meta2), DIM * 0.9);

      clutter_actor_set_position (CLUTTER_ACTOR (item->title2), 2, (gint) DIM);
      clutter_actor_set_position (CLUTTER_ACTOR (item->meta2), 2, (gint) DIM +
                                  clutter_actor_get_height (CLUTTER_ACTOR (
                                                              item
                                                              ->
                                                              title2))
                                  * 1.1);

      clutter_actor_set_parent (CLUTTER_ACTOR (item->title2),
                                CLUTTER_ACTOR (item));
      clutter_actor_set_parent (CLUTTER_ACTOR (item->meta2),
                                CLUTTER_ACTOR (item));


      /* we allocate here as well, since these actors might have been added on
       * demand for the paint */
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (item)->
                                                            title2), TRUE);
      clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (HRN_ITEM (item)->
                                                            meta2), TRUE);
    }
}
#endif

static GObject *
hrn_item_constructor (GType type, guint n_params, GObjectConstructParam *params)
{
  GObject *object;

  object = G_OBJECT_CLASS (hrn_item_parent_class)->constructor (
    type, n_params, params);

  clutter_actor_set_size (CLUTTER_ACTOR (object), DIM, DIM);

  return object;
}

static void
hrn_item_dispose (GObject *object)
{
#if HRN_USE_NORMAL_LABELS
  {
    HrnItem *item = HRN_ITEM (object);
    if (item->title)
      clutter_actor_destroy (CLUTTER_ACTOR (item->title));
    if (item->meta)
      clutter_actor_destroy (CLUTTER_ACTOR (item->meta));
    if (item->title2)
      clutter_actor_destroy (CLUTTER_ACTOR (item->title2));
    if (item->meta2)
      clutter_actor_destroy (CLUTTER_ACTOR (item->meta2));
    item->title = item->meta = item->title2 = item->meta2 = NULL;
  }
#endif
  G_OBJECT_CLASS (hrn_item_parent_class)->dispose (object);
}

#include <string.h>

ClutterActor *
hrn_make_item (BklItem *item, gboolean square)
{
  /* avoid creating an item at all, return an empty rendering
   * actor anyways, keeping remaining code more flexible for now
   */
  if (item == NULL)
    return g_object_new (HRN_TYPE_ITEM, NULL);

  if (IS_BKL_ITEM_IMAGE (item))
    {
      return CLUTTER_ACTOR (hrn_item_image_create (item, square));
    }
  else if (IS_BKL_ITEM_AUDIO (item))
    {
      return CLUTTER_ACTOR (hrn_item_audio_create (item, square));
    }
  else if (IS_BKL_ITEM_VIDEO (item))
    {
      return CLUTTER_ACTOR (hrn_item_video_create (item, square));
    }
  else
    {
      g_debug ("unhandled type %s\n", G_OBJECT_TYPE_NAME (item));
      return NULL;
    }
}

/* returns a human readable title guessed from the uri in an
 * allocated buffer (strips extension, and makes -_. into spaces)
 */
gchar *
bkl_item_uri_to_title (BklItem *item)
{
  gchar *basename = g_path_get_basename (bkl_item_get_uri (item));
  gchar *lastdot;
  gchar *p;

  lastdot = strrchr (basename, '.');
  if (lastdot)
    *lastdot = '\0';
  for (p = basename; *p; p++)
    {
      switch (*p)
        {
          case '_':
          case '-':
          case '.':
            *p = ' ';

          default:;
        }
    }

  return basename;
}

gchar *
hrn_item_get_title (BklItem *item)
{
  const gchar *title;

  switch (bkl_item_get_item_type (item))
    {
      case BKL_ITEM_TYPE_VIDEO:
        title = bkl_item_video_get_title (BKL_ITEM_VIDEO (item));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        title = bkl_item_audio_get_title (BKL_ITEM_AUDIO (item));
        break;

      case BKL_ITEM_TYPE_IMAGE:
        title = bkl_item_image_get_title (BKL_ITEM_IMAGE (item));
        break;

      default:
        title = "unhandled type";
        break;
    }
  if (title)
    {
      return g_strdup (title);
    }
  return bkl_item_uri_to_title (item);
}


/* returns a human readable category guessed from the uri (using the
 * parent directoys name in an allocated buffer.
 *
 * This should perhaps be used in an abstraction that returns better
 * categorization data like album or series.
 */
gchar *
hrn_item_uri_parent_to_meta (BklItem *item)
{
  char *parent_dir, *basename;
  char *p;

  if (bkl_item_get_uri (item) == NULL)
    return g_strdup ("eeek");

  parent_dir = g_path_get_dirname (bkl_item_get_uri (item));
  basename = g_path_get_basename (parent_dir);
  g_free (parent_dir);

  for (p = basename; *p; p++)
    {
      switch (*p)
        {
          case '_':
          case '-':
          case '.':
            *p = ' ';

          default:;
        }
    }

  return basename;
}

/* this should perhaps do different things for image/video/audio */
gchar *
hrn_item_get_meta (BklItem *item)
{
  gchar        buf[256];
  const gchar *meta = NULL;

  switch (bkl_item_get_item_type (item))
    {
      case BKL_ITEM_TYPE_VIDEO:
        meta = bkl_item_video_get_series_name (BKL_ITEM_VIDEO (item));
        /* XXX: directory name */
        if (!meta)
          meta = bkl_item_video_get_director (BKL_ITEM_VIDEO (item));

        if (!meta)
          {
            if (bkl_item_video_get_duration (BKL_ITEM_VIDEO (item)))
              {
                g_snprintf (buf, 255, "%2i:%02i",
                            bkl_item_video_get_duration (BKL_ITEM_VIDEO (
                                                           item)) / 60,
                            bkl_item_video_get_duration (BKL_ITEM_VIDEO (
                                                           item)) % 60);
                meta = buf;
              }
          }
        break;

      case BKL_ITEM_TYPE_AUDIO:
        if (!meta)
          meta = bkl_item_audio_get_album (BKL_ITEM_AUDIO (item));
        if (!meta)
          {
            GPtrArray *artists =
              bkl_item_audio_get_artists (BKL_ITEM_AUDIO (item));
            if (artists)
              meta = artists->pdata[0];
          }
        if (!meta)
          meta = bkl_item_audio_get_composer (BKL_ITEM_AUDIO (item));
        break;

      case BKL_ITEM_TYPE_IMAGE:
        meta = bkl_item_image_get_time (BKL_ITEM_IMAGE (item));
        /* FIXME: use location */
        if (!meta)
          meta = bkl_item_image_get_time_original (BKL_ITEM_IMAGE (item));
        if (!meta)
          meta = bkl_item_image_get_time_digitized (BKL_ITEM_IMAGE (item));
        if (!meta)
          {
            gint width, height;
            width  = bkl_item_image_get_width (BKL_ITEM_IMAGE (item));
            height = bkl_item_image_get_height (BKL_ITEM_IMAGE (item));
            return g_strdup_printf ("%ux%u", width, height);
          }
        break;

      default:
        break;
    }
  if (meta)
    {
      return g_strdup (meta);
    }
  return hrn_item_uri_parent_to_meta (item);
}

static void
hrn_item_set_property (GObject *gobject, guint prop_id, const GValue *value,
                       GParamSpec   *pspec)
{
  HrnItem *item = HRN_ITEM (gobject);

  switch (prop_id)
    {
      case PROP_ITEM:
        if (item->item)
          {
            /*XXX:unref*/
            item->item = NULL;
          }
        item->item = g_value_get_object (value);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

static void
hrn_item_get_property (GObject *gobject, guint prop_id, GValue     *value,
                       GParamSpec *pspec)
{
  HrnItem *item = HRN_ITEM (gobject);

  switch (prop_id)
    {
      case PROP_ITEM:
        g_value_set_object (value, item->item);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

#define MODE_NORMAL            0
#define MODE_ZOOM              1
#define MODE_SELECTED          2
#define MODE_SELECTED_HOVER    3
#define MODE_PLAY              4


static void
item_set_frame (HrnItem *item, gint no)
{
  HrnTextureFrame *frame = HRN_TEXTURE_FRAME (item->priv->frame);

  switch (no)
    {
      case MODE_ZOOM: 

	  switch (bkl_item_get_item_type (item->item))
	    {
	      case BKL_ITEM_TYPE_IMAGE:
                g_object_set (frame, "parent-texture",
                              frame_tex_zoom_image (), NULL);
		break;
	      case BKL_ITEM_TYPE_AUDIO:
                g_object_set (frame, "parent-texture",
                              frame_tex_zoom_audio (), NULL);
		break;
              default:
	      case BKL_ITEM_TYPE_VIDEO:
                g_object_set (frame, "parent-texture",
                              frame_tex_zoom_video (), NULL);
		break;
	    }
          break;

      case MODE_PLAY: 

	  switch (bkl_item_get_item_type (item->item))
	    {
	      case BKL_ITEM_TYPE_IMAGE:
                g_object_set (frame, "parent-texture",
                              frame_tex_play_image (), NULL);
		break;
	      case BKL_ITEM_TYPE_AUDIO:
                g_object_set (frame, "parent-texture",
                              frame_tex_play_audio (), NULL);
		break;
              default:
	      case BKL_ITEM_TYPE_VIDEO:
                g_object_set (frame, "parent-texture",
                              frame_tex_play_video (), NULL);
		break;
	    }
          break;

      case MODE_SELECTED: g_object_set (frame, "parent-texture",
                                        frame_tex3 (), NULL); break;

      case MODE_SELECTED_HOVER: g_object_set (frame, "parent-texture",
                                              frame_tex5 (), NULL); break;


      case MODE_NORMAL:
      default: g_object_set (frame, "parent-texture", frame_tex (), NULL);
    }
  hrn_texture_frame_set_draw_middle (frame, TRUE); /* this should be set only
                                                            for the modes that
                                                      need it */
}


static gboolean framed_enter (ClutterActor *actor,
                              ClutterEvent *event,
                              gpointer      item);
static gboolean framed_leave (ClutterActor *actor,
                              ClutterEvent *event,
                              gpointer      item);


static gboolean
playbutton_enter (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  if (hrn_get_zoom () >= 1.0)
    {
      if (hrn_item_get_selected (item))
        item_set_frame (item, MODE_SELECTED);
      else
        item_set_frame (item, MODE_PLAY);
    }
  return FALSE;
}

static gboolean
playbutton_leave (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  if (hrn_item_get_selected (item))
    item_set_frame (item, MODE_SELECTED_HOVER);
  else
    item_set_frame (item, MODE_ZOOM);

  /* Because the actors used for the hit detection are not ancestors,
   * we forward this leave to the rectangle representing the item as
   * well
   */
  return framed_leave (actor, event, item);
}






static void
hrn_view_toggle_selected (HrnView *view, const gchar *uri, BklItem     *item)
{
  if (hrn_view_is_selected (uri))
    {
      hrn_view_set_selected (view, uri, item, FALSE);
    }
  else
    {
      hrn_view_set_selected (view, uri, item, TRUE);
    }
  clutter_actor_queue_redraw (CLUTTER_ACTOR (view));
}

static gboolean
playbutton_release (ClutterActor *actor, ClutterEvent *event,
                    HrnItem      *item)
{
  gint          i = 0;
  ClutterActor *cluster;

  if (clutter_event_get_state (event) & CLUTTER_CONTROL_MASK)
    {
      hrn_view_toggle_selected (hrn_item_get_view (CLUTTER_ACTOR (
                                                     item)),
                                bkl_item_get_uri (item->item), item->item);
      return TRUE;
    }


  if (event->button.button == 3)
    {
      hrn_popup_actor (event->button.x, event->button.y,
                       hrn_popup_actions ((void*) actions, item->item));
      return TRUE;
    }
  cluster = clutter_actor_get_parent (CLUTTER_ACTOR (item));
  while (cluster && !HRN_IS_SWITCHER (cluster))
    cluster = clutter_actor_get_parent (cluster);

  {
    GList *children;
    GList *iter;
    children = clutter_container_get_children (
      CLUTTER_CONTAINER (hrn_switcher_get_current (HRN_SWITCHER (cluster))));
    gint     n       = 0;
    gboolean foundit = FALSE;

    for (iter = children; iter; iter = iter->next)
      {
        if (iter->data == item)
          {
            foundit = TRUE;
          }
        if (!foundit)
          i++;
        n++;
      }

    g_list_free (children);
    i = n - i - 1; /* playbutton is in there as well */
  }

  /* fixme: this should be abstracted up a level to the view */
  if (bkl_item_get_item_type (HRN_ITEM (item)->item) != BKL_ITEM_TYPE_AUDIO) {
      hrn_set_theatre ();
  }
  hrn_play_now (HRN_ITEM (item)->item);

  hrn_view_set_pos (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cluster),
                                                        "accumpos")) + i);


  return TRUE;
}

static gboolean
item_release (ClutterActor *actor, ClutterEvent *event, HrnItem      *item)
{
  gint          i = 0;
  ClutterActor *cluster;

  if (clutter_event_get_state (event) & CLUTTER_CONTROL_MASK)
    {
      hrn_view_toggle_selected (hrn_item_get_view (CLUTTER_ACTOR (item)),
                                bkl_item_get_uri (item->item), item->item);
      return TRUE;
    }

  if (event->button.button == 3)
    {
      hrn_popup_actor (event->button.x, event->button.y,
                       hrn_popup_actions ((void*) actions, item->item));
      return TRUE;
    }


  cluster = clutter_actor_get_parent (CLUTTER_ACTOR (item));
  while (cluster && !HRN_IS_SWITCHER (cluster))
    cluster = clutter_actor_get_parent (cluster);

  {
    GList *children;
    GList *iter;
    children = clutter_container_get_children (
      CLUTTER_CONTAINER (hrn_switcher_get_current (HRN_SWITCHER (cluster))));
    gint     n       = 0;
    gboolean foundit = FALSE;

    for (iter = children; iter; iter = iter->next)
      {
        if (iter->data == item)
          {
            foundit = TRUE;
          }
        if (!foundit)
          i++;
        n++;
      }

    g_list_free (children);
    i = n - i - 1; /* playbutton is in there as well */
  }

  hrn_view_set_pos (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cluster),
                                                        "accumpos")) + i);

  if (hrn_get_zoom () > 3.8)
    {
      hrn_set_theatre ();
      hrn_play_now (HRN_ITEM (item)->item);
    }
  else
    {
      hrn_set_zoom (hrn_get_zoom () * 2.0);
    }

  return TRUE;
}




static gboolean
playbutton_prepaint (ClutterActor *actor,
                     gpointer      data)
{
  gdouble zoom = hrn_get_zoom ();

  if (zoom < 2.0)
    {
      clutter_actor_set_size (actor, DIM / 3, DIM / 3);
    }
  else if (zoom < 4.0)
    {
      clutter_actor_set_size (actor, DIM /2, DIM /2);
    }
  else
    {
      clutter_actor_set_size (actor, DIM /3, DIM /3);

    }
  return TRUE;
}

ClutterActor *
hrn_play_button_new (HrnSwitcher *switcher, ClutterActor *group,
                     BklItem      *item)
{
  ClutterActor *foo = clutter_rectangle_new ();

  clutter_group_add (CLUTTER_GROUP (group), foo);
  clutter_actor_set_size (foo, DIM, DIM);
  clutter_actor_set_opacity (foo, 0x00);

  hrn_actor_make_draggable (
    CLUTTER_ACTOR (group), foo,
    HRN_DROP_MASK_QUEUE | HRN_DROP_MASK_SIDEBAR,
    NULL,
    G_CALLBACK (hrn_item_can_drop),
    G_CALLBACK (hrn_item_drop),
    item);
  g_signal_connect (foo, "button-release-event", G_CALLBACK (
                      item_release), switcher);

  ClutterActor *play_button = CLUTTER_ACTOR (hrn_button_new ());
  clutter_actor_set_name (play_button, "quick-play");
  clutter_group_add (CLUTTER_GROUP (group), play_button);
  clutter_actor_set_opacity (play_button, 0xcc);

  clutter_actor_set_size (play_button, DIM / 3.5, DIM / 3.5);
  clutter_actor_set_reactive (play_button, TRUE);
  clutter_actor_set_reactive (foo, TRUE);
  clutter_actor_set_position (play_button, DIM, DIM);
  clutter_actor_set_anchor_point_from_gravity (play_button,
                                               CLUTTER_GRAVITY_SOUTH_EAST);

  g_signal_connect (play_button, "paint", G_CALLBACK (playbutton_prepaint),
                    NULL);

  g_signal_connect (foo, "enter-event", G_CALLBACK (framed_enter), switcher);
  g_signal_connect (foo, "leave-event", G_CALLBACK (framed_leave), switcher);

  g_signal_connect (play_button, "enter-event", G_CALLBACK (playbutton_enter),
                    switcher);
  g_signal_connect (play_button, "leave-event", G_CALLBACK (playbutton_leave),
                    switcher);

  g_signal_connect (play_button, "button-release-event",
                    G_CALLBACK (playbutton_release), switcher);


  hrn_actor_make_draggable (
    CLUTTER_ACTOR (group), play_button,
    HRN_DROP_MASK_QUEUE | HRN_DROP_MASK_SIDEBAR,
    NULL,
    G_CALLBACK (hrn_item_can_drop),
    G_CALLBACK (hrn_item_drop),
    item);
  return play_button;
}


#include "hrn-texture-frame.h"

#if HRN_ITEM_USE_FRAME

static ClutterTexture *
frame_tex (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-frame.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex_zoom_audio (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-music-zoom.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_zoom_video (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-movie-zoom.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_zoom_image (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-image-zoom.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex3 (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-selected.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_play_audio (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-music-quick-play.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_play_video (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-movie-quick-play.png",
                                                    NULL));
  return ret;
}


static ClutterTexture *
frame_tex_play_image (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-image-quick-play.png",
                                                    NULL));
  return ret;
}

static ClutterTexture *
frame_tex5 (void)
{
  static ClutterTexture *ret = NULL;

  if (ret)
    return ret;
  ret =
    CLUTTER_TEXTURE (clutter_texture_new_from_file (PKGDATADIR
                                                    "hrn-item-selected-hover.png",
                                                    NULL));
  return ret;
}



static gboolean
framed_enter (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  if (hrn_get_zoom () >= 1.0 &&
      !hrn_actor_has_ancestor (event->crossing.related, actor))
    {
      if (hrn_item_get_selected (item))
        item_set_frame (item, MODE_SELECTED_HOVER);
      else
        item_set_frame (item, MODE_ZOOM);
    }
  return FALSE;
}

gboolean
hrn_item_get_selected (HrnItem *item)
{
  HrnItemPrivate *priv = HRN_ITEM_GET_PRIVATE (item);

  return priv->selected;
}

static gboolean
framed_leave (ClutterActor *actor, ClutterEvent *event, gpointer item)
{
  /* Sometimes event->crossing.related is an invalid object */
  if (hrn_get_zoom () >= 1.0
      /* && !hrn_actor_has_ancestor (event->crossing.related, actor)*/)
    {
      HrnItemPrivate *priv = HRN_ITEM_GET_PRIVATE (item);

      if (priv->selected)
        {
          item_set_frame (item, MODE_SELECTED);
        }
      else
        {
          item_set_frame (item, MODE_NORMAL);
        }
    }
  return FALSE;
}

void
hrn_item_set_selected (HrnItem *item, gboolean selected)
{
  HrnItemPrivate *priv = HRN_ITEM_GET_PRIVATE (item);

  if (priv->selected == selected)
    return;
  priv->selected = selected;
  if (selected)
    {
      item_set_frame (item, MODE_SELECTED);
    }
  else
    {
      item_set_frame (item, MODE_NORMAL);
    }
}

#endif
