#include "launcherapplication.h"
#include <gconf/gconf-client.h>
#include <QFile>
#include <QProcess>

#define NAME_MAX_LENGTH 50

QLauncherApplication::QLauncherApplication(QObject *parent) :
    QObject(parent), m_application(NULL)
{
}

QLauncherApplication::QLauncherApplication(QString desktop_file) : m_application(NULL)
{
    setDesktopFile(desktop_file);
}


QLauncherApplication::~QLauncherApplication()
{
    if(m_application != NULL)
    {
        /* Disconnect from all GObject properties */
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(runningChangedStatic), this, NULL);
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(nameChangedStatic), this, NULL);
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(iconNameChangedStatic), this, NULL);
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(desktopFileChangedStatic), this, NULL);
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(startupWMClassChangedStatic), this, NULL);
        g_object_disconnect(m_application, "any_signal", G_CALLBACK(connectionTypeChangedStatic), this, NULL);
        m_application = NULL;
    }
}

bool
QLauncherApplication::running() const
{
    if(m_application == NULL) return false;

    return launcher_application_get_running(m_application);
}

QString
QLauncherApplication::name() const
{
    if(m_application == NULL) return QString("");

    const gchar *g_name = launcher_application_get_name(m_application);
    QString name = QString::fromUtf8(g_name);

    /* Workaround the fact that ellipsizing with QML forces the text to be only
       on one line. We want the text to be on multiple lines yet ellipsized. */
    if(name.length() > NAME_MAX_LENGTH)
    {
        QString ellipsis = "...";
        name.truncate(NAME_MAX_LENGTH-ellipsis.length());
        name.append(ellipsis);
    }

    return name;
}


QString
QLauncherApplication::icon_name() const
{
    if(m_application == NULL) return QString("");

    const gchar* name = launcher_application_get_icon_name(m_application);
    if(name == NULL)
        return QString("");
    else
    {
        /* Heuristics to remove file extensions that some .desktop files have
           in their Icon field */
        QString icon_name = QString::fromUtf8(name);
        QStringList parts = icon_name.split(".");

        if(icon_name.startsWith("/") || parts.size() == 1 || parts.last().size() < 3)
            return icon_name;
        else
        {
            parts.removeLast();
            return parts.join("");
        }
    }
}


QString
QLauncherApplication::desktop_file() const
{
    if(m_application == NULL) return QString("");

    return QString::fromUtf8(launcher_application_get_desktop_file(m_application));
}

QString
QLauncherApplication::startup_wm_class() const
{
    if(m_application == NULL) return QString("");

    const gchar* startup_wm_class = launcher_application_get_startup_wm_class(m_application);
    if(startup_wm_class == NULL)
        return QString("");
    else
        return QString::fromUtf8(startup_wm_class);
}

QString
QLauncherApplication::connection_type() const
{
    if(m_application == NULL) return QString("");

    QString type;

    type = QString::fromUtf8(launcher_application_get_connection_type(m_application));

    /* Default to type 'application' if X-Hedley-ConnectionType key is empty */
    if(type.isEmpty())
        return QString("application");
    else
        return type;
}

void
QLauncherApplication::setDesktopFile(QString desktop_file)
{
    if(desktop_file == "") return;

    /* Accept fully formed URL and truncate anything else that is not the actual
       path. Only supports local URLs (e.g. 'file:///home/john' is converted
       into '/home/john') */
    desktop_file = QUrl(desktop_file).path();

     /* Ignore non existing files */
    QFile file(desktop_file);
    if(!file.exists()) return;

    QByteArray byte_array = desktop_file.toUtf8();
    gchar *filename = byte_array.data();

    if(m_application != NULL)
    {
        /* FIXME: that code path has not been thoroughly tested. */
        launcher_application_set_desktop_file(m_application, filename);
    }
    else
    {
        LauncherAppman *appman = launcher_appman_get_default();
        m_application = launcher_appman_get_application_for_desktop_file(appman, filename);

        /* Emit the Changed signal on all properties because m_application is new */
        if(running() != false)
            emit runningChanged();
        emit nameChanged();
        emit iconNameChanged();
        emit desktopFileChanged();
        emit startupWMClassChanged();
        emit connectionTypeChanged();

        /* FIXME: an exception should be raised instead */
        if(m_application == NULL) return;

        /* Connect to all GObject properties */
        g_signal_connect(m_application, "notify::running", G_CALLBACK(runningChangedStatic), this);
        g_signal_connect(m_application, "notify::name", G_CALLBACK(nameChangedStatic), this);
        g_signal_connect(m_application, "notify::icon_name", G_CALLBACK(iconNameChangedStatic), this);
        g_signal_connect(m_application, "notify::desktop_file", G_CALLBACK(desktopFileChangedStatic), this);
        g_signal_connect(m_application, "notify::startup_wm_class", G_CALLBACK(startupWMClassChangedStatic), this);
        g_signal_connect(m_application, "notify::connection_type", G_CALLBACK(connectionTypeChangedStatic), this);
    }
}


/* FIXME: apparently C++ methods with default values are not working
          properly with QML and all values passed are overriden by
          the default values.
          http://bugreports.qt.nokia.com/browse/QTBUG-11604
*/
bool
QLauncherApplication::launch_with_parameter(QString parameter)
{
    if(m_application == NULL) return false;

    GError *error = NULL;
    return (bool)launcher_application_launch(m_application, parameter.toUtf8().data(), &error);
}

bool
QLauncherApplication::launch()
{
    if(m_application == NULL) return false;

    GError *error = NULL;
    return (bool)launcher_application_launch(m_application, NULL, &error);
}

void
QLauncherApplication::close()
{
    if(m_application == NULL) return;

    launcher_application_close(m_application);
}

void
QLauncherApplication::show()
{
    if(m_application == NULL) return;

    launcher_application_show(m_application);
}

void
QLauncherApplication::removeFromConnectionsList(QString id)
{
    GConfClient *client = gconf_client_get_default();

    /* Remove item with 'id' from GConf stored connections list */
    GSList *connections = gconf_client_get_list(client,
                                                "/apps/hedley_launcher/connections/list",
                                                GCONF_VALUE_STRING,
                                                NULL);

    for(GSList *l = connections; l; l = g_slist_next(l))
    {
        gchar *d = (gchar*)l->data;
        if(QString::fromUtf8(d) == id)
        {
            connections = g_slist_remove(connections, d);
            g_free (d);
            break;
        }
    }

    gconf_client_set_list(client,
                          "/apps/hedley_launcher/connections/list",
                          GCONF_VALUE_STRING,
                          connections,
                          NULL);

    g_slist_foreach(connections, (GFunc)g_free, NULL);
    g_slist_free(connections);

    /* Delete GConf key corresponding to item with 'id' */
    QString key = "/apps/hedley_launcher/connections/" + id;
    QByteArray byte_array = key.toUtf8();
    gconf_client_recursive_unset(client, byte_array.data(),
                                 GCONF_UNSET_INCLUDING_SCHEMA_NAMES, NULL);

    g_object_unref(client);
}

void
QLauncherApplication::removeDesktopFile()
{
    /* execute a custom connection removal script passing to it as argument the
       filename of the .desktop file of the connection without its extension. */
    QString basename = desktop_file();
    int index = desktop_file().lastIndexOf("/")+1;
    basename.remove(0, index);
    basename.replace(".desktop", "");

    QString script = "/usr/bin/hedley-delete-action";
    QStringList arguments;
    arguments << basename;

    QProcess script_process;
    script_process.start(script, arguments);
    script_process.waitForFinished();

    /* Remove the actual .desktop file. */
    QFile file(desktop_file());
    file.remove();
}

void
QLauncherApplication::addToClass(const QString& className) const
{
    QString key("/apps/hedley_launcher/" + className + "/classes");
    QString wm_class = startup_wm_class();
    GConfClient* client = gconf_client_get_default();

    GSList* classes = gconf_client_get_list(client, key.toUtf8().constData(),
                                            GCONF_VALUE_STRING, NULL);
    GSList* il;
    bool found = false;

    for(il = classes; il; il = g_slist_next(il))
    {
        gchar* value = (gchar*)(il->data);
        if(QString::fromUtf8(value) == wm_class)
        {
            found = true;
            break;
        }
    }

    if (!found) {
        classes = g_slist_append(classes, g_strdup(wm_class.toUtf8().constData()));
        gconf_client_set_list(client, key.toUtf8().constData(),
                              GCONF_VALUE_STRING, classes, NULL);
    }

    g_slist_foreach(classes, (GFunc)g_free, NULL);
    g_slist_free(classes);

    g_object_unref(client);
}

void
QLauncherApplication::setIsSystemSettings() const
{
    addToClass("system_settings");
}

void
QLauncherApplication::setIsConnectionEditor() const
{
    addToClass("connection_editors");
}

