import cgi
from gettext import lgettext as _
from oauth import oauth

from gwibber.microblog import network, util
from gwibber.microblog.util import resources
from gwibber.microblog.util.time import parsetime
from gwibber.microblog.util.auth import Authentication

import logging
logger = logging.getLogger("Twitter")
logger.debug("Initializing.")

PROTOCOL_INFO = {
  "name": "Twitter",
  "version": "1.0",
  
  "config": [
    "private:secret_token",
    "access_token",
    "username",
    "color",
    "receive_enabled",
    "send_enabled",
  ],
 
  "authtype": "oauth1a",
  "color": "#729FCF",

  "features": [
    "send",
    "receive",
    "search",
    "tag",
    "reply",
    "responses",
    "private",
    "public",
    "delete",
    "follow",
    "unfollow",
    "profile",
    "retweet",
    "like",
    "send_thread",
    "send_private",
    "user_messages",
    "sinceid",
    "lists",
    "list",
  ],

  "default_streams": [
    "receive",
    "images",
    "responses",
    "private",
    "lists",
  ],
}

URL_PREFIX = "https://twitter.com"
API_PREFIX = "https://api.twitter.com/1"

class Client ():
  """Querys Twitter and converts the data.
  
  The Client class is responsible for querying Twitter and turning the data obtained
  into data that Gwibber can understand. Twitter uses a version of OAuth for security.
  
  Tokens have already been obtained when the account was set up in Gwibber and are used to 
  authenticate when getting data.
  
  """
  def __init__(self, acct):
    self.service = util.getbus("Service")
    self.account = acct
    self._loop = None

    self.sigmethod = oauth.OAuthSignatureMethod_HMAC_SHA1()
    self.token = None
    parameters = self.account["auth"]["parameters"]
    self.consumer = oauth.OAuthConsumer(parameters["ConsumerKey"],
                                        parameters["ConsumerSecret"])

  def _login(self):
    old_token = self.account.get("secret_token", None)
    with self.account.login_lock:
      # Perform the login only if it wasn't already performed by another thread
      # while we were waiting to the lock
      if self.account.get("secret_token", None) == old_token:
        self._locked_login(old_token)

    self.token = oauth.OAuthToken(self.account["access_token"],
                                  self.account["secret_token"])
    return "access_token" in self.account and \
        self.account["access_token"] != old_token

  def _locked_login(self, old_token):
    logger.debug("Re-authenticating" if old_token else "Logging in")

    auth = Authentication(self.account, logger)
    reply = auth.login()
    if reply and reply.has_key("AccessToken"):
      self.account["access_token"] = reply["AccessToken"]
      self.account["secret_token"] = reply["TokenSecret"]
      self.account["uid"] = reply["UserId"]
      self.account["username"] = reply["ScreenName"]
      logger.debug("User id is: %s, name is %s" % (self.account["uid"],
                                                   self.account["username"]))
    else:
      logger.error("Didn't find token in session: %s", (reply,))

  def _common(self, data):
    """Parses messages into Gwibber compatible forms.  
    
    This function is common to all tweet types
    and includes parsing of user mentions, hashtags, 
    urls, images and videos.
    
    Arguments: 
    data -- A data object obtained from Twitter containing a complete tweet
    
    Returns: 
    m -- A data object compatible with inserting into the Gwibber database for that tweet
    
    """
    m = {}
    try:
      m["mid"] = str(data["id"])
      m["service"] = "twitter"
      m["account"] = self.account["id"]
      if data.has_key("created_at"):
        m["time"] = parsetime(data["created_at"])
      m["text"] = util.unescape(data["text"])
      m["text"] = cgi.escape(m["text"])
      m["content"] = m["text"]
      
      # Go through the entities in the tweet and use them to linkify/filter tweets as appropriate
      if data.has_key("entities"):

        #Get mention entries
        if data["entities"].has_key("user_mentions"):
          names = []
          for mention in data["entities"]["user_mentions"]:
            if not mention["screen_name"] in names:
              try:
                m["content"] = m["content"].replace("@" + mention["screen_name"], "@<a href='gwibber:/user?acct=" + m["account"] + "&name=@" + mention["screen_name"] + "'>" + mention["screen_name"] + "</a>")
              except:
                pass
            names.append(mention["screen_name"])  

        #Get hashtag entities
        if data["entities"].has_key("hashtags"):
          hashtags = []
          for tag in data["entities"]["hashtags"]:
            if not tag["text"] in hashtags:
              try:
                m["content"] = m["content"].replace("#" + tag["text"], "#<a href='gwibber:/tag?acct=" + m["account"] + "&query=#" + tag["text"] + "'>" + tag["text"] + "</a>")
              except:
                pass
            hashtags.append(tag["text"])
        
        # Get url entities - These usually go in the link stream, but if they're pictures or videos, they should go in the proper stream
        if data["entities"].has_key("urls"):
          for urls in data["entities"]["urls"]:
              url = cgi.escape (urls["url"])
              expanded_url = url
              if urls.has_key("expanded_url"):
                if not urls["expanded_url"] is None:
                  expanded_url = cgi.escape(urls["expanded_url"])

              display_url  = url
              if urls.has_key("display_url"):
                display_url = cgi.escape (urls["display_url"])
 
              if url == m["content"]:
                m["content"] = "<a href='" + url + "' title='" + expanded_url + "'>" + display_url + "</a>"
              else:
                try:
                  startindex = m["content"].index(url)
                  endindex   = startindex + len(url)
                  start      = m["content"][0:startindex]
                  end        = m["content"][endindex:]
                  m["content"] = start + "<a href='" + url + "' title='" + expanded_url + "'>" + display_url + "</a>" + end
                except:
                  logger.debug ("Failed to set url for ID: %s",  m["mid"])

              m["type"] = "link"

              images = util.imgpreview(expanded_url)
              videos = util.videopreview(expanded_url)
              if images:
                m["images"] = images
                m["type"] = "photo"
              elif videos:
                m["images"] = videos
                m["type"] = "video"
              else:
                # Well, it's not anything else, so it must be a link
                m["link"] = {}
                m["link"]["picture"] = ""
                m["link"]["name"] = ""
                m["link"]["description"] = m["content"]
                m["link"]["url"] = url
                m["link"]["icon"] = ""
                m["link"]["caption"] = ""
                m["link"]["properties"] = {}
              
        if data["entities"].has_key("media"):
          for media in data["entities"]["media"]:
            try:
              url = cgi.escape (media["url"])
              media_url_https = media["media_url_https"]
              expanded_url = url
              if media.has_key("expanded_url"):
                expanded_url = cgi.escape(media["expanded_url"])
                
              display_url  = url
              if media.has_key("display_url"):
                display_url = cgi.escape (media["display_url"])
 
              startindex = m["content"].index(url)
              endindex   = startindex + len(url)
              start      = m["content"][0:startindex]
              end        = m["content"][endindex:]
              m["content"] = start + "<a href='" + url + "' title='" + expanded_url  + "'>" + display_url + "</a>" + end
              
              if media["type"] == "photo":
                m["type"] = "photo"
                m["photo"] = {}
                m["photo"]["picture"] = media_url_https
                m["photo"]["url"] = None
                m["photo"]["name"] = None

            except:
              pass

      else:
        m["content"] = util.linkify(util.unescape(m["text"]),
          ((util.PARSE_HASH, '#<a href="gwibber:/tag?acct=%s&query=\\1">\\1</a>' % m["account"]),
          (util.PARSE_NICK, '@<a href="gwibber:/user?acct=%s&name=\\1">\\1</a>' % m["account"])), escape=True)

      m["html"] = m["content"]
      
      m["to_me"] = ("@%s" % self.account["username"]) in data["text"]   # Check if it's a reply directed at the user
      m["favorited"] = data.get("favorited", False)                     # Check if the tweet has been favourited

    except: 
      logger.error("%s failure - %s", PROTOCOL_INFO["name"], data)
      return {}
 
    return m

  def _user(self, user):
    """Parses the user portion of a tweet.
    
    Arguments:
    user -- A user object from a tweet
    
    Returns:
    A user object in a format compatible with Gwibber's database.
    
    """
    return {
        "name": user.get("name", None),
        "nick": user.get("screen_name", None),
        "id": user.get("id", None),
        "location": user.get("location", None),
        "followers": user.get("followers_count", None),
        "friends": user.get("friends_count", None),
        "description": user.get("description", None),
        "following": user.get("following", None),
        "protected": user.get("protected", None),
        "statuses": user.get("statuses_count", None),
        "image": user.get("profile_image_url", None),
        "website": user.get("url", None),
        "url": "/".join((URL_PREFIX, user.get("screen_name", ""))) or None,
        "is_me": user.get("screen_name", None) == self.account["username"],
    }
    
  def _message(self, data):
    """Parses messages into gwibber compatible forms.
    
    This is the initial function for tweet parsing and parses 
    retweeted status (the shared-by portion), source (the program 
    the tweet was tweeted from) and reply details (the in-reply-to 
    portion). It sends the rest to _common() for further parsing.
    
    Arguments: 
    data -- A data object obtained from Twitter containing a complete tweet
    
    Returns: 
    m -- A data object compatible with inserting into the Gwibber database for that tweet
    
    """
    if type(data) != dict:
      logger.error("Cannot parse message data: %s", str(data))
      return {}

    n = {}
    if data.has_key("retweeted_status"):
      n["retweeted_by"] = self._user(data["user"] if "user" in data else data["sender"])
      if data.has_key("created_at"):
        n["time"] = parsetime(data["created_at"])
      data = data["retweeted_status"]
    else:
      n["retweeted_by"] = None
      if data.has_key("created_at"):
        n["time"] = parsetime(data["created_at"])

    m = self._common(data)
    for k in n:
      m[k] = n[k]

    m["source"] = data.get("source", False)
    
    if data.has_key("in_reply_to_status_id"):
      if data["in_reply_to_status_id"]:
        m["reply"] = {}
        m["reply"]["id"] = data["in_reply_to_status_id"]
        m["reply"]["nick"] = data["in_reply_to_screen_name"]
        if m["reply"]["id"] and m["reply"]["nick"]:
          m["reply"]["url"] = "/".join((URL_PREFIX, m["reply"]["nick"], "statuses", str(m["reply"]["id"])))
        else:
          m["reply"]["url"] = None

    m["sender"] = self._user(data["user"] if "user" in data else data["sender"])
    m["url"] = "/".join((m["sender"]["url"], "statuses", str(m.get("mid", None))))

    return m

  def _responses(self, data):
    """Sets the message type if the message should be in the replies stream. 
    
    It sends the rest to _message() for further parsing.
    
    Arguments: 
    data -- A data object obtained from Twitter containing a complete tweet
    
    Returns: 
    m -- A data object compatible with inserting into the Gwibber database for that tweet
    
    """
    m = self._message(data)
    m["type"] = None

    return m

  def _private(self, data):
    """Sets the message type and privacy.
    
    Sets the message type and privacy if the message should be in the private stream.
    Also parses the recipient as both sent & recieved messages can be in the private stream.
    It sends the rest to _message() for further parsing
    
    Arguments: 
    data -- A data object obtained from Twitter containing a complete tweet
    
    Returns: 
    m -- A data object compatible with inserting into the Gwibber database for that tweet
    
    """
    m = self._message(data)
    m["private"] = True
    m["type"] = None

    m["recipient"] = {}
    m["recipient"]["name"] = data["recipient"]["name"]
    m["recipient"]["nick"] = data["recipient"]["screen_name"]
    m["recipient"]["id"] = data["recipient"]["id"]
    m["recipient"]["image"] = data["recipient"]["profile_image_url"]
    m["recipient"]["location"] = data["recipient"]["location"]
    m["recipient"]["url"] = "/".join((URL_PREFIX, m["recipient"]["nick"]))
    m["recipient"]["is_me"] = m["recipient"]["nick"] == self.account["username"]
    m["to_me"] = m["recipient"]["is_me"]

    return m

  def _result(self, data):
    """Called when a search is done in Gwibber.  
    
    Parses the sender and sends the rest to _common()
    for further parsing.
    
    Arguments: 
    data -- A data object obtained from Twitter containing a complete tweet
    
    Returns: 
    m -- A data object compatible with inserting into the Gwibber database for that tweet
    
    """
    m = self._common(data)
    
    if data["to_user_id"]:
      m["reply"] = {}
      m["reply"]["id"] = data["to_user_id"]
      m["reply"]["nick"] = data["to_user"]

    m["sender"] = {}
    m["sender"]["nick"] = data["from_user"]
    m["sender"]["id"] = data["from_user_id"]
    m["sender"]["image"] = data["profile_image_url"]
    m["sender"]["url"] = "/".join((URL_PREFIX, m["sender"]["nick"]))
    m["sender"]["is_me"] = m["sender"]["nick"] == self.account["username"]
    m["url"] = "/".join((m["sender"]["url"], "statuses", str(m["mid"])))
    return m

  def _profile(self, data):
    """Called when a user is clicked on.
    
    Args: 
    data -- A data object obtained from Twitter containing a complete user
    
    Returns: 
    A data object compatible with inserting into the Gwibber database for that user.
    
    """
    if "error" in data:
      return {
          "error": data["error"]
        }
    return {
        "name": data.get("name", data["screen_name"]),
        "service": "twitter",
        "stream": "profile",
        "account": self.account["id"],
        "mid": data["id"],
        "text": data.get("description", ""),
        "nick": data["screen_name"],
        "url": data.get("url", ""),
        "protected": data.get("protected", False),
        "statuses": data.get("statuses_count", 0),
        "followers": data.get("followers_count", 0),
        "friends": data.get("friends_count", 0),
        "following": data.get("following", 0),
        "favourites": data.get("favourites_count", 0),
        "image": data["profile_image_url"],
        "utc_offset": data.get("utc_offset", 0),
        "id": data["id"],
        "lang": data.get("lang", "en"),
        "verified": data.get("verified", False),
        "geo_enabled": data.get("geo_enabled", False),
        "time_zone": data.get("time_zone", "")
    }

  def _list(self, data):    
    """Called when a list is clicked on.
    
    Args: 
    data -- A data object obtained from Twitter containing a complete list
    
    Returns: 
    A data object compatible with inserting into the Gwibber database for that list.
    
    """
    return {
        "mid": data["id"],
        "service": "twitter",
        "account": self.account["id"],
        "time": 0,
        "text": data["description"],
        "html": data["description"],
        "content": data["description"],
        "url": "/".join((URL_PREFIX, data["uri"])),
        "sender": self._user(data["user"]),
        "name": data["name"],
        "nick": data["slug"],
        "key": data["slug"],
        "full": data["full_name"],
        "uri": data["uri"],
        "mode": data["mode"],
        "members": data["member_count"],
        "followers": data["subscriber_count"],
        "kind": "list",
    }

  def _get(self, path, parse="message", post=False, single=False, **args):
    """Establishes a connection with Twitter and gets the data requested.  
    
    Requires authentication.
    
    Arguments: 
    path -- The end of the url to look up on Twitter
    parse -- The function to use to parse the data returned (message by default)
    post -- True if using POST, for example the send operation. False if using GET, most operations other than send. (False by default)
    single -- True if a single checkin is requested, False if multiple (False by default)
    **args -- Arguments to be added to the URL when accessed. 
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by the parse function.
    
    """
    if not self.token and not self._login():
      logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Authentication failed"), "Auth needs updating")
      logger.error("%s", logstr)
      return [{"error": {"type": "auth", "account": self.account, "message": _("Authentication failed, please re-authorize")}}]

    url = "/".join((API_PREFIX, path))

    request = oauth.OAuthRequest.from_consumer_and_token(self.consumer, self.token,
        http_method=post and "POST" or "GET", http_url=url, parameters=util.compact(args))
    request.sign_request(self.sigmethod, self.consumer, self.token)

    if post:
      headers = request.to_header()
      data = network.Download(url, util.compact(args), post, header=headers).get_json()
    else:
      data = network.Download(request.to_url(), None, post).get_json()
    resources.dump(self.account["service"], self.account["id"], data)

    if isinstance(data, dict) and data.get("errors", 0):
      if "authenticate" in data["errors"][0]["message"]:
        # Try again, if we get a new token
        if self._login():
          logger.debug("Authentication error, logging in again")
          return self._get(path, parse, post, single, args)
        else:
          logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Authentication failed"), data["errors"][0]["message"])
          logger.error("%s", logstr)
          return [{"error": {"type": "auth", "account": self.account, "message": data["errors"][0]["message"]}}]
      else:
        for error in data["errors"]:
          logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Unknown failure"), error["message"])
          return [{"error": {"type": "unknown", "account": self.account, "message": error["message"]}}]
    elif isinstance(data, dict) and data.get("error", 0):
      if "Incorrect signature" in data["error"]:
        logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Request failed"), data["error"])
        logger.error("%s", logstr)
        return [{"error": {"type": "auth", "account": self.account, "message": data["error"]}}]
    elif isinstance(data, str):
      logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Request failed"), data)
      logger.error("%s", logstr)
      return [{"error": {"type": "request", "account": self.account, "message": data}}]

    if parse == "follow" or parse == "unfollow":
      if isinstance(data, dict) and data.get("error", 0):
        logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("%s failed" % parse), data["error"])
        logger.error("%s", logstr)
        return [{"error": {"type": "auth", "account": self.account, "message": data["error"]}}]
      else:
        return [["friendships", {"type": parse, "account": self.account["id"], "service": self.account["service"],"user_id": data["id"], "nick": data["screen_name"]}]]
    
    if parse == "profile" and isinstance(data, dict):
      return self._profile(data)

    if parse == "list":
      return [self._list(l) for l in data["lists"]]

    if single: return [getattr(self, "_%s" % parse)(data)]
    if parse: return [getattr(self, "_%s" % parse)(m) for m in data]
    else: return []

  def _search(self, **args):
    """Establishes a connection with Twitter and gets the results of a search.
    
    Does not require authentication
    
    Arguments: 
    **args -- The search terms
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _result().
    
    """
    data = network.Download("http://search.twitter.com/search.json", util.compact(args))
    data = data.get_json()["results"]

    if type(data) != list:
      logger.error("Cannot parse search data: %s", str(data))
      return []

    return [self._result(m) for m in data]

  def __call__(self, opname, **args):
    return getattr(self, opname)(**args)
  
  def receive(self, count=util.COUNT, since=None):
    """Gets the latest tweets and adds them to the database.  
    
    Arguments: 
    count -- Number of updates to get
    since -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _message().
    
    """
    return self._get("statuses/home_timeline.json", include_entities=1, count=count, since_id=since)

  def responses(self, count=util.COUNT, since=None):
    """Gets the latest replies and adds them to the database.
    
    Arguments: 
    count  -- Number of updates to get
    since  -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _responses().
    
    """
    return self._get("statuses/mentions.json", "responses", include_entities=1, count=count, since_id=since)

  def private(self, count=util.COUNT, since=None):
    """Gets the latest direct messages sent and recieved and adds them to the database.
    
    Args: 
    count -- Number of updates to get
    since -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _private().
    
    """
    private = self._get("direct_messages.json", "private", include_entities=1, count=count, since_id=since) or []
    private_sent = self._get("direct_messages/sent.json", "private", count=count, since_id=since) or []
    return private + private_sent

  def public(self):
    """Gets the latest tweets from the public timeline and adds them to the database.
    
    Arguments: 
    None
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _message().
    
    """
    return self._get("statuses/public_timeline.json", include_entities=1)

  def lists(self, **args):
    """Gets subscribed lists and adds them to the database.
    
    Arguments: 
    None
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _list().
    
    """
    if not "username" in self.account:
      self._login()
    following = self._get("%s/lists/subscriptions.json" % self.account["username"], "list") or []
    lists = self._get("%s/lists.json" % self.account["username"], "list") or []
    return following + lists

  def list(self, user, id, count=util.COUNT, since=None):    
    """Gets the latest tweets from subscribed lists and adds them to the database.
    
    Arguments: 
    user -- The user's name whose lists are to be got
    id -- The user's id whose lists are to be got
    count -- Number of updates to get
    since -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _message().
    
    """
    return self._get("%s/lists/%s/statuses.json" % (user, id), include_entities=1, per_page=count, since_id=since)

  def search(self, query, count=util.COUNT, since=None):
    """Gets the latest results from a search and adds them to the database.
    
    Arguments: 
    query -- The search query
    count -- Number of updates to get
    since -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _search().
    
    """
    return self._search(include_entities=1, q=query, rpp=count, since_id=since)

  def tag(self, query, count=util.COUNT, since=None):
    """Gets the latest results from a hashtag search and adds them to the database.
    
    Arguments: 
    query -- The search query (hashtag without the #)
    count -- Number of updates to get
    since -- Time to get updates since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _search().
    
    """
    return self._search(q="#%s" % query, count=count, since_id=since)

  def delete(self, message):
    """Deletes a specified tweet from Twitter.
    
    Arguments: 
    message -- A Gwibber compatible message object (from gwibber's database)
    
    Returns:
    Nothing
    
    """
    return self._get("statuses/destroy/%s.json" % message["mid"], None, post=True, do=1)

  def like(self, message):
    """Favourites a specified tweet on Twitter.
    
    Arguments: 
    message -- A Gwibber compatible message object (from Gwibber's database)
    
    Returns:
    Nothing
    
    """
    return self._get("favorites/create/%s.json" % message["mid"], None, post=True, do=1)

  def send(self, message):
    """Sends a tweet to Twitter.
    
    Arguments: 
    message -- The tweet's text
    
    Returns:
    Nothing
    
    """
    return self._get("statuses/update.json", post=True, single=True,
        status=message)
  
  def send_private(self, message, private):
    """Sends a direct message to Twitter.
    
    Arguments: 
    message -- The tweet's text
    private -- A gwibber compatible user object (from gwibber's database)
    
    Returns:
    Nothing
    
    """
    return self._get("direct_messages/new.json", "private", post=True, single=True,
        text=message, screen_name=private["sender"]["nick"])

  def send_thread(self, message, target):
    """Sends a reply to a user on Twitter.
    
    Arguments: 
    message -- The tweet's text
    target -- A Gwibber compatible user object (from Gwibber's database)
    
    Returns:
    Nothing
    
    """
    return self._get("statuses/update.json", post=True, single=True,
        status=message, in_reply_to_status_id=target["mid"])

  def retweet(self, message):
    """Retweets a tweet.
    
    Arguments: 
    message -- A Gwibber compatible message object (from gwibber's database)
    
    Returns:
    Nothing
    
    """
    return self._get("statuses/retweet/%s.json" % message["mid"], None, post=True, do=1)

  def follow(self, screen_name):
    """Follows a user.
    
    Arguments: 
    screen_name -- The screen name (@someone without the @) of the user to be followed
    
    Returns:
    Nothing
    
    """
    return self._get("friendships/create.json", screen_name=screen_name, post=True, parse="follow")

  def unfollow(self, screen_name):
    """Unfollows a user.
    
    Arguments: 
    screen_name -- The screen name (@someone without the @) of the user to be unfollowed
    
    Returns:
    Nothing
    
    """
    return self._get("friendships/destroy.json", screen_name=screen_name, post=True, parse="unfollow")

  def profile(self, id=None, count=None, since=None):
    """Gets a user's profile.
    
    Arguments: 
    id -- The user's screen name
    count -- Number of tweets to get
    since -- Time to get tweets since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _profile().
    
    """
    return self._get("users/show.json", screen_name=id, count=count, since_id=since, parse="profile")

  def user_messages(self, id=None, count=util.COUNT, since=None):
    """Gets a user's profile & timeline.
    
    Arguments: 
    id -- The user's screen name
    count -- Number of tweets to get
    since -- Time to get tweets since
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by _profile().
    
    """
    profiles = [self.profile(id)] or []
    messages = self._get("statuses/user_timeline.json", id=id, include_entities=1, count=count, since_id=since) or []
    return messages + profiles
