/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License version 3, as published 
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along 
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      David King <david.king@canonical.com>
 */

/**
 * Button widget, additionally storing an AccountApplicationRow.
 */
private class Cc.Credentials.AccountApplicationButton : Gtk.Button
{
    /**
     * An AccountApplicationRow, for showing the configuration widget from
     * on_options_button_clicked().
     * 
     * This property is necessary because Vala generates incorrect C code if
     * using g_object_set_data() and g_object_get_data() with a boxed type.
     */
    public AccountApplicationRow application_row { get; construct; }

    public AccountApplicationButton (string label, AccountApplicationRow row)
    {
        Object (label: label, application_row: row);
    }
}

/**
 * Web credentials account details widget. Used inside a notebook page to list
 * the applications that use an account, provide a switch to enable or disable
 * the account and to provide a button for removing the account.
 */
public class Cc.Credentials.AccountDetailsPage : Gtk.Grid
{
    private AccountsModel accounts_store;
    private Gtk.InfoBar infobar;
    private Gtk.Label infobar_label;
    private Gtk.Notebook action_notebook;
    private Gtk.Switch enabled_switch;
    private Gtk.Button grant_button;
    private Gtk.Label applications_grid_description;
    private Gtk.ScrolledWindow applications_scroll;
    private Gtk.Grid applications_grid;
    private AccountApplicationsModel applications_model;
    private Ag.Account current_account;
    private bool needs_attention = false;

    /**
     * Pages for the action widget notebook.
     *
     * @param ENABLED_SWITCH the page containing the switch to enable or
     * disable the current account
     * @param ACCESS_BUTTON the page containing the button to trigger
     * reauthentication
     */
    private enum ActionPage
    {
        ENABLED_SWITCH = 0,
        ACCESS_BUTTON = 1
    }

    /**
     * Signal the preferences widget to switch to the authentication page.
     */
    public signal void reauthenticate_account_request (Ag.Account account);

    /**
     * Signal the preferences widget to switch to the options page.
     */
    public signal void account_options_request (AccountApplicationRow application_row);

    /**
     * Index of the selected account in the model.
     */
    public Gtk.TreeIter account_iter
    {
        set
        {
            Ag.AccountId account_id;
            accounts_store.get (value,
                                AccountsModel.ModelColumns.ACCOUNT_ID,
                                out account_id,
                                AccountsModel.ModelColumns.NEEDS_ATTENTION,
                                out needs_attention,
                                -1);
            account = accounts_store.manager.get_account (account_id);
        }
    }

    /**
     * Keep the UI state consistent when the account is changed.
     */
    public Ag.Account account
    {
        get
        {
            return current_account;
        }
        set
        {
            current_account = value;

            /* TODO: if needs_attention is true, we might want to hide the list
             * of the integrated applications.
             */

            enabled_switch.active = value.get_enabled ();
            value.enabled.connect (on_account_enabled);

            var manager = accounts_store.manager;
            var provider = manager.get_provider (value.get_provider_name ());
            var provider_display_name = provider.get_display_name ();

            if (needs_attention)
            {
                infobar_label.label = _("Please authorize Ubuntu to access your %s account:").printf
                                      (provider_display_name);
                infobar.message_type = Gtk.MessageType.WARNING;

                action_notebook.page = ActionPage.ACCESS_BUTTON;
            }
            else
            {
                infobar_label.label = "<b>" + provider_display_name + "</b>\n"
                                      + "<small><span foreground=\"#555555\">"
                                      + value.get_display_name ()
                                      + "</span></small>";
                infobar.message_type = Gtk.MessageType.INFO;

                action_notebook.page = ActionPage.ENABLED_SWITCH;
            }

            // Update the applications model.
            applications_model.account = value;

            // Special-case having no consumer applications installed.
            unowned List<AccountApplicationRow?> applications = applications_model.application_rows;

            if (applications == null)
            {
                applications_grid_description.label = _("There are currently no applications installed which integrate with your %s account.").printf
                                                      (provider_display_name);
            }
            else
            {
                applications_grid_description.label = _("The following applications integrate with your %s account:").printf
                                                      (provider_display_name);
            }

            populate_applications_grid ();
        }
    }

    public AccountDetailsPage (AccountsModel accounts_store)
    {
        Object ();
        this.accounts_store = accounts_store;
    }

    construct
    {
        orientation = Gtk.Orientation.VERTICAL;

        expand = true;

        this.add (create_infobar ());
        this.add (create_applications_frame ());

        show ();
    }

    /**
     * Create the frame to contain the other widgets for the account and show
     * it.
     *
     * @return a Gtk.Frame for presenting details of the account
     */
    private Gtk.Widget create_applications_frame ()
    {
        var eventbox = new Gtk.EventBox ();
        var frame = new Gtk.Frame (null);
        frame.shadow_type = Gtk.ShadowType.ETCHED_IN;

        var grid = new Gtk.Grid ();
        grid.orientation = Gtk.Orientation.VERTICAL;

        grid.add (create_applications_grid_description ());
        grid.add (create_applications_grid ());
        grid.add (create_remove_account_box ());

        frame.add (grid);
        eventbox.add (frame);

        // Override theme colors according to the UI specification.
        Gdk.RGBA color;
        var context = eventbox.get_style_context ();
        if (context.lookup_color ("base_color", out color))
        {
            eventbox.override_background_color (Gtk.StateFlags.NORMAL, color);
        }
        else
        {
            warning ("Error looking up theme color");
        }

        return eventbox;
    }

    /**
     * Create the infobar for the account and show it.
     *
     * @return a Gtk.InfoBar for presenting details of the account
     */
    private Gtk.Widget create_infobar ()
    {
        infobar = new Gtk.InfoBar ();
        action_notebook = new Gtk.Notebook ();
        enabled_switch = new Gtk.Switch ();
        infobar_label = new Gtk.Label (null);
        var content_area = infobar.get_content_area () as Gtk.Container;
        var action_area = infobar.get_action_area () as Gtk.Container;

        action_notebook.show_tabs = false;
        action_notebook.show_border = false;
        action_area.halign = Gtk.Align.END;
        action_area.valign = Gtk.Align.CENTER;
        infobar_label.use_markup = true;
        infobar_label.wrap = true;
        infobar_label.xalign = 0.0f;
        content_area.add (infobar_label);

        infobar.message_type = Gtk.MessageType.INFO;
        infobar.name = "account-details-infobar";
        action_notebook.append_page (enabled_switch, null);
        grant_button = new Gtk.Button.with_label (_("Grant access"));
        action_notebook.append_page (grant_button, null);

        action_area.add (action_notebook);

        enabled_switch.notify["active"].connect (on_enabled_switch_activated);
        grant_button.clicked.connect (on_grant_button_clicked);

        // Override theme colors according to the UI specification.
        try
        {
          var css = new Gtk.CssProvider ();
          css.load_from_data ("@define-color warning_bg_color rgb (222, 222, 222); @define-color info_bg_color @warning_bg_color; GtkInfoBar#account-details-infobar { color: @fg_color; -GtkInfoBar-action-area-border: 10 }", -1);
          var context = infobar.get_style_context ();
          context.add_provider (css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
        }
        catch (Error err)
        {
          warning ("Error processing CSS theme override: %s", err.message);
        }

        infobar.set_size_request (-1, 48);
        infobar.show_all ();

        return infobar;
    }

    /**
     * Create a description to place above the applications grid.
     *
     * @return a Gtk.Label for a description of the applications grid
     */
    private Gtk.Widget create_applications_grid_description ()
    {
        applications_grid_description = new Gtk.Label (null);

        applications_grid_description.margin = 6;
        applications_grid_description.xalign = 0.0f;

        applications_grid_description.set_line_wrap (true);
        applications_grid_description.set_size_request (414, -1);
        applications_grid_description.show ();

        return applications_grid_description;
    }

    /**
     * Create the grid with a list of applications using the current account.
     *
     * @return a Gtk.Grid containing a list of applications
     */
    private Gtk.Widget create_applications_grid ()
    {
        applications_model = new AccountApplicationsModel ();

        applications_scroll = new Gtk.ScrolledWindow (null, null);

        // Instantiates applications_grid.
        populate_applications_grid ();

        applications_scroll.window_placement_set = true;

        // Override theme colors according to the UI specification.
        Gdk.RGBA color;
        var viewport = applications_scroll.get_child ();
        var context = viewport.get_style_context ();
        if (context.lookup_color ("base_color", out color))
        {
            viewport.override_background_color (Gtk.StateFlags.NORMAL, color);
        }
        else
        {
            warning ("Error looking up theme color");
        }

        return applications_scroll;
    }

    /**
     * Create a button box for the account editing buttons.
     *
     * @return a Gtk.ButtonBox for the remove or edit account buttons
     */
    private Gtk.Widget create_remove_account_box ()
    {
        var buttonbox = new Gtk.ButtonBox (Gtk.Orientation.HORIZONTAL);
        buttonbox.set_layout (Gtk.ButtonBoxStyle.END);
        buttonbox.margin = 6;

        var remove_button = new Gtk.Button.with_label (_("Remove account"));

        remove_button.clicked.connect (on_remove_account_clicked);

        buttonbox.add (remove_button);
        buttonbox.show_all ();

        return buttonbox;
    }

    /**
     * Populate the grid of applications from the model. Instantiates
     * applications_grid.
     */
    private void populate_applications_grid ()
    {
        if (applications_grid != null)
        {
            applications_grid.destroy ();
        }

        applications_grid = new Gtk.Grid ();
        applications_grid.border_width = 6;
        applications_grid.column_spacing = 12;
        applications_grid.row_spacing = 12;
        applications_grid.expand = true;

        unowned List<AccountApplicationRow?> applications = applications_model.application_rows;

        applications.foreach (add_application);

        applications_scroll.add_with_viewport (applications_grid);

        applications_scroll.show_all ();
    }

    /**
     * Add an individual application from the model to the applications grid.
     *
     * @param application the description of the application
     */
    private void add_application (AccountApplicationRow? application)
    {
        applications_grid.insert_row (0);

        var image = new Gtk.Image.from_gicon (application.icon,
                                              Gtk.IconSize.DND);
        image.margin_left = 4;
        applications_grid.attach (image, 0, 0, 1, 1);

        var label = new Gtk.Label (application.description);
        label.hexpand = true;
        label.use_markup = true;
        label.xalign = 0.0f;
        applications_grid.attach_next_to (label,
                                          image,
                                          Gtk.PositionType.RIGHT,
                                          1,
                                          1);

        if (application.plugin_widget != null)
        {
            var button = new AccountApplicationButton (_("Options"),
                                                       application);
            applications_grid.attach_next_to (button,
                                              label,
                                              Gtk.PositionType.RIGHT,
                                              1,
                                              1);
            button.clicked.connect (on_options_button_clicked);
        }
    }

    /**
     * Handle the remove account button being clicked. The removal is
     * asynchronous, and on_remove_account_finished() is called when the
     * operation is complete.
     *
     * @see AccountDetailsPage.on_remove_account_finished
     */
    private void on_remove_account_clicked ()
    {
        // TODO: Pass parent window ID as first argument.
        var confirmation = new Gtk.MessageDialog (null,
                                                  Gtk.DialogFlags.DESTROY_WITH_PARENT,
                                                  Gtk.MessageType.QUESTION,
                                                  Gtk.ButtonsType.NONE,
                                                  "%s",
                                                  _("Are you sure that you wish to remove this Ubuntu Web Account?"));
        var manager = accounts_store.manager;
        var provider = manager.get_provider (current_account.get_provider_name ());
        var provider_display_name = provider.get_display_name ();
        var secondary_text = _("The Web Account which manages the integration of %s with your applications will be removed.").printf (provider_display_name)
                               + "\n\n"
                               + _("Your online %s account is not affected.").printf (provider_display_name);
        confirmation.secondary_text = secondary_text;
        confirmation.add_buttons (Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                  Gtk.Stock.REMOVE, Gtk.ResponseType.ACCEPT,
                                  null);

        var response = confirmation.run ();

        switch (response)
        {
            case Gtk.ResponseType.ACCEPT:
                // TODO: Set the UI to be insensitive during account removal?

                var plugin = Ap.client_load_plugin (current_account);
                if (plugin == null)
                {
                    /* This can really happen, if the plugin has been
                     * uninstalled; in this case, the user can still access the
                     * account (to disable or delete it).
                     */
                    warning ("No valid plugin found for provider %s",
                             current_account.get_provider_name ());
                    // TODO: Delete the account in this case.
                    break;
                }

                plugin.delete_account.begin ((obj, res) => {
                    try
                    {
                        plugin.delete_account.end (res);
                    }
                    catch (Error error)
                    {
                        critical ("Error deleting account: %s", error.message);
                    }
                    on_remove_account_finished (current_account);
                });
                break;
            case Gtk.ResponseType.CANCEL:
            case Gtk.ResponseType.DELETE_EVENT:
                break;
            default:
                assert_not_reached ();
        }

        confirmation.destroy ();
    }

    /**
     * Handle the completion of the asynchronous account removal operation.
     *
     * @param account the account that was removed
     */
    private void on_remove_account_finished (Ag.Account account)
    {
        /* TODO: Set the UI to be sensitive again? Switch to the add account
         * view.
         */
    }

    /**
     * Handle the account enabled switch being activated. Change the current
     * account to be the same state as the switch.
     */
    private void on_enabled_switch_activated ()
    {
        current_account.set_enabled (enabled_switch.active);

        // FIXME: Use asynchronous account.store ().
        try
        {
            current_account.store_blocking ();
        }
        catch (Error error)
        {
            critical ("Error changing enabled state of account: %s\nMessage: %s",
                      current_account.get_display_name (),
                      error.message);
        }
    }

    /**
     * Handle the reauthentication button being clicked.
     */
    private void on_grant_button_clicked ()
    {
        reauthenticate_account_request (current_account);
    }

    /**
     * Handle the account being enabled or disabled from elsewhere, and update
     * the switch state accordingly.
     */
    private void on_account_enabled ()
    {
        var enabled = current_account.get_enabled ();

        if (enabled != enabled_switch.active)
        {
            enabled_switch.active = enabled;
        }
    }

    /**
     * Handle the options button for an application being clicked.
     *
     * @param button the AccountApplicationButton that emitted the clicked
     * signal. The account application plugin is a property on the button.
     */
    private void on_options_button_clicked (Gtk.Button button)
    {
        var app_button = button as AccountApplicationButton;
        var application_row = app_button.application_row;

        if (application_row.plugin_widget != null)
        {
            account_options_request (application_row);
        }
    }
}
