------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-lib.ads,v 1.14 2004/06/28 04:57:05 vagul Exp $

with
   System,
   Interfaces.C.Pointers,
   Interfaces.C.Strings;

package OCI.Lib is

   package C renames Interfaces.C;
   package CStr renames C.Strings;

   subtype Text_Ptr is CStr.chars_ptr;
   subtype Text is C.char_array;
   subtype OraText is Text_Ptr;
   subtype Ub1 is C.unsigned_char;
   subtype Sb1 is C.signed_char;
   subtype Ub2 is C.unsigned_short;
   subtype Sb2 is C.short;
   subtype Ub4 is C.unsigned_long;
   subtype Sb4 is C.long;
   subtype Eb1 is C.char;
   subtype Eb2 is C.short;
   subtype Eb4 is C.long;
   subtype UWord is C.unsigned;

   subtype DVoid is System.Address;

   type A_Int is access all C.int;
   type A_Sb4 is access all Sb4;
   type A_Sb2 is access all Sb2;
   type A_Ub4 is access all Ub4;
   type A_Ub2 is access all Ub2;
   type A_Ub1 is access all Ub1;

   subtype Function_Pointer is DVoid;

   subtype OCIHandle is System.Address;
   Empty_Handle : constant OCIHandle := System.Null_Address;

   type OCIEnv is new OCIHandle;
   type OCIError is new OCIHandle;
   type OCISvcCtx is new OCIHandle;
   type OCIStmt is new OCIHandle;
   type OCIBind is new OCIHandle;
   type OCIDefine is new OCIHandle;
   type OCIDescribe is new OCIHandle;
   type OCIServer is new OCIHandle;
   type OCISession is new OCIHandle;
   type OCIComplexObject is new OCIHandle;
   type OCITrans is new OCIHandle;
   type OCISecurity is new OCIHandle;
   type OCISubscription is new OCIHandle;

   type OCISnapshot is new OCIHandle;
   type OCIResult is new OCIHandle;
   type OCILobLocator is new OCIHandle;
   type OCIParam is new OCIHandle;
   type OCIComplexObjectComp is new OCIHandle;
   type OCIRowid is new OCIHandle;
   type OCIDateTime is new OCIHandle;
   type OCIInterval is new OCIHandle;

   type OCIExtProcContext is new OCIHandle;

   type OCIAQEnqOptions is new OCIHandle;
   type OCIAQDeqOptions is new OCIHandle;
   type OCIAQMsgProperties is new OCIHandle;
   type OCIAQAgent is new OCIHandle;
   type OCIAQNfyDescriptor is new OCIHandle;

   subtype SWord is C.int;

   type Vector_Of_Ub1 is array (Integer range <>) of aliased Ub1;
   type Vector_Of_Ub4 is array (Integer range <>) of aliased Ub4;
   type Vector_Of_OraText is array (Integer range <>) of aliased OraText;
   type Vector_Of_OCIBind is array (Integer range <>) of aliased OCIBind;

   package Ub1_Ptr is new Interfaces.C.Pointers
      (Integer, Ub1, Vector_Of_Ub1, 0);

   subtype A_Vector_Of_Ub1 is Ub1_Ptr.Pointer;

   package OraText_Ptr is new Interfaces.C.Pointers
      (Integer, OraText, Vector_Of_OraText, CStr.Null_Ptr);

   subtype A_Vector_Of_OraText is OraText_Ptr.Pointer;

   package OCIBind_Ptr is new Interfaces.C.Pointers
      (Integer, OCIBind, Vector_Of_OCIBind, OCIBind (Empty_Handle));

   subtype A_Vector_Of_OCIBind is OCIBind_Ptr.Pointer;

   subtype Undefined is Integer;
   Empty : constant := 0;

   OCI_SUCCESS           : constant := 0;
   OCI_SUCCESS_WITH_INFO : constant := 1;
   OCI_NO_DATA           : constant := 100;
   OCI_ERROR             : constant := -1;
   OCI_INVALID_HANDLE    : constant := -2;
   OCI_NEED_DATA         : constant := 99;
   OCI_STILL_EXECUTING   : constant := -3123;
   OCI_CONTINUE          : constant := -24200;

   OCI_FETCH_NEXT       : constant := 2;
   OCI_FETCH_FIRST      : constant := 16#04#;
   OCI_FETCH_LAST       : constant := 16#08#;
   OCI_FETCH_PRIOR      : constant := 16#10#;
   OCI_FETCH_ABSOLUTE   : constant := 16#20#;
   OCI_FETCH_RELATIVE   : constant := 16#40#;
   OCI_FETCH_RESERVED_1 : constant := 16#80#;

   OCI_V7_SYNTAX  : constant := 2;
   OCI_V8_SYNTAX  : constant := 3;
   OCI_NTV_SYNTAX : constant := 1;

   OCI_OTYPE_NAME : constant := 1;
   OCI_OTYPE_REF  : constant := 2;
   OCI_OTYPE_PTR  : constant := 3;

   OCI_PTYPE_UNK         : constant := 0;
   OCI_PTYPE_TABLE       : constant := 1;
   OCI_PTYPE_VIEW        : constant := 2;
   OCI_PTYPE_PROC        : constant := 3;
   OCI_PTYPE_FUNC        : constant := 4;
   OCI_PTYPE_PKG         : constant := 5;
   OCI_PTYPE_TYPE        : constant := 6;
   OCI_PTYPE_SYN         : constant := 7;
   OCI_PTYPE_SEQ         : constant := 8;
   OCI_PTYPE_COL         : constant := 9;
   OCI_PTYPE_ARG         : constant := 10;
   OCI_PTYPE_LIST        : constant := 11;
   OCI_PTYPE_TYPE_ATTR   : constant := 12;
   OCI_PTYPE_TYPE_COLL   : constant := 13;
   OCI_PTYPE_TYPE_METHOD : constant := 14;
   OCI_PTYPE_TYPE_ARG    : constant := 15;
   OCI_PTYPE_TYPE_RESULT : constant := 16;
   OCI_PTYPE_SCHEMA      : constant := 17;
   OCI_PTYPE_DATABASE    : constant := 18;

   OCI_LTYPE_UNK           : constant := 0;
   OCI_LTYPE_COLUMN        : constant := 1;
   OCI_LTYPE_ARG_PROC      : constant := 2;
   OCI_LTYPE_ARG_FUNC      : constant := 3;
   OCI_LTYPE_SUBPRG        : constant := 4;
   OCI_LTYPE_TYPE_ATTR     : constant := 5;
   OCI_LTYPE_TYPE_METHOD   : constant := 6;
   OCI_LTYPE_TYPE_ARG_PROC : constant := 7;
   OCI_LTYPE_TYPE_ARG_FUNC : constant := 8;
   OCI_LTYPE_SCH_OBJ       : constant := 9;
   OCI_LTYPE_DB_SCH        : constant := 10;

   OCI_MEMORY_CLEARED : constant := 1;

   OCI_ATTR_FNCODE                  : constant := 1;
   OCI_ATTR_OBJECT                  : constant := 2;
   OCI_ATTR_NONBLOCKING_MODE        : constant := 3;
   OCI_ATTR_SQLCODE                 : constant := 4;
   OCI_ATTR_ENV                     : constant := 5;
   OCI_ATTR_SERVER                  : constant := 6;
   OCI_ATTR_SESSION                 : constant := 7;
   OCI_ATTR_TRANS                   : constant := 8;
   OCI_ATTR_ROW_COUNT               : constant := 9;
   OCI_ATTR_SQLFNCODE               : constant := 10;
   OCI_ATTR_PREFETCH_ROWS           : constant := 11;
   OCI_ATTR_NESTED_PREFETCH_ROWS    : constant := 12;
   OCI_ATTR_PREFETCH_MEMORY         : constant := 13;
   OCI_ATTR_NESTED_PREFETCH_MEMORY  : constant := 14;
   OCI_ATTR_CHAR_COUNT              : constant := 15;
   OCI_ATTR_PDSCL                   : constant := 16;
   OCI_ATTR_PDFMT                   : constant := 17;
   OCI_ATTR_PARAM_COUNT             : constant := 18;
   OCI_ATTR_ROWID                   : constant := 19;
   OCI_ATTR_CHARSET                 : constant := 20;
   OCI_ATTR_NCHAR                   : constant := 21;
   OCI_ATTR_USERNAME                : constant := 22;
   OCI_ATTR_PASSWORD                : constant := 23;
   OCI_ATTR_STMT_TYPE               : constant := 24;
   OCI_ATTR_INTERNAL_NAME           : constant := 25;
   OCI_ATTR_EXTERNAL_NAME           : constant := 26;
   OCI_ATTR_XID                     : constant := 27;
   OCI_ATTR_TRANS_LOCK              : constant := 28;
   OCI_ATTR_TRANS_NAME              : constant := 29;
   OCI_ATTR_HEAPALLOC               : constant := 30;
   OCI_ATTR_CHARSET_ID              : constant := 31;
   OCI_ATTR_CHARSET_FORM            : constant := 32;
   OCI_ATTR_MAXDATA_SIZE            : constant := 33;
   OCI_ATTR_CACHE_OPT_SIZE          : constant := 34;
   OCI_ATTR_CACHE_MAX_SIZE          : constant := 35;
   OCI_ATTR_PINOPTION               : constant := 36;
   OCI_ATTR_ALLOC_DURATION          : constant := 37;
   OCI_ATTR_PIN_DURATION            : constant := 38;
   OCI_ATTR_FDO                     : constant := 39;
   OCI_ATTR_POSTPROCESSING_CALLBACK : constant := 40;
   OCI_ATTR_POSTPROCESSING_CONTEXT  : constant := 41;
   OCI_ATTR_ROWS_RETURNED           : constant := 42;
   OCI_ATTR_FOCBK                   : constant := 43;
   OCI_ATTR_IN_V8_MODE              : constant := 44;
   OCI_ATTR_LOBEMPTY                : constant := 45;
   OCI_ATTR_SESSLANG                : constant := 46;

   OCI_ATTR_NUM_DML_ERRORS : constant := 73;
   OCI_ATTR_DML_ROW_OFFSET : constant := 74;

   OCI_ATTR_PARSE_ERROR_OFFSET  : constant := 129;

   OCI_ATTR_DATA_SIZE : constant := 1;
   OCI_ATTR_DATA_TYPE : constant := 2;
   OCI_ATTR_DISP_SIZE : constant := 3;
   OCI_ATTR_NAME      : constant := 4;
   OCI_ATTR_PRECISION : constant := 5;
   OCI_ATTR_SCALE     : constant := 6;
   OCI_ATTR_IS_NULL   : constant := 7;
   OCI_ATTR_TYPE_NAME : constant := 8;

   OCI_ATTR_SCHEMA_NAME : constant := 9;
   OCI_ATTR_SUB_NAME    : constant := 10;
   OCI_ATTR_POSITION    : constant := 11;

   --  complex object retrieval parameter attributes
   OCI_ATTR_COMPLEXOBJECTCOMP_TYPE       : constant := 50;
   OCI_ATTR_COMPLEXOBJECTCOMP_TYPE_LEVEL : constant := 51;
   OCI_ATTR_COMPLEXOBJECT_LEVEL          : constant := 52;
   OCI_ATTR_COMPLEXOBJECT_COLL_OUTOFLINE : constant := 53;

   OCI_ATTR_NOCACHE     : constant := 87; -- Temporary LOBs
   --  Strange. I think it is wrong attribute description -^.

   OCI_ATTR_SUBSCR_NAME      : constant := 94;  -- name of subscription
   OCI_ATTR_SUBSCR_CALLBACK  : constant := 95;  -- associated callback
   OCI_ATTR_SUBSCR_CTX       : constant := 96;  -- associated callback context
   OCI_ATTR_SUBSCR_PAYLOAD   : constant := 97;  -- associated payload
   OCI_ATTR_SUBSCR_NAMESPACE : constant := 98;  -- associated namespace

   OCI_ATTR_CONSUMER_NAME : constant := 50; -- consumer name
   OCI_ATTR_DEQ_MODE      : constant := 51; -- dequeue mode
   OCI_ATTR_NAVIGATION    : constant := 52; -- navigation
   OCI_ATTR_WAIT                : constant := 53; -- wait
   OCI_ATTR_DEQ_MSGID     : constant := 54; -- dequeue message id

   OCI_ATTR_QUEUE_NAME : constant := 70;  --  queue name
   OCI_ATTR_NFY_MSGID  : constant := 71;  --  message id
   OCI_ATTR_MSG_PROP   : constant := 72;        --  message properties

   --  only named type attributes
   OCI_ATTR_TYPECODE            : constant := 216; -- object or collection
   OCI_ATTR_COLLECTION_TYPECODE : constant := 217; -- varray or nested table
   OCI_ATTR_VERSION             : constant := 218; -- user assigned version
   OCI_ATTR_IS_INCOMPLETE_TYPE  : constant := 219;
   OCI_ATTR_IS_SYSTEM_TYPE      : constant := 220; -- a system type
   OCI_ATTR_IS_PREDEFINED_TYPE  : constant := 221; -- a predefined type
   OCI_ATTR_IS_TRANSIENT_TYPE   : constant := 222; -- a transient type
   OCI_ATTR_IS_SYSTEM_GENERATED_TYPE : constant := 223;
   OCI_ATTR_HAS_NESTED_TABLE    : constant := 224;
   OCI_ATTR_HAS_LOB             : constant := 225; -- has a lob attribute
   OCI_ATTR_HAS_FILE            : constant := 226; -- has a file attribute
   OCI_ATTR_COLLECTION_ELEMENT  : constant := 227;
   OCI_ATTR_NUM_TYPE_ATTRS      : constant := 228; -- number of attribute types
   OCI_ATTR_LIST_TYPE_ATTRS     : constant := 229; -- list of type attributes
   OCI_ATTR_NUM_TYPE_METHODS    : constant := 230; -- number of type methods
   OCI_ATTR_LIST_TYPE_METHODS   : constant := 231; -- list of type methods
   OCI_ATTR_MAP_METHOD          : constant := 232; -- map method of type
   OCI_ATTR_ORDER_METHOD        : constant := 233; -- order method of type

   OCI_DTYPE_FIRST             : constant := 50;
   OCI_DTYPE_LOB               : constant := 50;
   OCI_DTYPE_SNAP              : constant := 51;
   OCI_DTYPE_RSET              : constant := 52;
   OCI_DTYPE_PARAM             : constant := 53;
   OCI_DTYPE_ROWID             : constant := 54;
   OCI_DTYPE_COMPLEXOBJECTCOMP : constant := 55;
   OCI_DTYPE_FILE              : constant := 56;
   OCI_DTYPE_AQENQ_OPTIONS     : constant := 57;
   OCI_DTYPE_AQDEQ_OPTIONS     : constant := 58;
   OCI_DTYPE_AQMSG_PROPERTIES  : constant := 59;
   OCI_DTYPE_AQAGENT           : constant := 60;
   OCI_DTYPE_LOCATOR           : constant := 61;
   OCI_DTYPE_DATETIME          : constant := 62;
   OCI_DTYPE_INTERVAL          : constant := 63;
   OCI_DTYPE_AQNFY_DESCRIPTOR  : constant := 64;
   OCI_DTYPE_LAST              : constant := 64;

   OCI_BATCH_MODE             : constant := 16#01#;
   OCI_EXACT_FETCH            : constant := 16#02#;
   OCI_KEEP_FETCH_STATE       : constant := 16#04#;
   OCI_SCROLLABLE_CURSOR      : constant := 16#08#;
   OCI_DESCRIBE_ONLY          : constant := 16#10#;
   OCI_COMMIT_ON_SUCCESS      : constant := 16#20#;
   OCI_NON_BLOCKING           : constant := 16#40#;
   OCI_BATCH_ERRORS           : constant := 16#80#;
   OCI_PARSE_ONLY             : constant := 16#100#;
   OCI_EXACT_FETCH_RESERVED_1 : constant := 16#200#;

   ------------------------- Supported Namespaces ---------------------------
   OCI_SUBSCR_NAMESPACE_ANONYMOUS : constant := 0; -- Anonymous Namespace
   OCI_SUBSCR_NAMESPACE_AQ        : constant := 1; -- Advanced Queues
   OCI_SUBSCR_NAMESPACE_MAX       : constant := 2; -- Max Name Space Number


   SQLT_CHR : constant := 1;     -- (ORANET TYPE) character string
   SQLT_NUM : constant := 2;     -- (ORANET TYPE) oracle numeric
   SQLT_INT : constant := 3;     -- (ORANET TYPE) integer
   SQLT_FLT : constant := 4;     -- (ORANET TYPE) Floating point number
   SQLT_STR : constant := 5;     -- zero terminated string
   SQLT_VNU : constant := 6;     -- NUM with preceding length byte
   SQLT_PDN : constant := 7;     -- (ORANET TYPE) Packed Decimal Numeric
   SQLT_LNG : constant := 8;     -- long
   SQLT_VCS : constant := 9;     -- Variable character string
   SQLT_NON : constant := 10;    -- Null/empty PCC Descriptor entry
   SQLT_RID : constant := 11;    -- rowid
   SQLT_DAT : constant := 12;    -- date in oracle format
   SQLT_VBI : constant := 15;    -- binary in VCS format
   SQLT_BIN : constant := 23;    -- binary data(DTYBIN)
   SQLT_LBI : constant := 24;    -- long binary
   SQLT_UIN : constant := 68;    -- unsigned integer
   SQLT_SLS : constant := 91;    -- Display sign leading separate
   SQLT_LVC : constant := 94;    -- Longer longs (char)
   SQLT_LVB : constant := 95;    -- Longer long binary
   SQLT_AFC : constant := 96;    -- Ansi fixed char
   SQLT_AVC : constant := 97;    -- Ansi Var char
   SQLT_CUR : constant := 102;   -- cursor  type
   SQLT_RDD : constant := 104;   -- rowid descriptor
   SQLT_LAB : constant := 105;   -- label type
   SQLT_OSL : constant := 106;   -- oslabel type

   SQLT_NTY    : constant := 108;  -- named object type
   SQLT_REF    : constant := 110;  -- ref type
   SQLT_CLOB   : constant := 112;  -- character lob
   SQLT_BLOB   : constant := 113;  -- binary lob
   SQLT_BFILEE : constant := 114;  -- binary file lob
   SQLT_CFILEE : constant := 115;  -- character file lob
   SQLT_RSET   : constant := 116;  -- result set type
   SQLT_NCO    : constant := 122;
   --  named collection type (varray or nested table)

   SQLT_VST    : constant := 155;  -- OCIString type
   SQLT_ODT    : constant := 156;  -- OCIDate type

   SQLT_DATE          : constant := 184; -- ANSI Date
   SQLT_TIME          : constant := 185; -- TIME
   SQLT_TIME_TZ       : constant := 186; -- TIME WITH TIME ZONE
   SQLT_TIMESTAMP     : constant := 187; -- TIMESTAMP
   SQLT_TIMESTAMP_TZ  : constant := 188; -- TIMESTAMP WITH TIME ZONE
   SQLT_INTERVAL_YM   : constant := 189; -- INTERVAL YEAR TO MONTH
   SQLT_INTERVAL_DS   : constant := 190; -- INTERVAL DAY TO SECOND
   SQLT_TIMESTAMP_LTZ : constant := 232; -- TIMESTAMP WITH LOCAL TZ

   SQLT_PNTY          : constant := 241;
   --  pl/sql representation of named types

   SQLT_FILE  : constant := SQLT_BFILEE;
   SQLT_CFILE : constant := SQLT_CFILEE;
   SQLT_BFILE : constant := SQLT_BFILEE;

   OCI_TYPECODE_REF             : constant := SQLT_REF;
   OCI_TYPECODE_DATE            : constant := SQLT_DAT;
   OCI_TYPECODE_SIGNED8         : constant := 27;
   OCI_TYPECODE_SIGNED16        : constant := 28;
   OCI_TYPECODE_SIGNED32        : constant := 29;
   OCI_TYPECODE_REAL            : constant := 21;
   OCI_TYPECODE_DOUBLE          : constant := 22;
   OCI_TYPECODE_FLOAT           : constant := SQLT_FLT;
   OCI_TYPECODE_NUMBER          : constant := SQLT_NUM;
   OCI_TYPECODE_DECIMAL         : constant := SQLT_PDN;
   OCI_TYPECODE_UNSIGNED8       : constant := SQLT_BIN;
   OCI_TYPECODE_UNSIGNED16      : constant := 25;
   OCI_TYPECODE_UNSIGNED32      : constant := 26;
   OCI_TYPECODE_OCTET           : constant := 245;
   OCI_TYPECODE_SMALLINT        : constant := 246;
   OCI_TYPECODE_INTEGER         : constant := SQLT_INT;
   OCI_TYPECODE_RAW             : constant := SQLT_LVB;
   OCI_TYPECODE_PTR             : constant := 32;
   OCI_TYPECODE_VARCHAR2        : constant := SQLT_VCS;
   OCI_TYPECODE_CHAR            : constant := SQLT_AFC;
   OCI_TYPECODE_VARCHAR         : constant := SQLT_CHR;
   OCI_TYPECODE_MLSLABEL        : constant := SQLT_LAB;
   OCI_TYPECODE_VARRAY          : constant := 247;
   OCI_TYPECODE_TABLE           : constant := 248;
   OCI_TYPECODE_OBJECT          : constant := SQLT_NTY;
   OCI_TYPECODE_OPAQUE          : constant := 58;
   OCI_TYPECODE_NAMEDCOLLECTION : constant := SQLT_NCO;
   OCI_TYPECODE_BLOB            : constant := SQLT_BLOB;
   OCI_TYPECODE_BFILE           : constant := SQLT_BFILE;
   OCI_TYPECODE_CLOB            : constant := SQLT_CLOB;
   OCI_TYPECODE_CFILE           : constant := SQLT_CFILE;

   OCI_TYPECODE_OTMFIRST : constant := 228;
   OCI_TYPECODE_OTMLAST  : constant := 320;
   OCI_TYPECODE_SYSFIRST : constant := 228;
   OCI_TYPECODE_SYSLAST  : constant := 235;

   OCI_DEFAULT : constant := 0;

   OCI_THREADED : constant := 1;
   OCI_OBJECT   : constant := 2;
   OCI_EVENTS   : constant := 4;
   OCI_SHARED   : constant := 16#10#;

   OCI_NO_UCB   : constant := 16#40#;
   OCI_NO_MUTEX : constant := 16#80#;

   OCI_ENV_NO_UCB   : constant := 1;
   OCI_ENV_NO_MUTEX : constant := 8;

   OCI_HTYPE_FIRST                : constant := 1;
   OCI_HTYPE_ENV                  : constant := 1;
   OCI_HTYPE_ERROR                : constant := 2;
   OCI_HTYPE_SVCCTX               : constant := 3;
   OCI_HTYPE_STMT                 : constant := 4;
   OCI_HTYPE_BIND                 : constant := 5;
   OCI_HTYPE_DEFINE               : constant := 6;
   OCI_HTYPE_DESCRIBE             : constant := 7;
   OCI_HTYPE_SERVER               : constant := 8;
   OCI_HTYPE_SESSION              : constant := 9;
   OCI_HTYPE_TRANS                : constant := 10;
   OCI_HTYPE_COMPLEXOBJECT        : constant := 11;
   OCI_HTYPE_SECURITY             : constant := 12;
   OCI_HTYPE_SUBSCRIPTION         : constant := 13;
   OCI_HTYPE_DIRPATH_CTX          : constant := 14;
   OCI_HTYPE_DIRPATH_COLUMN_ARRAY : constant := 15;
   OCI_HTYPE_DIRPATH_STREAM       : constant := 16;
   OCI_HTYPE_PROC                 : constant := 17;
   OCI_HTYPE_LAST                 : constant := 17;

   OCI_ONE_PIECE   : constant := 0;
   OCI_FIRST_PIECE : constant := 1;
   OCI_NEXT_PIECE  : constant := 2;
   OCI_LAST_PIECE  : constant := 3;

   OCI_DATA_AT_EXEC  : constant := 2;
   OCI_DYNAMIC_FETCH : constant := 2;
   OCI_PIECEWISE     : constant := 4;

   OCI_TEMP_BLOB  : constant := 1;
   OCI_TEMP_CLOB  : constant := 2;
   OCI_TEMP_NCLOB : constant := 3;

   --  CHAR/NCHAR/VARCHAR2/NVARCHAR2/CLOB/NCLOB char set "form" information
   SQLCS_IMPLICIT : constant := 1; -- CHAR, VARCHAR2, CLOB w/o a specified set
   SQLCS_NCHAR    : constant := 2; -- NCHAR, NCHAR VARYING, NCLOB
   SQLCS_EXPLICIT : constant := 3; -- CHAR, etc, with "CHARACTER SET..." syntax
   SQLCS_FLEXIBLE : constant := 4; -- PL/SQL "flexible" parameters
   SQLCS_LIT_NULL : constant := 5; -- typecheck of NULL and empty_clob() lits

   -------------------------Credential Types-------------------------
   OCI_CRED_RDBMS   : constant := 1;  -- database username/password
   OCI_CRED_EXT     : constant := 2;  -- externally provided credentials
   OCI_CRED_PROXY   : constant := 3;  -- proxy authentication
   ------------------------------------------------------------------

   TRUE  : constant := 1;
   FALSE : constant := 0;

   subtype Deword is Eb4;
   subtype Dsize_T is Ub4;
   subtype Dboolean is Sb4;
   subtype Duword is Ub4;
   subtype DSWord is Sb4;
   subtype Dword is DSWord;

   Null_Indicator : constant := -1;
   Not_Null_Indicator : constant := 0;

   OCI_STMT_SELECT  : constant := 1;
   OCI_STMT_UPDATE  : constant := 2;
   OCI_STMT_DELETE  : constant := 3;
   OCI_STMT_INSERT  : constant := 4;
   OCI_STMT_CREATE  : constant := 5;
   OCI_STMT_DROP    : constant := 6;
   OCI_STMT_ALTER   : constant := 7;
   OCI_STMT_BEGIN   : constant := 8;
   OCI_STMT_DECLARE : constant := 9;

   type OCIDuration is new Ub2;

   OCI_DURATION_BEGIN   : constant OCIDuration := 10;
   OCI_DURATION_NULL    : constant OCIDuration := OCI_DURATION_BEGIN - 1;
   OCI_DURATION_DEFAULT : constant OCIDuration := OCI_DURATION_BEGIN - 2;
   OCI_DURATION_NEXT    : constant OCIDuration := OCI_DURATION_BEGIN - 3;
   OCI_DURATION_SESSION : constant OCIDuration := OCI_DURATION_BEGIN;
   OCI_DURATION_TRANS   : constant OCIDuration := OCI_DURATION_BEGIN + 1;


   OCI_UCBTYPE_ENTRY   : constant := 1; --  entry callback
   OCI_UCBTYPE_EXIT    : constant := 2; --  exit callback
   OCI_UCBTYPE_REPLACE : constant := 3; --  replacement callback

   type OCITime is record
      OCITimeHH : Ub1;
      OCITimeMI : Ub1;
      OCITimeSS : Ub1;
   end record;

   type OCIDate is record
      OCIDateYYYY : Sb2;
      OCIDateMM   : Ub1;
      OCIDateDD   : Ub1;
      OCIDateTime : OCITime;
   end record;

   OCI_NUMBER_SIZE : constant := 22;

   type OCINumber is array (1 .. OCI_NUMBER_SIZE) of Ub1;

   pragma Convention (C, OCINumber);

   type OCIString is new DVoid;

   type A_OCILobLocator is access all OCILobLocator;

   subtype OCICLOBLOCATOR is OCILobLocator;

   subtype OCIBLOBLOCATOR is OCILobLocator;

   subtype OCIBFILELOCATOR is OCILobLocator;

   function OCIEnvCreate
     (Envh      : access OCIEnv;
      Mode      : in     Ub4;
      Ctxp      : in     DVoid     := Empty_Handle;
      Malocfp   : in     DVoid     := Empty_Handle;
      Ralocfp   : in     DVoid     := Empty_Handle;
      Mfreefp   : in     DVoid     := Empty_Handle;
      Xtramemsz : in     Undefined := Empty;
      Usrmempp  : in     Undefined := Empty)
      return    SWord;

   function OCIErrorGet
     (Hndlp    : in     OCIHandle;
      Recordno : in     Ub4       := 1;
      Sqlstate : in     Undefined := Empty;
      Errcodep : access Sb4;
      Bufp     : in     Text_Ptr;
      Bufsiz   : in     Ub4;
      Htype    : in     Ub4)
      return   SWord;

   function OCIErrorGet
     (Hndlp    : in     OCIError;
      Recordno : in     Ub4       := 1;
      Sqlstate : in     Undefined := Empty;
      Errcodep : access Sb4;
      Bufp     : in     Text_Ptr;
      Bufsiz   : in     Ub4;
      Htype    : in     Ub4)
      return   SWord;

   function OCILogon
     (Envhp      : in     OCIEnv;
      Errhp      : in     OCIError;
      Svchp      : access OCISvcCtx;
      Username   : in     Text;
      Uname_Len  : in     Ub4;
      PasSWord   : in     Text;
      Passwd_Len : in     Ub4;
      Dbname     : in     Text;
      Dbname_Len : in     Ub4)
      return SWord;

   function OCILogoff
     (Svchp  : in OCISvcCtx;
      Errhp  : in OCIError)
      return SWord;

   function OCIInitialize
     (Mode    : in Ub4;
      Ctxp    : in Undefined := Empty;
      Malocfp : in Undefined := Empty;
      Ralocfp : in Undefined := Empty;
      Mfreefp : in Undefined := Empty)
      return  SWord;

   function OCIEnvInit
     (Envh      : access OCIEnv;
      Mode      : in     Ub4;
      Xtramemsz : in     Undefined := Empty;
      Usrmempp  : in     Undefined := Empty)
      return    SWord;

   function OCIHandleAlloc
     (Parenth    : in     OCIHandle;
      Hndlpp     : access OCIHandle;
      Htype      : in     Ub4;
      Xtramem_Sz : in     C.size_t  := 0;
      Usrmempp   : in     Undefined := Empty)
      return     SWord;

   function OCIHandleFree
     (Hndlp : OCIHandle;
      Htype : Ub4)
      return SWord;

   function OCIStmtPrepare
     (Stmtp    : OCIStmt;
      Errhp    : OCIError;
      Stmt     : Text;
      Stmt_Len : Ub4;
      Language : Ub4      := OCI_NTV_SYNTAX;
      Mode     : Ub4      := OCI_DEFAULT)
      return SWord;

   function OCIStmtExecute
     (Svchp    : OCISvcCtx;
      Stmtp    : OCIStmt;
      Errhp    : OCIError;
      Iters    : Ub4;
      Rowoff   : Ub4       := 0;
      Snap_In  : OCIHandle := Empty_Handle;
      Snap_Out : OCIHandle := Empty_Handle;
      Mode     : Ub4       := OCI_DEFAULT)
      return SWord;

   function OCIParamGet
     (Hndlp   :        OCIHandle;
      Htype   :        Ub4;
      Errhp   :        OCIError;
      Parmdpp : access OCIParam;
      Pos     :        Ub4)
      return SWord;

   function OCIAttrGet
     (Trgthndlp  : in OCIHandle;
      Trghndltyp : in Ub4;
      Attributep : in DVoid;
      Sizep      : in A_Ub4;
      Attrtype   : in Ub4;
      Errhp      : in OCIError)
      return SWord;

   function OCIAttrGet
     (Trgthndlp  :        OCIHandle;
      Trghndltyp :        Ub4;
      Attributep : access CStr.chars_ptr;
      Sizep      : access Ub4;
      Attrtype   :        Ub4;
      Errhp      :        OCIError)
      return SWord;

   function OCIAttrSet
     (Trgthndlp  :  OCIHandle;
      Trghndltyp :  Ub4;
      Attributep :  Text;
      Size       :  Ub4;
      Attrtype   :  Ub4;
      Errhp      :  OCIError)
      return SWord;

   function OCIAttrSet
     (Trgthndlp  :  OCIHandle;
      Trghndltyp :  Ub4;
      Attributep :  DVoid;
      Size       :  Ub4;
      Attrtype   :  Ub4;
      Errhp      :  OCIError)
      return SWord;

   function OCIStmtFetch (
         Stmtp       : OCIStmt;
         Errhp       : OCIError;
         Nrows       : Ub4      := 1;
         Orientation : Ub2      := OCI_FETCH_NEXT;
         Mode        : Ub4      := OCI_DEFAULT)
     return SWord;

   function OCIBindByName
     (Stmtp       : in     OCIStmt;
      Bindpp      : access OCIBind;
      Errhp       : in     OCIError;
      Placeholder : in     Text;
      Placeh_Len  : in     Sb4;
      Valuep      : in     DVoid;
      Value_Sz    : in     Sb4;
      Dty         : in     Ub2;
      Indp        : in     A_Sb2;
      Alenp       : in     A_Ub2    := null;
      Rcodep      : in     A_Ub2    := null;
      Maxarr_Len  : in     Ub4      := 0;
      Curelep     : in     A_Ub4    := null;
      Mode        : in     Ub4      := OCI_DEFAULT)
      return      SWord;

   function OCIBindByName
     (Stmtp       :        OCIStmt;
      Bindpp      : access OCIBind;
      Errhp       :        OCIError;
      Placeholder :        Text;
      Placeh_Len  :        Sb4;
      Valuep      :        Text_Ptr;
      Value_Sz    :        Sb4;
      Dty         :        Ub2;
      Indp        : access Sb2;
      Alenp       :        A_Ub2    := null;
      Rcodep      :        A_Ub2    := null;
      Maxarr_Len  :        Ub4      := 0;
      Curelep     :        A_Ub4    := null;
      Mode        :        Ub4      := OCI_DEFAULT)
      return SWord;

   function OCIBindByPos
     (Stmtp      : in     OCIStmt;
      Bindpp     : access OCIBind;
      Errhp      : in     OCIError;
      Position   : in     Ub4;
      Valuep     : in     DVoid;
      Value_Sz   : in     Sb4;
      Dty        : in     Ub2;
      Indp       : in     A_Sb2;
      Alenp      : in     A_Ub2    := null;
      Rcodep     : in     A_Ub2    := null;
      Maxarr_Len : in     Ub4      := 0;
      Curelep    : in     A_Ub4    := null;
      Mode       : in     Ub4      := OCI_DEFAULT)
      return     SWord;

   function OCIBindByPos (
         Stmtp      :        OCIStmt;
         Bindpp     : access OCIBind;
         Errhp      :        OCIError;
         Position   :        Ub4;
         Valuep     :        Text_Ptr;
         Value_Sz   :        Sb4;
         Dty        :        Ub2;
         Indp       : access Sb2;
         Alenp      :        A_Ub2    := null;
         Rcodep     :        A_Ub2    := null;
         Maxarr_Len :        Ub4      := 0;
         Curelep    :        A_Ub4    := null;
         Mode       :        Ub4      := OCI_DEFAULT)
     return SWord;

   function OCIDateToText
     (Err         :        OCIError;
      Date        :        OCIDate;
      Fmt         :        Text;
      Fmt_Length  :        Ub1;
      Lang_Name   :        Text_Ptr;
      Lang_Length :        Ub4;
      Buf_Size    : access Ub4;
      Buf         :        Text_Ptr)
      return SWord;

   function OCIDateFromText
     (Err          :        OCIError;
      Date_Str     :        Text;
      d_Str_Length :        Ub4;
      Fmt          :        Text;
      Fmt_Length   :        Ub1;
      Lang_Name    :        Text_Ptr;
      Lang_Length  :        Ub4;
      Date         : access OCIDate)
      return SWord;

   function OCIDateCompare
     (Err    : in     OCIError;
      date1  : in     OCIDate;
      date2  : in     OCIDate;
      Result : access SWord) return SWord;

   function OCIDefineByPos
     (Stmtp    : in     OCIStmt;
      Defnpp   : access OCIDefine;
      Errhp    : in     OCIError;
      Position : in     Ub4;
      Value    : in     DVoid;
      Value_Sz : in     Sb4;
      Dty      : in     Ub2;
      Indp     : in     A_Sb2;
      Rlenp    : in     A_Ub2     := null;
      Rcodep   : in     A_Ub2     := null;
      Mode     : in     Ub4       := OCI_DEFAULT)
      return   SWord;

   function OCIDefineByPos
     (Stmtp    : in     OCIStmt;
      Defnpp   : access OCIDefine;
      Errhp    : in     OCIError;
      Position : in     Ub4;
      Value    : in     Text_Ptr;
      Value_Sz : in     Sb4;
      Dty      : in     Ub2;
      Indp     : access Sb2;
      Rlenp    : in     A_Ub2     := null;
      Rcodep   : in     A_Ub2     := null;
      Mode     : in     Ub4       := OCI_DEFAULT)
      return   SWord;

   function OCINumberToText
     (Err        :        OCIError;
      Date       :        OCINumber;
      Fmt        :        Text;
      Fmt_Length :        Ub1;
      Nls_Name   :        Undefined;
      Nls_Length :        Ub4;
      Buf_Size   : access Ub4;
      Buf        :        Text_Ptr)
      return     SWord;

   function OCINumberFromText
     (Err          :        OCIError;
      Buf          :        Text;
      Str_Length   :        Ub4;
      Fmt          :        Text;
      Fmt_Length   :        Ub4;
      NLS_params   :        Undefined;
      nls_p_length :        Ub4;
      number       : access OCINumber)
      return SWord;

   function OCIDateSysDate
     (Err  :        OCIError;
      Date : access OCIDate)
      return SWord;

   function OCIStringResize
     (Env      :        OCIEnv;
      Err      :        OCIError;
      New_Size :        Ub4;
      Str      : access OCIString)
      return SWord;

   function OCIStringAssignText
     (Env     :        OCIEnv;
      Err     :        OCIError;
      Rhs     :        Text;
      Rhs_Len :        Ub2;
      Lhs     : access OCIString)
      return SWord;

   function OCIStringPtr
     (Env : OCIEnv;
      Vs  : OCIString)
      return CStr.chars_ptr;

   function OCIStringSize
     (Env : OCIEnv;
      Vs  : OCIString)
      return Ub4;

   function OCIStringAllocSize
     (Env       :        OCIEnv;
      Vs        :        OCIString;
      Allocsize : access Ub4)
      return SWord;

   function OCIDescriptorAlloc
     (Parenth    :        OCIEnv;
      Descpp     : access DVoid;
      Htype      :        Ub4;
      Xtramem_Sz :        C.size_t  := 0;
      Usrmempp   :        Undefined := 0)
      return SWord;

   function OCIDescriptorFree
     (Descp : OCIHandle;
      Dtype : Ub4)
      return SWord;

   function OCIServerVersion
     (Hndlp    : OCIHandle;
      Errhp    : OCIError;
      Bufp     : Text_Ptr;
      Bufsz    : Ub4;
      Hndltype : Ub1)
      return SWord;

   function OCITransCommit
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Flags : Ub4       := OCI_DEFAULT)
      return SWord;

   function OCITransRollback
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Flags : Ub4       := OCI_DEFAULT)
      return SWord;

   function OCINumberAbs
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberArcCos
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberArcSin
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberArcTan
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberAssign
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberCeil
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberCos
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberExp
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberFloor
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberHypCos
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberHypSin
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberHypTan
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberLn
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberNeg
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberSin
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberSqrt
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberTan
     (Err    :        OCIError;
      Number :        OCINumber;
      Result : access OCINumber)
      return SWord;

   function OCINumberArcTan2
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberAdd
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberDiv
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberLog
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberMod
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberMul
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberPower
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberSub
     (Err     :        OCIError;
      Number1 :        OCINumber;
      Number2 :        OCINumber;
      Result  : access OCINumber)
      return SWord;

   function OCINumberToReal
     (Err        :        OCIError;
      Number     :        OCINumber;
      Rsl_Length :        UWord;
      Rsl        : access Long_Float)
      return SWord;

   function OCINumberFromReal
     (Err         :        OCIError;
      Rnum        : access Long_Float;
      Rnum_Length :        UWord;
      Number      : access OCINumber)
      return SWord;

   function OCILobAppend
     (Svchp    : OCISvcCtx;
      Errhp    : OCIError;
      Dst_Locp : OCILobLocator;
      Src_Locp : OCILobLocator)
      return SWord;

   function OCILobAssign
     (Envhp     : OCIEnv;
      Errhp     : OCIError;
      Src_Locp  : OCILobLocator;
      Dst_Locpp : A_OCILobLocator)
      return SWord;

   function OCILobCharSetForm
     (Envhp : OCIEnv;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Csfrm : A_Ub1)
      return SWord;

   function OCILobCharSetId
     (Envhp : OCIEnv;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Csid  : A_Ub2)
      return SWord;

   function OCILobCopy
     (Svchp      : OCISvcCtx;
      Errhp      : OCIError;
      Dst_Locp   : OCILobLocator;
      Src_Locp   : OCILobLocator;
      Amount     : Ub4;
      Dst_Offset : Ub4;
      Src_Offset : Ub4)
      return SWord;

   function OCILobCreateTemporary
     (Svchp    : OCISvcCtx;
      Errhp    : OCIError;
      Locp     : OCILobLocator;
      Csid     : Ub2;
      Csfrm    : Ub1;
      Lobtype  : Ub1;
      Cache    : C.int;
      Duration : OCIDuration)
      return SWord;

   function OCILobClose
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator)
      return SWord;

   function OCILobDisableBuffering
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator)
      return SWord;

   function OCILobEnableBuffering
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator)
      return SWord;

   function OCILobErase
     (Svchp  : OCISvcCtx;
      Errhp  : OCIError;
      Locp   : OCILobLocator;
      Amount : A_Ub4;
      Offset : Ub4)
      return SWord;

   function OCILobFileClose
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Filep : OCILobLocator)
      return SWord;

   function OCILobFileCloseAll
     (Svchp : OCISvcCtx;
      Errhp : OCIError)
      return SWord;

   function OCILobFileExists
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Filep : OCILobLocator;
      Flag  : C.int)
      return SWord;

   function OCILobFileGetName
     (Envhp     : OCIEnv;
      Errhp     : OCIError;
      Filep     : OCILobLocator;
      Dir_Alias : OraText;
      D_Length  : A_Ub2;
      Filename  : OraText;
      F_Length  : A_Ub2)
      return SWord;

   function OCILobFileIsOpen
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Filep : OCILobLocator;
      Flag  : A_Int)
      return SWord;

   function OCILobFileOpen
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Filep : OCILobLocator;
      Mode  : Ub1)
      return SWord;

   function OCILobFileSetName
     (Envhp     : OCIEnv;
      Errhp     : OCIError;
      Filepp    : A_OCILobLocator;
      Dir_Alias : OraText;
      D_Length  : Ub2;
      Filename  : OraText;
      F_Length  : Ub2)
      return SWord;

   function OCILobFlushBuffer
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Flag  : Ub4)
      return SWord;

   function OCILobFreeTemporary
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator)
      return SWord;

   function OCILobGetChunkSize
     (Svchp      : OCISvcCtx;
      Errhp      : OCIError;
      Locp       : OCILobLocator;
      Chunksizep : A_Ub4)
      return SWord;

   function OCILobGetLength
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Lenp  : A_Ub4)
      return SWord;

   function OCILobIsEqual
     (Envhp    : OCIEnv;
      X        : OCILobLocator;
      Y        : OCILobLocator;
      Is_Equal : A_Int)
      return SWord;

   function OCILobIsOpen
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Flag  : A_Int)
      return SWord;

   function OCILobIsTemporary
     (Envhp        : OCIEnv;
      Errhp        : OCIError;
      Locp         : OCILobLocator;
      Is_Temporary : A_Int)
      return SWord;

   function OCILobLoadFromFile
     (Svchp      : OCISvcCtx;
      Errhp      : OCIError;
      Dst_Locp   : OCILobLocator;
      Src_Filep  : OCILobLocator;
      Amount     : Ub4;
      Dst_Offset : Ub4;
      Src_Offset : Ub4)
      return SWord;

   function OCILobLocatorAssign
     (Svchp     : OCISvcCtx;
      Errhp     : OCIError;
      Src_Locp  : OCILobLocator;
      Dst_Locpp : A_OCILobLocator)
      return SWord;

   function OCILobLocatorIsInit
     (Envhp          : OCIEnv;
      Errhp          : OCIError;
      Locp           : OCILobLocator;
      Is_Initialized : A_Int)
      return SWord;

   function OCILobOpen
     (Svchp : OCISvcCtx;
      Errhp : OCIError;
      Locp  : OCILobLocator;
      Mode  : Ub1)
      return SWord;

   function OCILobRead
     (Svchp  : OCISvcCtx;
      Errhp  : OCIError;
      Locp   : OCILobLocator;
      Amtp   : A_Ub4;
      Offset : Ub4;
      Bufp   : System.Address;
      Bufl   : Ub4;
      Ctxp   : System.Address;
      Cbfp   : Function_Pointer;
      Csid   : Ub2;
      Csfrm  : Ub1 := SQLCS_IMPLICIT)
      return SWord;

   function OCILobTrim
     (Svchp  : OCISvcCtx;
      Errhp  : OCIError;
      Locp   : OCILobLocator;
      Newlen : Ub4)
      return SWord;

   function OCILobWrite
     (Svchp  : OCISvcCtx;
      Errhp  : OCIError;
      Locp   : OCILobLocator;
      Amtp   : A_Ub4;
      Offset : Ub4;
      Bufp   : System.Address;
      Buflen : Ub4;
      Piece  : Ub1;
      Ctxp   : System.Address;
      Cbfp   : Function_Pointer;
      Csid   : Ub2;
      Csfrm  : Ub1 := SQLCS_IMPLICIT)
      return SWord;

   function OCILobWriteAppend (
         Svchp : OCISvcCtx;
         Errhp : OCIError;
         Lobp  : OCILobLocator;
         Amtp  : A_Ub4;
         Bufp  : System.Address;
         Bufl  : Ub4;
         Piece : Ub1;
         Ctxp  : System.Address;
         Cbfp  : Function_Pointer;
         Csid  : Ub2;
         Csfrm : Ub1 := SQLCS_IMPLICIT)
     return SWord;

   function OCIStmtGetBindInfo (
         Stmtp    : OCIStmt;
         Errhp    : OCIError;
         Size     : Ub4;
         Startloc : Ub4;
         Found    : A_Sb4;
         Bvnp     : Vector_Of_OraText;
         Bvnl     : Vector_Of_Ub1;
         Invp     : Vector_Of_OraText;
         Inpl     : Vector_Of_Ub1;
         Dupl     : Vector_Of_Ub1;
         Hndl     : Vector_Of_OCIBind)
     return SWord;

   function OCIStmtGetBindInfo (
         Stmtp    : OCIStmt;
         Errhp    : OCIError;
         Size     : Ub4;
         Startloc : Ub4;
         Found    : A_Sb4;
         Bvnp     : A_Vector_Of_OraText;
         Bvnl     : A_Vector_Of_Ub1;
         Invp     : A_Vector_Of_OraText;
         Inpl     : A_Vector_Of_Ub1;
         Dupl     : A_Vector_Of_Ub1;
         Hndl     : A_Vector_Of_OCIBind)
     return SWord;

   function OCIDescribeAny (
         Svchp      : OCISvcCtx;
         Errhp      : OCIError;
         Objptr     : DVoid;
         Objnm_Len  : Ub4;
         Objptr_Typ : Ub1;
         Info_Level : Ub1;
         Objtyp     : Ub1;
         Dschp      : OCIDescribe)
     return SWord;

   type OCICallbackInBind is access function (
                                ictxp  : DVoid;
                                bindp  : OCIBind;
                                iter   : Ub4;
                                index  : Ub4;
                                bufpp  : access DVoid;
                                alenp  : access Ub4;
                                piecep : access Ub1;
                                indp   : access DVoid) return SWord;
   pragma Convention (C, OCICallbackInBind);

   type OCICallbackOutBind is access function (
                                octxp   : DVoid;
                                bindp   : OCIBind;
                                iter    : Ub4;
                                index   : Ub4;
                                bufp    : access DVoid;
                                alenpp  : access A_Ub4;
                                piecep  : access Ub1;
                                indpp   : access DVoid;
                                rcodepp : access A_Ub2) return SWord;
   pragma Convention (C, OCICallbackOutBind);

   function OCIBindDynamic (
           bindp  : OCIBind;
           errhp  : OCIError;
           ictxp  : DVoid;
           icbfp  : DVoid;
           octxp  : DVoid;
           ocbfp  : DVoid) return SWord;

   function OCIBindDynamic (
           bindp  : OCIBind;
           errhp  : OCIError;
           ictxp  : DVoid;
           icbfp  : OCICallbackInBind;
           octxp  : DVoid;
           ocbfp  : OCICallbackOutBind) return SWord;


   type OCICallbackDefine is access function (
                                octxp   : DVoid;
                                defnp   : OCIDefine;
                                iter    : Ub4;
                                bufpp   : access DVoid;
                                alenpp  : access A_Ub4;
                                piecep  : access Ub1;
                                indpp   : access DVoid;
                                rcodepp : access A_Ub2) return SWord;
   pragma Convention (C, OCICallbackDefine);

   function OCIDefineDynamic
     (defnp  : in OCIDefine;
      errhp  : in OCIError;
      octxp  : in DVoid;
      ocbfp  : in OCICallbackDefine)
      return SWord;

   function OCIDefineDynamic
     (defnp  : in OCIDefine;
      errhp  : in OCIError;
      octxp  : in DVoid;
      ocbfp  : in DVoid)
      return SWord;

   --  External Proc Callbacks  --
   function OCIExtProcGetEnv
     (octxp  : in OCIExtProcContext;
      Envhp  : access OCIEnv;
      Svchp  : access OCISvcCtx;
      Errhp  : access OCIError) return SWord;

   function OCITerminate (Mode : Ub4 := OCI_DEFAULT) return SWord;

   type OCIUserCallback is access function
     (ctxp       : in DVoid;
      hndlp      : in OCIHandle;
      htype      : in Ub4;
      fcode      : in Ub4;
      when_call  : in Ub1;
      returnCode : in SWord;
      errnop     : access Sb4;
      arglist    : in DVoid)
      return     SWord;
   pragma Convention (C, OCIUserCallback);

   function OCIUserCallbackRegister (hndlp     : in OCIHandle;
                                     htype     : in Ub4;
                                     ehndlp    : in OCIError;
                                     callback  : in OCIUserCallback;
                                     ctxp      : in DVoid;
                                     fcode     : in Ub4;
                                     when_call : in Ub1)
                                     return    SWord;

   function OCIUserCallbackRegister (hndlp     : in OCIHandle;
                                     htype     : in Ub4;
                                     ehndlp    : in OCIError;
                                     callback  : in DVoid;
                                     ctxp      : in DVoid;
                                     fcode     : in Ub4;
                                     when_call : in Ub1)
                                     return    SWord;

   function OCIServerAttach (srvhp      : OCIServer;
                             errhp      : OCIError;
                             dblink     : Text;
                             dblink_len : Sb4;
                             mode       : Ub4 := OCI_DEFAULT) return SWord;

   function OCIServerDetach (srvhp      : OCIServer;
                             errhp      : OCIError;
                             mode       : Ub4 := OCI_DEFAULT) return SWord;

   function OCISessionBegin (svchp :  OCISvcCtx;
                             errhp : OCIError;
                             usrhp : OCISession;
                             credt : Ub4 := OCI_CRED_RDBMS;
                             mode : Ub4 := OCI_DEFAULT) return SWord;

   function OCISessionEnd (svchp :  OCISvcCtx;
                             errhp : OCIError;
                             usrhp : OCISession;
                             mode : Ub4 := OCI_DEFAULT) return SWord;

   function OCIBreak (hndlp : OCIHandle;
                       ehndlp : OCIError) return SWord;

   function OCIReset (hndlp : OCIHandle;
                       ehndlp : OCIError) return SWord;

   function OCISubscriptionRegister
     (svchp     : OCISvcCtx;
      subscrhpp : access OCISubscription;
      count     : Ub2;
      errhp     : OCIError;
      mode      : Ub4) return SWord;

   function OCISubscriptionUnRegister
     (svchp    : OCISvcCtx;
      subscrhp : OCISubscription;
      errhp    : OCIError;
      mode     : Ub4)
      return   SWord;

   function OCISubscriptionEnable
     (subscrhp : OCISubscription;
      errhp    : OCIError;
      mode     : Ub4)
      return SWord;

   function OCISubscriptionDisable
     (subscrhp : OCISubscription;
      errhp    : OCIError;
      mode     : Ub4)
      return SWord;

   function OCISubscriptionPost
     (svchp      : in     OCISvcCtx;
      subscrhpp  : access OCISubscription;
      count      : in     Ub2;
      errhp      : in     OCIError;
      mode       : in     Ub4)
      return SWord;

private

   pragma Import (C, OCIEnvCreate, "OCIEnvCreate");
   pragma Import (C, OCIErrorGet, "OCIErrorGet");
   pragma Import (C, OCILogon, "OCILogon");
   pragma Import (C, OCILogoff, "OCILogoff");
   pragma Import (C, OCIInitialize, "OCIInitialize");
   pragma Import (C, OCIEnvInit, "OCIEnvInit");
   pragma Import (C, OCIHandleAlloc, "OCIHandleAlloc");
   pragma Import (C, OCIHandleFree, "OCIHandleFree");
   pragma Import (C, OCIStmtPrepare, "OCIStmtPrepare");
   pragma Import (C, OCIStmtExecute, "OCIStmtExecute");
   pragma Import (C, OCIParamGet, "OCIParamGet");
   pragma Import (C, OCIAttrGet, "OCIAttrGet");
   pragma Import (C, OCIAttrSet, "OCIAttrSet");
   pragma Import (C, OCIStmtFetch, "OCIStmtFetch");
   pragma Import (C, OCIBindByName, "OCIBindByName");
   pragma Import (C, OCIBindByPos, "OCIBindByPos");
   pragma Import (C, OCIDefineByPos, "OCIDefineByPos");
   pragma Import (C, OCIDateToText, "OCIDateToText");
   pragma Import (C, OCIDateFromText, "OCIDateFromText");
   pragma Import (C, OCIDateCompare, "OCIDateCompare");
   pragma Import (C, OCIDateSysDate, "OCIDateSysDate");
   pragma Import (C, OCIStringResize, "OCIStringResize");
   pragma Import (C, OCIStringAssignText, "OCIStringAssignText");
   pragma Import (C, OCIStringPtr, "OCIStringPtr");
   pragma Import (C, OCIStringSize, "OCIStringSize");
   pragma Import (C, OCIStringAllocSize, "OCIStringAllocSize");
   pragma Import (C, OCIDescriptorAlloc, "OCIDescriptorAlloc");
   pragma Import (C, OCIDescriptorFree, "OCIDescriptorFree");
   pragma Import (C, OCIServerVersion, "OCIServerVersion");
   pragma Import (C, OCITransCommit, "OCITransCommit");
   pragma Import (C, OCITransRollback, "OCITransRollback");
   pragma Import (C, OCINumberAbs, "OCINumberAbs");
   pragma Import (C, OCINumberArcCos, "OCINumberArcCos");
   pragma Import (C, OCINumberArcSin, "OCINumberArcSin");
   pragma Import (C, OCINumberArcTan, "OCINumberArcTan");
   pragma Import (C, OCINumberAssign, "OCINumberAssign");
   pragma Import (C, OCINumberCeil, "OCINumberCeil");
   pragma Import (C, OCINumberCos, "OCINumberCos");
   pragma Import (C, OCINumberExp, "OCINumberExp");
   pragma Import (C, OCINumberFloor, "OCINumberFloor");
   pragma Import (C, OCINumberHypCos, "OCINumberHypCos");
   pragma Import (C, OCINumberHypSin, "OCINumberHypSin");
   pragma Import (C, OCINumberHypTan, "OCINumberHypTan");
   pragma Import (C, OCINumberLn, "OCINumberLn");
   pragma Import (C, OCINumberNeg, "OCINumberNeg");
   pragma Import (C, OCINumberSin, "OCINumberSin");
   pragma Import (C, OCINumberSqrt, "OCINumberSqrt");
   pragma Import (C, OCINumberTan, "OCINumberTan");
   pragma Import (C, OCINumberArcTan2, "OCINumberArcTan2");
   pragma Import (C, OCINumberAdd, "OCINumberAdd");
   pragma Import (C, OCINumberDiv, "OCINumberDiv");
   pragma Import (C, OCINumberLog, "OCINumberLog");
   pragma Import (C, OCINumberMod, "OCINumberMod");
   pragma Import (C, OCINumberMul, "OCINumberMul");
   pragma Import (C, OCINumberPower, "OCINumberPower");
   pragma Import (C, OCINumberSub, "OCINumberSub");
   pragma Import (C, OCINumberToReal, "OCINumberToReal");
   pragma Import (C, OCINumberFromReal, "OCINumberFromReal");
   pragma Import (C, OCINumberToText, "OCINumberToText");
   pragma Import (C, OCINumberFromText, "OCINumberFromText");
   pragma Import (C, OCILobAppend, "OCILobAppend");
   pragma Import (C, OCILobAssign, "OCILobAssign");
   pragma Import (C, OCILobCharSetForm, "OCILobCharSetForm");
   pragma Import (C, OCILobCharSetId, "OCILobCharSetId");
   pragma Import (C, OCILobCopy, "OCILobCopy");
   pragma Import (C, OCILobCreateTemporary, "OCILobCreateTemporary");
   pragma Import (C, OCILobClose, "OCILobClose");
   pragma Import (C, OCILobDisableBuffering, "OCILobDisableBuffering");
   pragma Import (C, OCILobEnableBuffering, "OCILobEnableBuffering");
   pragma Import (C, OCILobErase, "OCILobErase");
   pragma Import (C, OCILobFileClose, "OCILobFileClose");
   pragma Import (C, OCILobFileCloseAll, "OCILobFileCloseAll");
   pragma Import (C, OCILobFileExists, "OCILobFileExists");
   pragma Import (C, OCILobFileGetName, "OCILobFileGetName");
   pragma Import (C, OCILobFileIsOpen, "OCILobFileIsOpen");
   pragma Import (C, OCILobFileOpen, "OCILobFileOpen");
   pragma Import (C, OCILobFileSetName, "OCILobFileSetName");
   pragma Import (C, OCILobFlushBuffer, "OCILobFlushBuffer");
   pragma Import (C, OCILobFreeTemporary, "OCILobFreeTemporary");
   pragma Import (C, OCILobGetChunkSize, "OCILobGetChunkSize");
   pragma Import (C, OCILobGetLength, "OCILobGetLength");
   pragma Import (C, OCILobIsEqual, "OCILobIsEqual");
   pragma Import (C, OCILobIsOpen, "OCILobIsOpen");
   pragma Import (C, OCILobIsTemporary, "OCILobIsTemporary");
   pragma Import (C, OCILobLoadFromFile, "OCILobLoadFromFile");
   pragma Import (C, OCILobLocatorAssign, "OCILobLocatorAssign");
   pragma Import (C, OCILobLocatorIsInit, "OCILobLocatorIsInit");
   pragma Import (C, OCILobOpen, "OCILobOpen");
   pragma Import (C, OCILobRead, "OCILobRead");
   pragma Import (C, OCILobTrim, "OCILobTrim");
   pragma Import (C, OCILobWrite, "OCILobWrite");
   pragma Import (C, OCILobWriteAppend, "OCILobWriteAppend");
   pragma Import (C, OCIStmtGetBindInfo, "OCIStmtGetBindInfo");
   pragma Import (C, OCIDescribeAny, "OCIDescribeAny");
   pragma Import (C, OCIBindDynamic, "OCIBindDynamic");
   pragma Import (C, OCIDefineDynamic, "OCIDefineDynamic");
   pragma Import (C, OCIExtProcGetEnv, "ociepgoe");
   pragma Import (C, OCITerminate, "OCITerminate");
   pragma Import (C, OCIUserCallbackRegister, "OCIUserCallbackRegister");
   pragma Import (C, OCIServerAttach, "OCIServerAttach");
   pragma Import (C, OCIServerDetach, "OCIServerDetach");
   pragma Import (C, OCISessionBegin, "OCISessionBegin");
   pragma Import (C, OCISessionEnd, "OCISessionEnd");
   pragma Import (C, OCIBreak, "OCIBreak");
   pragma Import (C, OCIReset, "OCIReset");
   pragma Import (C, OCISubscriptionRegister, "OCISubscriptionRegister");
   pragma Import (C, OCISubscriptionUnRegister, "OCISubscriptionUnRegister");
   pragma Import (C, OCISubscriptionDisable, "OCISubscriptionDisable");
   pragma Import (C, OCISubscriptionEnable, "OCISubscriptionEnable");
   pragma Import (C, OCISubscriptionPost, "OCISubscriptionPost");

end OCI.Lib;
