/* $Id: serial_modem_dialout.c,v 1.11 2009-01-28 12:59:22 potyra Exp $ 
 *
 *  Dialout architecture of the serial modem. Handles connections to the 
 *  teleophone line.
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <inttypes.h>

#include "glue-log.h"
#include "glue-main.h"

#include "serial_modem_dialout.h"

/* ********************* DEFINITIONS ************************** */
/* timeout until DIALTONE must be received */
#define TIMEOUT_DIALTONE	5 * TIME_HZ

/* timeout until CONNECT must be received (after DIAL) */
#define TIMEOUT_CONNECT		50 * TIME_HZ

/* busy led timeout */
#define TIMEOUT_BUSY_LED	0.5 * TIME_HZ


/* different states of internal fsm */
enum dialout_states {
	STATE_OFFLINE, 		/* not connected */
	STATE_ONLINE,		/* connection established */
	STATE_INCOMING, 	/* incoming call, not answered. */
	STATE_WAIT_DIALT,	/* wait for dialtone */
	STATE_WAIT_LIFT		/* wait for peer answering the call */
};

/* ****************** GLOBAL VARIABLES ************************ */

/* internal state */
static enum dialout_states dialout_state;
/* number that should get dialed */
static uint32_t stored_number;

/* use input from serial interface */
bool use_ser_bus;

/* ******************** IMPLEMENTAION ************************* */

static int
modem_dialout_get_power_state(void)
{
	return serial_modem_power_state();
}

/* timer waiting for peer to lift the receiver */
static void
modem_dialout_timer_lift(void *data) 
{
	if ( modem_dialout_get_power_state() 
	 && (dialout_state == STATE_WAIT_LIFT)
	 ) {
		modem_result(__cpssp, RESULT_NO_ANSWER);
		dialout_state = STATE_OFFLINE;
		modem_phone_send_ctrl(SIG_TELE_HANGUP);
		modem_opt_online_led_set(0);
	}
}

/* timer waiting for dial tone */
static void
modem_dialout_timer_dialt(void *data)
{
	if ( modem_dialout_get_power_state() 
	 && (dialout_state == STATE_WAIT_DIALT)) {
	 	faum_log(FAUM_LOG_DEBUG, "modem", "dialout",
			"not dialtone.\n");
		modem_result(__cpssp, RESULT_NO_DIALTONE);
		dialout_state = STATE_OFFLINE;
	}
}

static void
modem_dialout_rxd_busy_led(void *data)
{
	modem_opt_rxd_led_set(0);
}

static void
modem_dialout_txd_busy_led(void *data)
{
	modem_opt_txd_led_set(0);
}

static void
modem_dialout_recv_dialt(enum sig_telephone_protocol p)
{
	bool ret;

	ret = time_call_delete(modem_dialout_timer_dialt, NULL);
	assert(ret == 0);

	if (p != SIG_TELE_DIALTONE) {
		modem_result(__cpssp, RESULT_NO_DIALTONE);
		dialout_state = STATE_OFFLINE;
		return;
	}

	dialout_state = STATE_WAIT_LIFT;

	faum_log(FAUM_LOG_DEBUG, "serial-modem", "dialout", 
			"actually dialing now.\n");

	time_call_after(TIMEOUT_CONNECT, modem_dialout_timer_lift, NULL);
	modem_phone_dial(stored_number);
}

static void
modem_dialout_recv_tele_online(enum sig_telephone_protocol p) 
{
	/* sanity check: only data packets allowed here */
	switch(p) {
	case SIG_TELE_BUSY:
		/* peer hung up */
		dialout_state = STATE_OFFLINE;
		
		faum_log(FAUM_LOG_DEBUG, "serial-modem", "dialout",
				"peer hung up.\n");

		/* acknowledge offline to fsm */
		modem_fsm_set_linestate(__cpssp, false);
		modem_opt_online_led_set(0);
		break;
	default:
		faum_log(FAUM_LOG_ERROR, "serial-modem", "dialout",
				"invalid control event on tele line.\n");
	}
}

/* check for incoming call */
static void
modem_dialout_recv_tele_offline(enum sig_telephone_protocol p) 
{
	switch(p) {
	case SIG_TELE_RING:
		dialout_state = STATE_INCOMING;
		modem_result(__cpssp, RESULT_RING);
		break;
	case SIG_TELE_BUSY:
		/* peer hung up on incoming call */
		dialout_state = STATE_OFFLINE;
		break;
	default:
		faum_log(FAUM_LOG_ERROR, "serial-modem", "dialout",
				"invalid protocol %d on offline event.\n", p);
	}
}

static void
modem_dialout_recv_lift(enum sig_telephone_protocol p)
{
	bool ret;
	ret = time_call_delete(modem_dialout_timer_lift, NULL);
	assert(ret == 0);

	switch(p) {
	case SIG_TELE_CONNECT:
		dialout_state = STATE_ONLINE;
		/* TODO handshake first */
		modem_fsm_set_connected(__cpssp);
		modem_result(__cpssp, RESULT_OK);
		modem_opt_online_led_set(1);
		break;
	case SIG_TELE_BUSY:
		modem_result(__cpssp, RESULT_BUSY);
		dialout_state = STATE_OFFLINE;
		break;
	case SIG_TELE_INVALID_NUMBER:
		modem_result(__cpssp, RESULT_NO_CARRIER);
		dialout_state = STATE_OFFLINE;
		break;
	default:
		modem_result(__cpssp, RESULT_ERROR);
		dialout_state = STATE_OFFLINE;
		break;
	}
}

static void
modem_dialout_handle_event(enum sig_telephone_protocol p)
{
	if (! modem_dialout_get_power_state()) {
		return;
	}

	switch(dialout_state) {
	case STATE_ONLINE:
		modem_dialout_recv_tele_online(p);
		break;
	case STATE_OFFLINE:
	case STATE_INCOMING:
		modem_dialout_recv_tele_offline(p);
		break;
	case STATE_WAIT_DIALT:
		modem_dialout_recv_dialt(p);
		break;
	case STATE_WAIT_LIFT:
		modem_dialout_recv_lift(p);
		break;
	}
}

void
dialout_interrupt_data(void *s, uint8_t data)
{
	if (! modem_dialout_get_power_state()) {
		return;
	}

	if (dialout_state == STATE_ONLINE) {
		modem_serial_push_data(data);
		modem_opt_rxd_led_set(1);
		time_call_delete(modem_dialout_rxd_busy_led, NULL);
		time_call_after(TIMEOUT_BUSY_LED, 
				modem_dialout_rxd_busy_led, NULL);
	}
}

void
dialout_interrupt_ctrl(void *s, enum sig_telephone_protocol event)
{
	modem_dialout_handle_event(event);
}

void
dialout_interrupt_dial(void *s, uint32_t number)
{
	faum_log(FAUM_LOG_WARNING, "serial-modem", "dialout",
				"received invalid DIAL event.\n");
}

/* ************ global functions ************ */

static void
modem_dialout_reset(struct cpssp *cpssp) 
{
	/* send hangup */
	if (dialout_state == STATE_ONLINE) {
		modem_phone_send_ctrl(SIG_TELE_HANGUP);
	}

	dialout_state = STATE_OFFLINE;
	/* eventually remove timers */
	time_call_delete(modem_dialout_timer_dialt, NULL);
	time_call_delete(modem_dialout_timer_lift, NULL);
	modem_opt_online_led_set(0);

	use_ser_bus = false;
}

void
modem_dialout_dial(unsigned long number, enum dial_modes mode) 
{
	if (! modem_dialout_get_power_state()) {
		return;
	}

	if (dialout_state == STATE_INCOMING) {
		modem_result(__cpssp, RESULT_BUSY);
	}

	if (dialout_state == STATE_OFFLINE) {
	
		if (mode == DIAL_MODE_TONE) {
			faum_log(FAUM_LOG_DEBUG, "serial-modem", "dialout",
					"tone-dialing %ld\n", number);
		} else if (mode == DIAL_MODE_PULSE) {
			faum_log(FAUM_LOG_DEBUG, "serial-modem", "dialout",
					"pulse-dialing %ld\n", number);
		}
		
		/* lift the receiver ... */
		dialout_state = STATE_WAIT_DIALT;
		time_call_after(TIMEOUT_DIALTONE, 
					modem_dialout_timer_dialt, NULL);
		stored_number = number;
		modem_phone_send_ctrl(SIG_TELE_LIFT);
	}
}

void
modem_dialout_set_serbusstate(bool useit)
{
	use_ser_bus=useit;
}

/* lift the receiver */
void
modem_dialout_lift(void) 
{
	/* do nothing when already connected */
	if (dialout_state == STATE_ONLINE) {
		modem_result(__cpssp, RESULT_ERROR);
		return;
	}

	/* send lift signal */
	modem_phone_send_ctrl(SIG_TELE_LIFT);

	if (dialout_state == STATE_INCOMING) {
		dialout_state = STATE_ONLINE;
		modem_result(__cpssp, RESULT_CONNECT);
		modem_opt_online_led_set(1);
		modem_fsm_set_connected(__cpssp);
		return;
	}

	/* no incoming call: */
	/* TODO */
}

/* hangup the receiver */
void
modem_dialout_hangup(void)
{
	switch(dialout_state) {
	case STATE_OFFLINE:
	case STATE_INCOMING:
		return;
	default:
		modem_dialout_reset(__cpssp);
	}
}

void
modem_dialout_push_char(unsigned char c) 
{
	if (use_ser_bus && modem_dialout_get_power_state()) {
		switch (dialout_state) {
		case STATE_ONLINE:
			modem_phone_send_data(c);

			modem_opt_txd_led_set(1);
			time_call_delete(modem_dialout_txd_busy_led, NULL);
			time_call_after(TIMEOUT_BUSY_LED, 
					modem_dialout_txd_busy_led,  NULL);
			break;
		default:
			/* silently discard data... */
			break;
		}
	}
}

void
modem_dialout_exit(void)
{
	/* still connected --> hang up */
	if (dialout_state == STATE_ONLINE) {
		modem_phone_send_ctrl(SIG_TELE_HANGUP);
	}
}
