/* $Id: screen_split.c,v 1.9 2009-01-27 16:27:03 potyra Exp $ 
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define MAX_WIDTH	1280
#define MAX_HEIGHT	1024


#define GLYPH_WIDTH	8
#define GLYPH_HEIGHT	16
#if GLYPH_WIDTH > 8
#error "GLYPH_WIDTH can't be bigger than 8 at the moment"
#endif

#include <assert.h>
#include <getopt.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glyphs.c>

#include "glue-png.h"

char *progname;
char **screenshots;

static unsigned int width;
static unsigned int height;
static struct {
	unsigned char r, g, b;
} pixel[MAX_WIDTH][MAX_HEIGHT];

static unsigned char pattern[MAX_WIDTH / GLYPH_WIDTH * MAX_HEIGHT / GLYPH_HEIGHT][GLYPH_HEIGHT];
static unsigned int npatterns = 0;

static void
screen_read(char *filename)
{
	uint32_t *data;
	unsigned int x;
	unsigned int y;

	png_read(&data, &width, &height, filename);

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x++) {
			pixel[x][y].r = (data[y * width + x] >>  0) & 0xff;
			pixel[x][y].g = (data[y * width + x] >>  8) & 0xff;
			pixel[x][y].b = (data[y * width + x] >> 16) & 0xff;
		}
	}

	free(data);
}

static void
pattern_add(unsigned char *bits)
{
	unsigned int n;

	for (n = 0; ; n++) {
		if (n == sizeof(glyph) / sizeof(glyph[0])) {
			/* Not in old glyphs. */
			break;
		}
		if (memcmp(glyph[n].pattern, bits, GLYPH_HEIGHT) == 0) {
			/* Same pattern already in old glyph.c file. */
			return;
		}
	}
	for (n = 0; ; n++) {
		if (n == npatterns) {
			/* Add new pattern. */
			memcpy(pattern[npatterns], bits, GLYPH_HEIGHT);
			npatterns++;
			break;
		}
		if (memcmp(pattern[n], bits, GLYPH_HEIGHT) == 0) {
			/* Same pattern found. */
			return;
		}
	}
}

static void
screen_split(void)
{
	unsigned int x0;
	unsigned int y0;

	for (y0 = 0; y0 < height; y0 += GLYPH_HEIGHT) {
		for (x0 = 0; x0 < width; x0 += 8) {
			unsigned char bgr, bgg, bgb;
			unsigned int x1;
			unsigned int y1;
			unsigned char bits[GLYPH_HEIGHT];

			bgr = pixel[x0][y0].r;
			bgg = pixel[x0][y0].g;
			bgb = pixel[x0][y0].b;

			for (y1 = 0; y1 < GLYPH_HEIGHT; y1++) {
				bits[y1] = 0x00;
				for (x1 = 0; x1 < 8; x1++) {
					bits[y1] <<= 1;
					if (pixel[x0 + x1][y0 + y1].r != bgr
					 || pixel[x0 + x1][y0 + y1].g != bgg
					 || pixel[x0 + x1][y0 + y1].b != bgb) {
						bits[y1] |= 1;
					}
				}
			}

			pattern_add(bits);
		}
	}
}

static void
screen_out(void)
{
	unsigned int n;
	unsigned int y;
	unsigned int x;

	for (n = 0; n < npatterns; n++) {
		printf("UNKNOWN %d\n", n);
		for (y = 0; y < GLYPH_HEIGHT; y++) {
			unsigned char bits;

			bits = pattern[n][y];
			for (x = 0; x < GLYPH_WIDTH; x++) {
				if (bits & 0x80) {
					printf("@");
				} else {
					printf(".");
				}
				bits <<= 1;
			}
			printf("\n");
		}
		printf("\n");
	}
}

static void
usage(int ret_val) __attribute__((__noreturn__));
static void
usage(int ret_val)
{
	fprintf(stderr, "Usage: %s [-h] <screen0> ...\n", progname);
	exit(ret_val);
}

int
main(int argc, char **argv)
{
	int c;

	progname = *argv;

	while ((c = getopt(argc, argv, "")) != -1) {
		switch (c) {
		case 'h':
			usage(0);
			/*NOTREACHED*/
		default:
			usage(1);
			/*NOTREACHED*/
		}
	}
	argc -= optind;
	argv += optind;

	if (0 < argc) {
		screenshots = argv;
		argv += argc;
		argc -= argc;
	} else {
		usage(1);
		/*NOTREACHED*/
	}

	while (*screenshots != (char *) 0) {
		screen_read(*screenshots);
		screen_split();
		screenshots++;
	}
	screen_out();

	return 0;
}
