/*
 * $Id: glue-audio_alsa1x.c,v 1.12 2009-02-25 10:46:08 potyra Exp $
 *
 * ALSA-1.x output plugin for FAUmachine
 *
 * Derived from MPlayer (libao2 - ao_alsa1x),
 * originally written by:
 *
 * Copyright (C) FAUmachine Team.
 * Copyright (C) Alex Beregszaszi.
 * Copyright (C) MPlayer Team.
 *
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 *
 * modified for real alsa-0.9.0-support by Joy Winter <joy@pingfm.org>
 * additional AC3 passthrough support by Andy Lo A Foe <andy@alsaplayer.org>  
 * 08/22/2002 iec958-init rewritten and merged with common init, joy
 *
 * Trivial port to ALSA 1.x API by Jindrich Makovicka
 *
 * Any bugreports regarding to this driver are welcome.
 */

#include <errno.h>
#include <sys/time.h>
#include <stdlib.h>
#include <math.h>
#include <sys/poll.h>
#include <ctype.h>
#include <dlfcn.h>

#include "config.h"

#if HAVE_ALSA_ASOUNDLIB_H
#include <alsa/asoundlib.h>
#elif HAVE_SYS_ASOUNDLIB_H
#include <sys/asoundlib.h>
#else
#error "asoundlib.h is not in sys/ or alsa/ - please bugreport"
#endif

#include "glue-audio_internal.h"

static ao_info_t info = 
{
	"ALSA-1.x audio output",
	"alsa1x",
	"Alex Beregszaszi, Joy Winter <joy@pingfm.org>",
	"under developement"
};

LIBAO_EXTERN(alsa1x);

int (*s_card_next)(int *card);
int (*s_pcm_close)(snd_pcm_t *pcm);
int (*s_pcm_drop)(snd_pcm_t *pcm);
int (*s_pcm_hw_params_any)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int (*s_pcm_hw_params_get_buffer_size)(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val);
int (*s_pcm_hw_params_set_access)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_t val);
int (*s_pcm_hw_params_set_channels)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val);
int (*s_pcm_hw_params_set_format)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t val);
int (*s_pcm_hw_params_set_period_size_near)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir);
int (*s_pcm_hw_params_set_periods_near)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int (*s_pcm_hw_params_set_rate_near)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int (*s_pcm_hw_params)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int (*s_pcm_hw_params_test_format)(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t val);
int (*s_pcm_hw_params_malloc)(snd_pcm_hw_params_t **ptr);
void (*s_pcm_hw_params_free)(snd_pcm_hw_params_t *obj);
int (*s_pcm_info_malloc)(snd_pcm_info_t **ptr);
int (*s_pcm_nonblock)(snd_pcm_t *pcm, int nonblock);
int (*s_pcm_open)(snd_pcm_t **pcm, const char *name, snd_pcm_stream_t stream, int mode);
int (*s_pcm_prepare)(snd_pcm_t *pcm);
int (*s_pcm_resume)(snd_pcm_t *pcm);
int (*s_pcm_status_malloc)(snd_pcm_status_t **ptr);
int (*s_pcm_status)(snd_pcm_t *pcm, snd_pcm_status_t *status);
int (*s_pcm_wait)(snd_pcm_t *pcm, int timeout);
snd_pcm_sframes_t (*s_pcm_avail_update)(snd_pcm_t *pcm);
snd_pcm_sframes_t (*s_pcm_status_get_delay)(const snd_pcm_status_t *obj);
snd_pcm_sframes_t (*s_pcm_writei)(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size);
snd_pcm_state_t (*s_pcm_status_get_state)(const snd_pcm_status_t *obj);
snd_pcm_uframes_t (*s_pcm_status_get_avail)(const snd_pcm_status_t *obj);
unsigned int (*s_pcm_info_get_device)(const snd_pcm_info_t *obj);
unsigned int (*s_pcm_info_get_subdevice)(const snd_pcm_info_t *obj);
void (*s_pcm_info_free)(snd_pcm_info_t *obj);
void (*s_pcm_status_free)(snd_pcm_status_t *obj);
void (*s_pcm_status_get_trigger_tstamp)(const snd_pcm_status_t *obj, snd_timestamp_t *ptr);
size_t (*s_pcm_status_sizeof)(void);
size_t (*s_pcm_hw_params_sizeof)(void);
size_t (*s_pcm_sw_params_sizeof)(void);

static struct {
	const char * name;
	void * funcpp;
} alsa_array[] = {
	{ "snd_card_next", &s_card_next },
	{ "snd_pcm_avail_update", &s_pcm_avail_update },
	{ "snd_pcm_close", &s_pcm_close },
	{ "snd_pcm_drop", &s_pcm_drop },
	{ "snd_pcm_hw_params", &s_pcm_hw_params },
	{ "snd_pcm_hw_params_any", &s_pcm_hw_params_any },
	{ "snd_pcm_hw_params_get_buffer_size", &s_pcm_hw_params_get_buffer_size },
	{ "snd_pcm_hw_params_set_access", &s_pcm_hw_params_set_access },
	{ "snd_pcm_hw_params_set_channels", &s_pcm_hw_params_set_channels },
	{ "snd_pcm_hw_params_set_format", &s_pcm_hw_params_set_format },
	{ "snd_pcm_hw_params_set_period_size_near", &s_pcm_hw_params_set_period_size_near },
	{ "snd_pcm_hw_params_set_periods_near", &s_pcm_hw_params_set_periods_near },
	{ "snd_pcm_hw_params_set_rate_near", &s_pcm_hw_params_set_rate_near },
	{ "snd_pcm_hw_params_test_format", &s_pcm_hw_params_test_format },
	{ "snd_pcm_hw_params_malloc", &s_pcm_hw_params_malloc },
	{ "snd_pcm_hw_params_free", &s_pcm_hw_params_free },
	{ "snd_pcm_info_free", &s_pcm_info_free },
	{ "snd_pcm_info_get_device", &s_pcm_info_get_device },
	{ "snd_pcm_info_get_subdevice", &s_pcm_info_get_subdevice },
	{ "snd_pcm_info_malloc", &s_pcm_info_malloc },
	{ "snd_pcm_nonblock", &s_pcm_nonblock },
	{ "snd_pcm_open", &s_pcm_open },
	{ "snd_pcm_prepare", &s_pcm_prepare },
	{ "snd_pcm_resume", &s_pcm_resume },
	{ "snd_pcm_status", &s_pcm_status },
	{ "snd_pcm_status_free", &s_pcm_status_free },
	{ "snd_pcm_status_get_avail", &s_pcm_status_get_avail },
	{ "snd_pcm_status_get_delay", &s_pcm_status_get_delay },
	{ "snd_pcm_status_get_state", &s_pcm_status_get_state },
	{ "snd_pcm_status_get_trigger_tstamp", &s_pcm_status_get_trigger_tstamp },
	{ "snd_pcm_status_malloc", &s_pcm_status_malloc },
	{ "snd_pcm_wait", &s_pcm_wait },
	{ "snd_pcm_writei", &s_pcm_writei },
	{ "snd_pcm_status_sizeof", &s_pcm_status_sizeof },
        { "snd_pcm_sw_params_sizeof", &s_pcm_sw_params_sizeof },
        { "snd_pcm_hw_params_sizeof", &s_pcm_hw_params_sizeof },
	{ NULL, NULL }
};

static ao_data_t ao_data={0,0,0,0,OUTBURST,-1,0};

static void * alsa_lib_handle = NULL;

static snd_pcm_t *alsa_handler;
static snd_pcm_format_t alsa_format;

static int alsa_fragcount = 16;
static snd_pcm_uframes_t chunk_size = 1024; //is alsa_fragsize / 4

static size_t bytes_per_sample;

/* possible 4096, original 8192 */
#define MIN_CHUNK_SIZE 1024
#define ALSA_DEVICE_SIZE 48

/* open & setup audio device
 * return: 1=success 0=fail
 */
static int
init(void)
{
	int err;
	int cards = -1;
	int dir, i;
	snd_pcm_uframes_t bufsize;
	snd_pcm_info_t *alsa_info;
	snd_pcm_hw_params_t *alsa_hwparams;
	char *alsa_device;
	int tmp_device, tmp_subdevice;

	alsa_lib_handle = dlopen("libasound.so", RTLD_NOW);
	if(alsa_lib_handle == NULL) {
		return 0;
	}

	for(i=0; alsa_array[i].name; i++) {
		dlerror();
		*(void**) alsa_array[i].funcpp = dlsym(alsa_lib_handle,alsa_array[i].name);
		if(dlerror() != NULL) {
			dlclose(alsa_lib_handle);
			return 0;
		}
	}

	alsa_handler = NULL;

	if ((err = s_card_next(&cards)) < 0 || cards < 0)
	{
		return 0;
	}

	ao_data.samplerate = GLUE_AUDIO_RATE;
	ao_data.bps = GLUE_AUDIO_CHANNELS * GLUE_AUDIO_RATE;
	ao_data.format = GLUE_AUDIO_FORMAT;
	ao_data.channels = GLUE_AUDIO_CHANNELS;
	ao_data.outburst = OUTBURST;

#if GLUE_AUDIO_FORMAT == AFMT_S16_BE
	alsa_format = SND_PCM_FORMAT_S16_BE;
#elif GLUE_AUDIO_FORMAT == AFMT_S16_LE
	alsa_format = SND_PCM_FORMAT_S16_LE;
#else
#error audio format not supported
#endif

	ao_data.bps *= GLUE_AUDIO_FORMAT_BYTES;

	if(s_pcm_info_malloc(&alsa_info) < 0) {
		return 0;
	}

	if((tmp_device = s_pcm_info_get_device(alsa_info)) < 0) {
		s_pcm_info_free(alsa_info);
		return 0;
	}

	if((tmp_subdevice = s_pcm_info_get_subdevice(alsa_info)) < 0) {
		s_pcm_info_free(alsa_info);
		return 0;
	}

	if((alsa_device = (char*) malloc(ALSA_DEVICE_SIZE)) == NULL) {
		s_pcm_info_free(alsa_info);
		return 0;
	}

	s_pcm_info_free(alsa_info);

	snprintf(alsa_device, ALSA_DEVICE_SIZE, "hw:%1d,%1d", tmp_device, tmp_subdevice);

	if(!alsa_handler) {
		if(s_pcm_open(&alsa_handler, alsa_device, SND_PCM_STREAM_PLAYBACK, 0) < 0) {
			free(alsa_device);
			return 0;
		}

		s_pcm_nonblock(alsa_handler, 0);

		s_pcm_hw_params_malloc(&alsa_hwparams);

		/* setting hw-parameters */
		if(s_pcm_hw_params_any(alsa_handler, alsa_hwparams) < 0) {
			goto bail_out;
		}

		err = s_pcm_hw_params_set_access(alsa_handler, alsa_hwparams,SND_PCM_ACCESS_RW_INTERLEAVED);
		if(err < 0) {
			goto bail_out;
		}

		/* workaround for nonsupported formats
		 * sets default format to S16_LE if the given formats aren't supported
		 */
		if(s_pcm_hw_params_test_format(alsa_handler, alsa_hwparams, alsa_format) < 0) {
			alsa_format = SND_PCM_FORMAT_S16_LE;
			ao_data.format = AFMT_S16_LE;
			ao_data.bps = GLUE_AUDIO_CHANNELS * GLUE_AUDIO_RATE * 2;
		}

		bytes_per_sample = ao_data.bps / ao_data.samplerate; /* it should be here */

		if(s_pcm_hw_params_set_format(alsa_handler, alsa_hwparams, alsa_format) < 0) {
			goto bail_out;
		}

		if(s_pcm_hw_params_set_channels(alsa_handler, alsa_hwparams, ao_data.channels) < 0) {
			goto bail_out;
		}

		dir = 0;
		if(s_pcm_hw_params_set_rate_near(alsa_handler, alsa_hwparams, &ao_data.samplerate, &dir) < 0) {
			goto bail_out;
		}

		/* set chunksize */
		dir = 0;
		if(s_pcm_hw_params_set_period_size_near(alsa_handler, alsa_hwparams, &chunk_size, &dir) < 0) {
			goto bail_out;
		}

		dir = 0;
		s_pcm_hw_params_set_periods_near(alsa_handler, alsa_hwparams, &alsa_fragcount, &dir);

		/* finally install hardware parameters */
		if(s_pcm_hw_params(alsa_handler, alsa_hwparams) < 0) {
			goto bail_out;
		}

		/* gets buffersize for control */
		if(s_pcm_hw_params_get_buffer_size(alsa_hwparams, &bufsize) < 0) {
			goto bail_out;
		} else {
			ao_data.buffersize = bufsize * bytes_per_sample;
		}

		s_pcm_hw_params_free(alsa_hwparams);

		if(s_pcm_prepare(alsa_handler) < 0) {
			return 0;
		}
	}

	free(alsa_device);

	return 1;

bail_out:
	s_pcm_hw_params_free(alsa_hwparams);
	return 0;
}

/* close audio device */
static void
uninit(int immed)
{
	if (alsa_handler) {
		if (s_pcm_drop(alsa_handler) < 0) {
			return;
		}

		if (s_pcm_close(alsa_handler) < 0)
		{
			return;
		} else {
			alsa_handler = NULL;
		}
	}
}

#ifndef timersub
#define timersub(a, b, result) \
	do { \
		(result)->tv_sec = (a)->tv_sec - (b)->tv_sec; \
		(result)->tv_usec = (a)->tv_usec - (b)->tv_usec; \
		if ((result)->tv_usec < 0) { \
			--(result)->tv_sec; \
			(result)->tv_usec += 1000000; \
		} \
	} while (0)
#endif

/* I/O error handler */
static int
xrun()
{
	snd_pcm_status_t *status;

	if(s_pcm_status_malloc(&status) < 0) {
		return 0;
	}

	if(s_pcm_status(alsa_handler, status) < 0) {
		s_pcm_status_free(status);
		return 0;
	}

	if(s_pcm_status_get_state(status) == SND_PCM_STATE_XRUN) {
		struct timeval now, diff, tstamp;
		gettimeofday(&now, 0);
		s_pcm_status_get_trigger_tstamp(status, &tstamp);
		timersub(&now, &tstamp, &diff);
	}

	s_pcm_status_free(status);

	if(s_pcm_prepare(alsa_handler) < 0) {
		return 0;
	}

	return 1; /* ok, data should be accepted again */
}

/* plays 'len' bytes of 'data'
 * returns: number of bytes played
 * modified last at 29.06.02 by jp
 * thanxs for marius <marius@rospot.com> for giving us the light ;)
 */
static int
play(void* data, int len)
{

	//bytes_per_sample is always 4 for 2 chn S16_LE
	int num_frames = len / bytes_per_sample;
	char *output_samples = (char *) data;
	snd_pcm_sframes_t res = 0;

	if (!alsa_handler) {
		return 0;
	}

	while (num_frames > 0) {

		res = s_pcm_writei(alsa_handler, (void *)output_samples, num_frames);

		if (res == -EAGAIN) {
			s_pcm_wait(alsa_handler, 1000);
		}
		else if (res == -EPIPE) {  /* underrun */
			if (xrun("play") <= 0) {
				return 0;
			}
		}
		else if (res == -ESTRPIPE) {	/* suspend */
			while ((res = s_pcm_resume(alsa_handler)) == -EAGAIN)
				sleep(1);
		}
		else if (res < 0) {
			if ((res = s_pcm_prepare(alsa_handler)) < 0) {
				return 0;
				break;
			}
		}

		if (res > 0) {
			output_samples += res * bytes_per_sample;
			num_frames -= res;
		}
	}

	if (res < 0) {
		return 0;
	}
	return res < 0 ? (int)res : len - len % bytes_per_sample;
}

/* how many byes are free in the buffer */
static int
get_space()
{
	snd_pcm_status_t *status;
	int ret=0;

	if(s_pcm_status_malloc(&status) < 0) {
		return 0;
	}

	if(s_pcm_status(alsa_handler, status) < 0) {
		s_pcm_status_free(status);
		return 0;
	}

	switch(s_pcm_status_get_state(status)) {
		case SND_PCM_STATE_OPEN:
		case SND_PCM_STATE_PREPARED:
		case SND_PCM_STATE_RUNNING:
			ret = s_pcm_status_get_avail(status) * bytes_per_sample;
			break;
		case SND_PCM_STATE_PAUSED:
			ret = 0;
			break;
		case SND_PCM_STATE_XRUN:
			xrun();
			ret = 0;
			break;
		default:
			ret = s_pcm_status_get_avail(status) * bytes_per_sample;
			if (ret <= 0) {
				xrun();
			}
	}

	s_pcm_status_free(status);

	/* workaround for too small value returned */
	if(ret < MIN_CHUNK_SIZE)
		ret = 0;

	return ret;
}

/* delay in seconds between first and last sample in buffer */
static float
get_delay()
{
	snd_pcm_status_t *status;
	float ret = 0;

	if(alsa_handler) {

		if(s_pcm_status_malloc(&status) < 0) {
			return 0;
		}

		if(s_pcm_status(alsa_handler, status) < 0) {
			s_pcm_status_free(status);
			return 0;
		}

		switch(s_pcm_status_get_state(status)) {
			case SND_PCM_STATE_OPEN:
			case SND_PCM_STATE_PREPARED:
			case SND_PCM_STATE_RUNNING:
				ret = (float)s_pcm_status_get_delay(status)/(float)ao_data.samplerate;
				break;
			default:
				break;
		}

		s_pcm_status_free(status);

		if (ret < 0)
			ret = 0;
	}

	return ret;
}
