# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import re
import sys
import shutil
import signal
import gettext
import subprocess

import debconf
import parted
import _ped

from ubiquity.plugin import *
from ubiquity import parted_server
from ubiquity.misc import *
from ubiquity import osextras

gettext.bindtextdomain('fastboot-config')
gettext.textdomain('fastboot-config')

f = open('/proc/cmdline','r')
kernelcmdline = f.read()
if "webnow-recovery" in kernelcmdline:
    NAME='omsk-recovery'
    BEFORE='partman'
    WEIGHT=11

# Import locale
# References:
#  http://msdn.microsoft.com/en-us/goglobal/bb896001.aspx
#  /usr/share/i18n/SUPPORTED
#  /usr/sbin/omsk-import-settings

def regget(path, key):
    regline = subprocess.Popen(["reglookup", "-H", "-p", key, path], stdout=subprocess.PIPE).communicate()[0]
    if not regline:
        return None
    fields = regline.split(',')
    if len(fields) < 3:
        return None
    return fields[2]

def issupportedlocale(locale):
    issupported = False
    try:
        f = open('/usr/share/i18n/SUPPORTED')
        text = f.read()
        f.close()
        if re.search('^%s\\s' % re.escape(locale), text, re.MULTILINE):
            issupported = True
    except:
        pass
    return issupported

def getfallbacklocale(lang, country):
    try:
        f = open('/usr/share/localechooser/languagelist')
        text = f.read()
        f.close()
        match = re.search('^%s_%s;.*;(.*);' % (lang, country), text, re.MULTILINE)
        if not match:
            match = re.search('^%s;.*;(.*);' % lang, text, re.MULTILINE)
        if not match:
            f = open('/usr/share/i18n/SUPPORTED')
            text = f.read()
            f.close()
            match = re.search('^%s_?(.*)\\.UTF-8 ' % lang, text, re.MULTILINE)
        if not match:
            return 'en_US.UTF-8'
        elif match.group(1):
            fallback = '%s_%s.UTF-8' % (lang, match.group(1))
        else:
            fallback = '%s.UTF-8' % lang
        if issupportedlocale(fallback):
            return fallback
        else:
            return 'en_US.UTF-8'
    except:
        return 'en_US.UTF-8'

def getLocale(windir):
    winuser = subprocess.Popen(['sh', '-c', '. %s/webnow.sh; echo -n $WEBNOW_WINDOWS_USER' % windir], stdout=subprocess.PIPE).communicate()[0].strip()
    winlocale = regget('%s/Users/%s/NTUSER.DAT' % (windir, winuser), '/Control Panel/Desktop/PreferredUILanguages')
    if winlocale:
       winlocale = winlocale.split('|')[0]
    else:
       winlocale = regget('%s/Users/%s/NTUSER.DAT' % (windir, winuser), '/Control Panel/International/LocaleName')
    ubulocale = None
    if winlocale:
        winlocalesplit = winlocale.split('-')
        ll = winlocalesplit[0]
        if len(winlocalesplit) > 1:
            cc = winlocalesplit[-1]
        else:
            cc = None
        ubulocale = ll + ('_' + cc if cc else '') + '.UTF-8'
        
        # Now check if locale is supported in Ubuntu
        issupported = issupportedlocale(ubulocale)
        if not issupported:
            ubulocale = getfallbacklocale(ll, cc)

        return ubulocale
    else:
        return 'en_US.UTF-8'

def importLang(frontend):
    '''Tries to grab the language from the Windows partition.'''
    with raised_privileges():
        dev = parted.Device("/dev/sda")
        disk = parted.Disk(dev)

        ntfs_partition = None
        for i in disk.getPrimaryPartitions():
            if (i.fileSystem and i.fileSystem.type.startswith("ntfs") and i.getSize() > 7*1024):
                ntfs_partition = i
        if not ntfs_partition:
            return

        if os.system('sudo mount /dev/%s /mnt' % ntfs_partition.getDeviceNodeName()):
            return
        try:
            locale = getLocale('/mnt')
            os.system('sudo umount /mnt')
        except:
            os.system('sudo umount /mnt')
            return

        frontend.db.set('debian-installer/locale', locale)
        frontend.locale = locale
        os.environ['LANG'] = locale
        os.environ['LANGUAGE'] = locale
        gettext.install('fastboot-config')

class PageGtk (PluginUI):
    plugin_prefix = 'fastboot-config'

    def __init__(self, controller, *args, **kwargs):
        importLang(controller._wizard)
        import gtk
        ui_file="omsk-recovery.ui"
        self.controller = controller
        builder = gtk.Builder()
        builder.set_translation_domain('fastboot-config')
        builder.add_from_file('/usr/share/ubiquity/gtk/%s' % ui_file)
        builder.connect_signals(self)
        self.controller.add_builder(builder)
        self.page = builder.get_object('stepOMSKRecovery')
        self.label = builder.get_object('label_message')
        self.forward = builder.get_object('omsk_forward')
        self.reboot_warning = builder.get_object('reboot_warning')
        self.page.set_name("omsk-recovery")
        self.plugin_widgets = self.page
        self.error_dialog = builder.get_object('error_dialog1')

    def setMessage(self, text, isError = False):
        self.label.set_label(text)
        if isError:
            self.reboot_warning.hide()
        else:
            self.reboot_warning.show()

    def on_button1_clicked(self, widget):
        self.controller.go_forward()

    def on_button2_clicked(self, widget):
        with raised_privileges():
            os.system('sudo shutdown -r now')

    def on_button3_clicked(self, widget):
        with raised_privileges():
            os.system('sudo shutdown -r now')

    def hideButtons(self):
        self.controller.toggle_navigation_control(False)

    def setAllowForward(self, allowed):
        self.controller.allow_go_forward(allowed)
        self.forward.set_sensitive(allowed)

def check_vendor():
    if os.path.exists('/sys/class/dmi/id/bios_vendor'):
        with open('/sys/class/dmi/id/bios_vendor') as file:
            vendor = file.readline().split()[0].lower()
    else:
        vendor = ''
    return (vendor == 'dell' or vendor == 'innotek')

class Page(Plugin):

    def prepare(self, unfiltered=False):
        self.ui.hideButtons()
        self.actionC = "undecide"
        self.errorText = _("Ubuntu Light requires 5 GB of unpartitioned disk space, but there is not enough available space.  Recovery cannot continue.")
        self.expert_recipe_str=""
        self.replace_partition = 0
        self.ntfs_partition = 0
        self.num_of_partitions = 0
        self.freeMB = 0
        self.required_space = 5*1000
        with raised_privileges():
            self.dev = parted.Device("/dev/sda")
            self.disk = parted.Disk(self.dev)

            self.num_of_partitions = self.disk.primaryPartitionCount
            
            if not check_vendor():
                self.actionC = "error"
                self.errorText = _("This is not a supported machine.  Recovery cannot continue.");

            if (self.actionC=="undecide") and (self.disk.type == "gpt"):
                self.actionC = "error"
                self.errorText = _("GUID partition table has been detected. This is not supported. Recovery cannot continue.");

            if (self.actionC=="undecide"):
                for i in self.disk.getPrimaryPartitions():
                    if (i.fileSystem and i.fileSystem.type.startswith("ext") and i.getSize()>3*1024):
                        self.replace_partition = i
                        self.actionC = "replace"
                        break
            
            if (self.actionC=="undecide"):
                freeSpaceList = self.disk.getFreeSpaceRegions()
                if (freeSpaceList != 0):
                    for i in freeSpaceList:
                        if (i.getSize() > self.freeMB):
                            self.freeMB = i.getSize()
                    if (self.freeMB >= self.required_space and self.num_of_partitions <= 3):
                        self.actionC = "create"
            
            if (self.actionC=="undecide"):
                for i in self.disk.getPrimaryPartitions():
                    if (i.fileSystem and i.fileSystem.type.startswith("ntfs") and i.getSize() > 7*1024):
                        self.ntfs_partition = i
                        if (self.num_of_partitions <= 3):
                            self.actionC = "shrink"
            
            if (self.actionC == "shrink"):
                self.ui.setMessage(_('Windows will be resized to make room for Ubuntu Light.'))
            elif (self.actionC == "create"):
                self.ui.setMessage(_('Ubuntu Light will be installed into the available disk space.'))
            elif (self.actionC == "replace"):
                self.ui.setMessage(_('Ubuntu Light will be reset to factory defaults.  Any personal files or Web history you have in Ubuntu Light will be deleted.  However, your personal files from Windows will not be affected.'))
            else:
                self.actionC = "error"
                # What went wrong?
                if self.num_of_partitions >= 4:
                    self.errorText = _("There are already four or more partitions.  Recovery cannot continue.  Please make sure there are no more than three partitions and restart recovery.");

            if (self.actionC == "error"):
                self.ui.setMessage(self.errorText, isError=True)
                self.ui.setAllowForward(False)
            
        return Plugin.prepare(self, unfiltered=unfiltered);

    def ok_handler(self):
        if (self.actionC == "replace" and self.replace_partition != 0):
            deviceName = self.replace_partition.getDeviceNodeName()
            with raised_privileges():
                self.disk.deletePartition(self.replace_partition)
                self.disk.commit()
            self.preseed_bool('partman-basicfilesystems/no_swap',False)
            self.preseed_bool('grub-installer/only_debian',False)
            self.preseed_bool('grub-installer/with_other_os',False)
            if (deviceName != 0):
                self.preseed('grub-installer/bootdev','/dev/'+deviceName)
            self.preseed('partman-auto/automatically_partition','biggest_free')
            self.preseed('partman-auto/init_automatically_partition','biggest_free')
            self.preseed('partman-auto/disk','/dev/sda')
            self.expert_recipe_str = "boot-root :: "
            self.expert_recipe_str += "2000 5000 -1 ext4 "
            self.expert_recipe_str += '$primary{ } '
            self.expert_recipe_str += "method{ format } format{ } use_filesystem{ } filesystem{ ext4 } mountpoint{ / } ."
            self.preseed('partman-auto/expert_recipe',self.expert_recipe_str)
            self.preseed('partman-auto/choose_recipe','boot-root')
        elif (self.actionC == "shrink" and self.ntfs_partition != 0):
            deviceName = self.ntfs_partition.getDeviceNodeName()
            with raised_privileges():
                cmd = "ntfsresize --size "+str(int(self.ntfs_partition.getSize()*1024*1024/1000/1000-self.required_space-200))+"M"+self.ntfs_partition.path
                res = subprocess.Popen(
                          cmd,
                          shell=True,
                          stderr=subprocess.PIPE
                          ).communicate()[0]
                if res != None:
                    self.ui.setMessage(self.errorText, isError=True)
                    self.ui.setAllowForward(False)
                    return
                # Windows 7 is not cylinder aligned, so turn off that flag
                prevFlag = self.disk.getFlag(parted.DISK_CYLINDER_ALIGNMENT)
                if prevFlag and not self.disk.unsetFlag(parted.DISK_CYLINDER_ALIGNMENT):
                    self.ui.setMessage(self.errorText, isError=True)
                    self.ui.setAllowForward(False)
                    return
                self.disk.setPartitionGeometry(self.ntfs_partition, self.dev.getConstraint(), self.ntfs_partition.geometry.start, self.dev.endCylinderToSector(self.dev.endSectorToCylinder(self.ntfs_partition.geometry.end-self.required_space*1000*1000/self.dev.sectorSize)))
                if prevFlag:
                    self.disk.setFlag(parted.DISK_CYLINDER_ALIGNMENT)
                self.disk.commit()
            self.preseed_bool('partman-basicfilesystems/no_swap',False)
            self.preseed_bool('grub-installer/only_debian',False)
            self.preseed_bool('grub-installer/with_other_os',False)

            self.preseed('grub-installer/bootdev','/dev/sda%d' % (int(deviceName[3:])+1))
            self.preseed('partman-auto/automatically_partition','biggest_free')
            self.preseed('partman-auto/init_automatically_partition','biggest_free')
            self.preseed('partman-auto/disk','/dev/sda')
            self.expert_recipe_str = "boot-root :: "
            self.expert_recipe_str += "2000 5000 -1 ext4 "
            self.expert_recipe_str += '$primary{ } '
            self.expert_recipe_str += "method{ format } format{ } use_filesystem{ } filesystem{ ext4 } mountpoint{ / } ."
            self.preseed('partman-auto/expert_recipe',self.expert_recipe_str)
            self.preseed('partman-auto/choose_recipe','boot-root')
        elif (self.actionC == "create"):
            self.preseed_bool('partman-basicfilesystems/no_swap',False)
            self.preseed_bool('grub-installer/only_debian',False)
            self.preseed_bool('grub-installer/with_other_os',False)

            self.preseed('grub-installer/bootdev','/dev/sda%d' % (self.num_of_partitions+1))
            self.preseed('partman-auto/init_automatically_partition','biggest_free')
            self.preseed('partman-auto/automatically_partition','biggest_free')
            self.preseed('partman-auto/disk','/dev/sda')
            self.expert_recipe_str = "boot-root :: "
            self.expert_recipe_str += "2000 5000 -1 ext4 "
            self.expert_recipe_str += '$primary{ } '
            self.expert_recipe_str += "method{ format } format{ } use_filesystem{ } filesystem{ ext4 } mountpoint{ / } ."
            self.preseed('partman-auto/expert_recipe',self.expert_recipe_str)
            self.preseed('partman-auto/choose_recipe','boot-root')
            
        Plugin.ok_handler(self)

class Install(InstallPlugin):
    def install (self,target,progress,*args,**kwargs):
        return InstallPlugin.install(self, target, progress, *args, **kwargs)
