#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
#
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
#
# Author: Enthought, Inc.
# Description: <Enthought pyface package component>
#------------------------------------------------------------------------------
""" A simple 'About' dialog. """


# Standard library imports.
import sys

# Major package imports.
import wx
import wx.html
import wx.lib.wxpTag

# Enthought library imports.
from enthought.traits.api import Instance, List, String

# Local imports.
from dialog import Dialog
from image_resource import ImageResource


class AboutDialog(Dialog):
    """ A simple 'About' dialog. """

    text = '''
<html>
  <body>
    <center>
      <table width="100%%" cellspacing="4" cellpadding="0" border="0">
        <tr>
          <td align="center">
          <p>
            <img src="%s" alt="">
          </td>
        </tr>
      </table>

      <p>
      %s<br>
      <br>
      Python %s<br>
      wxPython %s<br>
      </p>
      <p>
      Copyright &copy; 2003-2007 Enthought, Inc.
      </p>

      <p>
        <wxp module="wx" class="Button">
          <param name="label" value="OK">
          <param name="id"    value="ID_OK">
        </wxp>
      </p>
  </center>
  </body>
</html>
'''
    #### 'AboutDialog' interface ##############################################

    # The image displayed in the dialog.
    image = Instance(ImageResource, ImageResource('about'))

    # Additional strings to be added to the dialog.
    additions = List(String)

    ###########################################################################
    # Protected 'Window' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Creates the window contents. """

        if parent.GetParent() is not None:
            title = parent.GetParent().GetTitle()

        else:
            title = ""

        # Set the title.
        self.title = "About %s" % title

        # Load the image to be displayed in the about box.
        image = self.image.create_image()
        path  = self.image.absolute_path

        # The additional strings.
        additions = '<br />'.join(self.additions)

        # The width of a wx HTML window is fixed (and  is given in the
        # constructor). We set it to the width of the image plus a fudge
        # factor! The height of the window depends on the content.
        width = image.GetWidth() + 80
        html = wx.html.HtmlWindow(parent, -1, size=(width, -1))

        # Get the version of Python we are running on.
        # ...remove the more detailed info (build date, platform, etc.)
        # ...this may be needed in the future though
        py_version = sys.version[0:sys.version.find( "(" )]

        # Get the version of wx.
        wx_version = wx.VERSION_STRING

        # Set the page contents.
        html.SetPage(
            self.text % (path, additions, py_version, wx_version)
        )

        # Make the 'OK' button the default button.
        ok_button = html.FindWindowById(wx.ID_OK)
        ok_button.SetDefault()

        # Set the height of the HTML window to match the height of the content.
        internal = html.GetInternalRepresentation()
        html.SetSize((-1, internal.GetHeight()))

        # Make the dialog client area big enough to display the HTML window.
        # We add a fudge factor to the height here, although I'm not sure why
        # it should be necessary, the HTML window should report its required
        # size!?!
        width, height = html.GetSize()
        parent.SetClientSize((width, height + 10))
        parent.CentreOnParent(wx.BOTH)

        return html


# So that you can test the dialog without having to run an application.
if __name__ == '__main__':
    app = wx.PySimpleApp()
    dialog = AboutDialog()
    dialog.open()

### EOF #######################################################################
