# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL.EXCEPTION" in the root of this distribution
# including a special exception to use Elisa with Fluendo's plugins and
# about affiliation parameters.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>

from elisa.core.media_uri import MediaUri
from elisa.core import common
from elisa.core.utils import defer
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.yesfm.controllers import BrowseAlbumAction, \
                                            EnterTrackAction

from pgm.timing import implicit


_ = install_translation('yesfm')


class NoAvailableCompletionError(Exception):
    pass

class CompleteAlbumsForArtist(Action):
    toggled_title = _("Remove Yes.fm Albums")
    untoggled_title = _("Add Yes.fm Albums")
    toggled_icon = "elisa.plugins.yesfm.uncomplete_album"
    untoggled_icon = "elisa.plugins.yesfm.complete_album"

    toggled = False
    title = untoggled_title
    icon = untoggled_icon

    def __init__(self, controller):
        super(CompleteAlbumsForArtist, self).__init__(controller)
        self.completion_available = True

    def refresh(self):
        # FIXME: hackish refresh
        try:
            index = self.controller.actions.index(self)
            self.controller.actions[index] = self
        except ValueError, e:
            pass

    def retrieve_albums(self, artist):
        return artist.albums

    def insert_albums_for_artist(self, albums):
        # keep track of what albums were added
        self.added_albums = []
        # create a copy of the controller's model where yes.fm's albums will
        # be inserted
        updated_model = []
        updated_model.extend(self.controller.model)
        for album in albums:
            # FIXME: 'action' not defined in the API but used by
            # database.music_controller.GenericAlbumsDbController; that
            # mechanism could be extended thus making Elisa a lot more
            # pluggable
            album.action = BrowseAlbumAction
            if insert_sorted(updated_model, album, album_matcher,
                             compare_albums):
                self.added_albums.append(album)

        return updated_model

    def remove_added_albums(self, result):
        updated_model = []
        updated_model.extend(self.controller.model)
        for album in self.added_albums:
            updated_model.remove(album)

        return updated_model

    def check_for_new_albums(self, updated_model):
        if len(updated_model) == len(self.controller.model):
            raise NoAvailableCompletionError()
        else:
            return updated_model

    def no_completion_available(self, failure):
        failure.trap(NoAvailableCompletionError)

        self.completion_available = False
        self.title = _("No Yes.fm Albums")
        self.icon = "elisa.plugins.yesfm.no_album_available"
        self.refresh()
        return failure

    def fade_out(self, result):
        dfr = defer.Deferred()
        animated = self.controller.widget_animated

        def faded_out(ignored):
            animated.setup_next_animations(end_callback=None)
            dfr.callback(result)

        animated.stop_animations()
        animated.setup_next_animations(transformation=implicit.DECELERATE,
                                       duration=300, end_callback=faded_out)
        animated.opacity = 0
        return dfr

    def update_model(self, updated_model):
        self.controller.model[0:] = updated_model

    def fade_in(self, result):
        self.controller.widget_animated.opacity = 255
        return result

    def toggle(self, result):
        self.toggled = not self.toggled
        if self.toggled:
            self.title = self.toggled_title
            self.icon = self.toggled_icon
        else:
            self.title = self.untoggled_title
            self.icon = self.untoggled_icon
        self.refresh()

    def stop_loading_animation(self, result):
        # FIXME: stopping the loading animation is done here but clearly the
        # design is wrong: ListController.stop_loading_animation should not
        # exist; instead the deferred returned by the actions should be used to
        # stop the feedback at the right moment.
        self.controller.stop_loading_animation()
        return result

    def run(self):
        if not self.completion_available:
            dfr = defer.fail(NoAvailableCompletionError())
        else:
            if not self.toggled:
                artist_name = self.controller.artist.name
                # Yes.fm's API specifies that spaces have to be replaced with
                # dashes
                artist_name = artist_name.replace(" ", "-")
                uri = MediaUri("http://api.yes.fm/v1/artist/%s" % artist_name)
                model, dfr = common.application.resource_manager.get(uri)
                dfr.addCallback(self.retrieve_albums)
                dfr.addCallback(self.insert_albums_for_artist)
                dfr.addCallback(self.check_for_new_albums)
                dfr.addErrback(self.no_completion_available)
            else:
                dfr = defer.succeed(None)
                dfr.addCallback(self.remove_added_albums)

            dfr.addCallback(self.fade_out)
            dfr.addCallback(self.update_model)
            dfr.addCallback(self.fade_in)
            dfr.addCallback(self.toggle)

        dfr.addBoth(self.stop_loading_animation)
        return dfr

def add_action_complete_albums_for_artist(controller):
    if controller.list_controller.artist:
        action = CompleteAlbumsForArtist(controller.list_controller)
        controller.list_controller.actions.append(action)
    return defer.succeed(None)


class CompleteTracksForAlbum(Action):
    toggled_title = _("Remove Yes.fm Tracks")
    untoggled_title = _("Add Yes.fm Tracks")
    toggled_icon = "elisa.plugins.yesfm.uncomplete_album"
    untoggled_icon = "elisa.plugins.yesfm.complete_album"

    toggled = False
    title = untoggled_title
    icon = untoggled_icon

    def __init__(self, controller):
        super(CompleteTracksForAlbum, self).__init__(controller)
        self.completion_available = True

    def refresh(self):
        # FIXME: hackish refresh
        try:
            index = self.controller.actions.index(self)
            self.controller.actions[index] = self
        except ValueError, e:
            pass

    def retrieve_best_match_id(self, search_result):
        # look for the best match based on the scores returned by Yes.fm
        matching_album = None
        for album in search_result.result:
            if matching_album != None and album.score > matching_album.score or \
                album.score >= 90:
                matching_album = album

        if matching_album == None:
            raise NoAvailableCompletionError()
        else:
            return matching_album.yesfmid

    def get_album(self, album_id):
        uri = MediaUri("http://api.yes.fm/v1/album/%s" % album_id)
        model, dfr = common.application.resource_manager.get(uri)
        return dfr

    def retrieve_tracks(self, album):
        return album.tracks

    def insert_tracks_for_album(self, tracks, model):
        # keep track of what tracks were added
        self.added_tracks = []
        # create a copy of the controller's model where yes.fm's tracks will
        # be inserted
        updated_model = []
        updated_model.extend(self.controller.model)

        for track in tracks:
            # FIXME: 'action' not defined in the API but used by
            # database.music_controller.GenericAlbumsDbController; that
            # mechanism could be extended thus making Elisa a lot more
            # pluggable
            track.action = EnterTrackAction
            if insert_sorted(updated_model, track,
                             track_matcher, compare_tracks):
                self.added_tracks.append(track)

        return updated_model

    def check_for_new_tracks(self, updated_model):
        if len(updated_model) == len(self.controller.model):
            raise NoAvailableCompletionError()
        else:
            return updated_model

    def no_completion_available(self, failure):
        failure.trap(NoAvailableCompletionError)

        self.completion_available = False
        self.title = _("No More Tracks from Yes.fm")
        self.icon = "elisa.plugins.yesfm.no_album_available"
        self.refresh()
        return failure

    def remove_added_tracks(self, result):
        updated_model = []
        updated_model.extend(self.controller.model)
        for track in self.added_tracks:
            updated_model.remove(track)

        return updated_model

    def fade_out(self, result):
        dfr = defer.Deferred()
        animated = self.controller.widget_animated

        def faded_out(ignored):
            animated.setup_next_animations(end_callback=None)
            dfr.callback(result)

        animated.stop_animations()
        animated.setup_next_animations(transformation=implicit.DECELERATE,
                                       duration=300, end_callback=faded_out)
        animated.opacity = 0
        return dfr

    def update_model(self, updated_model):
        self.controller.model[0:] = updated_model

    def fade_in(self, result):
        self.controller.widget_animated.opacity = 255
        return result

    def toggle(self, result):
        self.toggled = not self.toggled
        if self.toggled:
            self.title = self.toggled_title
            self.icon = self.toggled_icon
        else:
            self.title = self.untoggled_title
            self.icon = self.untoggled_icon
        self.refresh()

    def stop_loading_animation(self, result):
        # FIXME: stopping the loading animation is done here but clearly the
        # design is wrong: ListController.stop_loading_animation should not
        # exist; instead the deferred returned by the actions should be used to
        # stop the feedback at the right moment.
        self.controller.stop_loading_animation()
        return result

    def run(self):
        if not self.completion_available:
            dfr = defer.fail(NoAvailableCompletionError())
        else:
            if not self.toggled:
                album_name = self.controller.album.name
                # Yes.fm's API specifies that spaces have to be replaced with
                # dashes
                album_name = album_name.replace(" ", "-")
                uri = MediaUri("http://api.yes.fm/v1/search/album/%s" % album_name)
                model, dfr = common.application.resource_manager.get(uri)
                dfr.addCallback(self.retrieve_best_match_id)
                dfr.addCallback(self.get_album)
                dfr.addCallback(self.retrieve_tracks)
                dfr.addCallback(self.insert_tracks_for_album,
                                self.controller.model)
                dfr.addCallback(self.check_for_new_tracks)
                dfr.addErrback(self.no_completion_available)
            else:
                dfr = defer.succeed(None)
                dfr.addCallback(self.remove_added_tracks)

            dfr.addCallback(self.fade_out)
            dfr.addCallback(self.update_model)
            dfr.addCallback(self.fade_in)
            dfr.addCallback(self.toggle)

        dfr.addBoth(self.stop_loading_animation)
        return dfr

def add_action_complete_tracks_for_album(controller):
    if controller.list_controller.album:
        action = CompleteTracksForAlbum(controller.list_controller)
        controller.list_controller.actions.append(action)
    return defer.succeed(None)


def album_matcher(album1, album2):
    return album1.name.lower() == album2.name.lower()

def track_matcher(track1, track2):
    if track1.track_number != None and track2.track_number != None:
        return track1.track_number == track2.track_number
    else:
        return track1.title.lower() == track2.title.lower()

def compare_albums(album1, album2):
    return album1.name.lower() <= album2.name.lower()

def compare_tracks(track1, track2):
    if track1.track_number != None and track2.track_number != None:
        return track1.track_number <= track2.track_number
    else:
        return track1.title.lower() <= track2.title.lower()

def insert_sorted(model, item, matcher, comparator):
    """
    Insert L{item} into L{model} keeping L{model} sorted.
    It uses L{matcher} to make sure L{model} is not already in L{model}.
    It uses L{comparator} to compare L{item} with the other elements of
    L{model}.
    Return True if L{item} was inserted, False otherwise.

    @param model:      list in which to insert the item
    @type model:       list
    @param item:       item to insert
    @type item:        object
    @param matcher:    function returning True if its two arguments are
                       identical, False otherwise
    @type matcher:     callable
    @param comparator: function returning True if its first argument is
                       smaller than its second one, False otherwise
    @type comparator:  callable

    @rtype:    bool
    """
    index = len(model)
    for i, e in enumerate(model):
        # workaround the fact that model contains in our use case Actions and
        # fake albums with an all_tracks attribute. We want our items to be
        # inserted always after them.
        if isinstance(e, Action) or hasattr(e, "all_tracks"):
            continue

        if matcher(item, e):
            # item already existing in model
            return False
        if comparator(item, e):
            index = i
            break

    model.insert(index, item)
    return True

