# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets.list_carrousel import ListCarrousel
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.pigment.graph.image import Image

from elisa.core.input_event import *

import pgm
from pgm.utils.image import cairo_gradient
import math

class SectionsMenuWidget(ListCarrousel):
    pass

class SectionItemWidget(Widget):
    def __init__(self):
        super(SectionItemWidget, self).__init__()

        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.x, self.image.y = (0.0, 0.0)
        self.image.width, self.image.height = (1.0, 0.49)
        self.image.visible = True

        # reflection
        self.reflection = Image()
        self.add(self.reflection)
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.x, self.reflection.y = (0.0, 0.5)
        self.reflection.width, self.reflection.height = (1.0, 0.5)
        self.reflection.opacity = 120
        self.reflection.visible = True

class SectionsMenuController(PigmentController):
    def set_frontend(self, frontend):
        super(SectionsMenuController, self).set_frontend(frontend)

        # reference to the section (a elisa.plugins.poblesec.link.Link instance)
        # that has been clicked and for which the corresponding controller is
        # being created
        self._previously_clicked_section = None

        # menu widget setup
        self.menu = SectionsMenuWidget(SectionItemWidget)
        self.widget.add(self.menu)
        self.menu.width, self.menu.height = (1.0, 1.0)
        self.menu.x, self.menu.y = (0.0, -0.05)
        self.menu.visible = True

        self.model = notifying_list.List()

        def renderer(item, widget):
            self.frontend.load_from_theme(item.icon, widget.image)
            # FIXME: not updated automatically on theme change
            theme = self.frontend.get_theme()
            icon_path = theme.get_resource(item.icon)
            cairo_gradient(icon_path, widget.reflection, 0.4)

        self.menu.set_model(self.model)
        self.menu.set_renderer(renderer)
        self.menu.connect('item-clicked', self._section_clicked_handler)

    def _reset_previously_clicked_section(self):
        self._previously_clicked_section = None

    def _section_clicked_handler(self, menu, section):
        # only try creating the controller behind the section that was clicked
        # if the user has not already clicked on it and if it was the one
        # selected
        selected_section = self.model[self.menu.selected_item_index]
        if section != self._previously_clicked_section and \
            section == selected_section:
            path = '/poblesec/browser'
            controllers = self.frontend.retrieve_controllers(path)
            # we assume there was only one controller created with that path
            browser = controllers[0]
            path = section.controller_path
            label = section.label
            dfr = browser.history.append_controller(path, label)
            self._previously_clicked_section = section

            # make sure that user will still be able to try entering any
            # section in the future regardless of what might have happened
            dfr.addErrback(lambda x: self._reset_previously_clicked_section())

    def removed(self):
        self._reset_previously_clicked_section()

    def handle_input(self, manager, input_event):
        if self.widget.focus == True:
            if input_event.value == EventValue.KEY_GO_LEFT:
                self.menu.selected_item_index -= 1
                return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                self.menu.selected_item_index += 1
                return True
            elif input_event.value == EventValue.KEY_OK:
                try:
                    item = self.menu.model[self.menu.selected_item_index]
                    self._section_clicked_handler(self.menu, item)
                except IndexError:
                    self.debug("selected-item-index out of range: %s" % self.menu.selected_item_index)
                return True
        else:
            return super(SectionsMenuController, self).handle_input(manager, input_event)
