# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core.input_event import EventValue, EventSource

from elisa.core.media_uri import MediaUri
from elisa.core.common import application

from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.osk import OSK
from elisa.plugins.poblesec.widgets.entry import EntryBar, EntryField

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.list_vertical import ListVertical
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.widgets.button import Button

from pgm.timing import implicit
import pgm

import string

import gobject

_ = install_translation('poblesec')


class LoginSettingsController(PigmentController):

    # The 'validate' signal is sent along with the contents of the entry.
    # The 'text-changed' signal is sent when the contents of the entry is
    # updated.
    __gsignals__ = {'validate': (gobject.SIGNAL_RUN_LAST,
                                 gobject.TYPE_BOOLEAN,
                                 (str,)),
                    'text-changed': (gobject.SIGNAL_RUN_LAST,
                                     gobject.TYPE_BOOLEAN,
                                     (str,)),
                   }

    def initialize(self, service):
        dfr = super(LoginSettingsController, self).initialize()

        self.service = service

        return dfr

    def set_frontend(self, frontend):
        super(LoginSettingsController, self).set_frontend(frontend)

        theme = frontend.get_theme()

        self.connect('validate', self._on_validate)

        self.title_widget = Text()
        self.widget.add(self.title_widget)
        self.title_widget.weight = pgm.TEXT_WEIGHT_BOLD
        self.title_widget.alignment = pgm.TEXT_ALIGN_CENTER
        self.title_widget.label = self.title
        self.title_widget.x, self.title_widget.y = (0.0, 0.10)
        self.title_widget.width, self.title_widget.height = (1.0, 0.05)
        self.title_widget.bg_color = (0, 0, 0, 0)
        self.title_widget.visible = True

        self.subtitle_widget = Text()
        self.widget.add(self.subtitle_widget)
        self.subtitle_widget.weight = pgm.TEXT_WEIGHT_BOLD
        self.subtitle_widget.alignment = pgm.TEXT_ALIGN_CENTER
        self.subtitle_widget.label = self.subtitle
        self.subtitle_widget.x, self.subtitle_widget.y = (0.0, 0.16)
        self.subtitle_widget.width, self.subtitle_widget.height = (1.0, 0.04)
        self.subtitle_widget.bg_color = (0, 0, 0, 0)
        self.subtitle_widget.fg_color = (200, 200, 200, 255)
        self.subtitle_widget.visible = True

        self.entrybar = EntryBar()
        self.widget.add(self.entrybar)
        self.entrybar.size = (1.0, 0.14)
        self.entrybar.position = (0.0, 0.35, 0.0)
        self.entrybar.visible = True

        self.entry = EntryField()
        self.widget.add(self.entry)
        self.entry.size = (0.5, 0.08)
        self.entry.position = (0.2, 0.38, 0.0)
        self.entry.visible = True

        self.button = Image()
        self.widget.add(self.button)
        img_file = theme.get_resource('elisa.plugins.poblesec.enter_button')
        self.button.set_from_file(img_file)
        self.button.bg_color = (0, 0, 0, 0)
        self.button.image = img_file
        self.button.size = (0.1, 0.08)
        self.button.position = (0.71, 0.38, 0.0)
        self.button.visible = True

        self.button.connect("clicked", lambda *args: self.emit("validate", self.entry.text))

        self._signal_handler_ids = []

        alphabetic = 'elisa.plugins.poblesec.osk_alphabetic'
        special_chars = 'elisa.plugins.poblesec.osk_numeric_special_chars'
        self.keyboards = [OSK(theme.get_resource(alphabetic)),
                          OSK(theme.get_resource(special_chars))]
        self._animated_keyboards = []
        for keyboard in self.keyboards:
            self.widget.add(keyboard)

            keyboard.size = (0.6, 0.5)
            keyboard.position = (0.2, 0.5, 0.0)
            keyboard.opacity = 0
            keyboard.visible = True

            char_id = keyboard.connect('key-press-char', self._key_pressed_cb)
            self._signal_handler_ids.append((keyboard, char_id))
            special_id = keyboard.connect('key-press-special',
                                          self._special_key_pressed_cb)
            self._signal_handler_ids.append((keyboard, special_id))

            animated_keyboard = implicit.AnimatedObject(keyboard)
            animated_keyboard.setup_next_animations(duration=300)
            self._animated_keyboards.append(animated_keyboard)

        self.current_keyboard = 0
        self._animated_keyboards[self.current_keyboard].opacity = 255

        # Select the switch key; this works because we know it is the last one
        # for each keyboard.
        # FIXME: this is very weak, and assume that the position of the switch
        # key be always the same for all keyboard layout
        for i in xrange(1, len(self.keyboards)):
            keyboard = self.keyboards[i]
            keyboard.select_key(keyboard.keys[-1][-1])

        entry_focus_id = self.entry.connect('focus', self._on_entry_focus)
        self._signal_handler_ids.append((self.entry, entry_focus_id))
        entrybar_clicked_id = self.entrybar.connect('clicked',
                                                    self._entrybar_clicked_cb)
        self._signal_handler_ids.append((self.entrybar, entrybar_clicked_id))
        widget_focus_id = self.widget.connect('focus', self._on_focus)
        self._signal_handler_ids.append((self.widget, widget_focus_id))

    def clean(self):
        for widget, signal_id in self._signal_handler_ids:
            widget.disconnect(signal_id)
        return super(LoginSettingsController, self).clean()

    def _on_validate(self, controller, text):
        raise NotImplementedError

    def _on_focus(self, widget, focus):
        if focus:
            self.entry.focus = True

    def _on_entry_focus(self, widget, focus):
        self.entrybar.highlight(focus)

    def _entrybar_clicked_cb(self, button, x, y, z, mbutton, time, data):
        self.entry.focus = True

    def _switch_clicked_cb(self, button, x, y, z, mbutton, time, data):
        self._switch_keyboard()

    def _switch_keyboard(self):
        self._animated_keyboards[self.current_keyboard].opacity = 0
        self.current_keyboard = \
            (self.current_keyboard + 1) % len(self.keyboards)
        self.keyboards[self.current_keyboard].focus = True
        self._animated_keyboards[self.current_keyboard].opacity = 255

    def _key_pressed_cb(self, keyboard, value):
        self.entry.text += value
        self.emit('text-changed', self.entry.text)

    def _special_key_pressed_cb(self, keyboard, value):
        if value == 'delete':
            self.entry.text = self.entry.text[:-1]
            self.emit('text-changed', self.entry.text)
        elif value == 'switch_keyboard':
            self._switch_keyboard()

    # FIXME: special characters (like dot, at, etc.) are not passed by the elisa's
    # pigment input provider. Consider adding them in the future
    def handle_input(self, manager, input_event):
        keyboard = self.keyboards[self.current_keyboard]
        if keyboard.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                keyboard.activate_key(keyboard.current_key)
            elif input_event.value == EventValue.KEY_GO_LEFT:
                keyboard.move_selector(LEFT)
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                keyboard.move_selector(RIGHT)
            elif input_event.value == EventValue.KEY_GO_UP:
                top_row = keyboard.keys[0]
                if keyboard.current_key in top_row:
                    self.entry.focus = True
                else:
                    keyboard.move_selector(TOP)
            elif input_event.value == EventValue.KEY_GO_DOWN:
                keyboard.move_selector(BOTTOM)
            else:
                return super(LoginSettingsController, self).handle_input(manager,
                                                                   input_event)
            return True
        elif self.entry.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                self.emit('validate', self.entry.text)
            elif len(str(input_event.value)) == 5:
                key = str(input_event.value)[-1]
                self._key_pressed_cb(keyboard, key)
            elif input_event.value == EventValue.KEY_SPACE:
                self._key_pressed_cb(keyboard, ' ')
            # at present "backspace" is wrapped into KEY_MENU
            elif (input_event.value == EventValue.KEY_MENU) and \
                 (input_event.source == EventSource.KEYBOARD):
                if self.entry.text:
                    self._special_key_pressed_cb(keyboard, 'delete')
                else:
                    # passing control to the parent controller to activate "Go
                    # back" of the history
                    return False
            elif input_event.value == EventValue.KEY_GO_DOWN:
                keyboard.focus = True
            else:
                return super(LoginSettingsController, self).handle_input(manager,
                                                                   input_event)
            return True

class UsernameSettingsController(LoginSettingsController):

    def initialize(self, service, password_controller, login_result_controller,
                   try_login=None, login_actions=[]):
        dfr = super(UsernameSettingsController, self).initialize(service)

        self.password_controller = password_controller
        self.login_result_controller = login_result_controller
        self.try_login = try_login
        self.login_actions = login_actions
        self.title = _('Step 1: Enter Your Username')
        self.subtitle = _('Enter Username & Hit Enter')

        return dfr

    def _on_validate(self, controller, username):
        if not username:
            return

        history = self.frontend.retrieve_controllers('/poblesec/browser')[0].history

        dfr = history.append_controller(self.password_controller,
                                        'Password',
                                        service=self.service,
                                        username=username,
                                        login_result_controller=self.login_result_controller,
                                        try_login=self.try_login,
                                        login_actions=self.login_actions)


class PasswordSettingsController(LoginSettingsController):

    def initialize(self, service, username, login_result_controller, try_login, login_actions):
        dfr = super(PasswordSettingsController, self).initialize(service)

        self.service = service
        self.username = username
        self.login_result_controller = login_result_controller
        self.try_login = try_login
        self.login_actions = login_actions
        self.title = _('Step 2: Enter Your Password')
        self.subtitle = _('Enter Password & Hit Enter')

        self._validating = False

        return dfr

    def _on_validate(self, controller, password):
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]

        # prevent multiple validation while testing the account data.
        # FIXME: an animation would be nice
        if self._validating:
            return

        self._validating = True

        if not self.try_login:
            dfr = defer.succeed(False)
        else:
            dfr = self.try_login(self.username, password)

        def login_result(result):
            self._validating = False
            dfr = browser.history.append_controller(self.login_result_controller,
                                                    _('Login result'),
                                                    result=result,
                                                    service=self.service,
                                                    login_actions=self.login_actions)
            return dfr

        def login_failure(err):
            self.debug("Cannot login into %s: %s" % (self.service, err))
            return login_result(False)

        dfr.addCallback(login_result)
        dfr.addErrback(login_failure)


class ReturnButton(Widget):
    def __init__(self):
        super(ReturnButton, self).__init__()

        # Add a background drawable to receive events
        self._background = Image()
        self.add(self._background)
        self._background.visible = True
        self._background.bg_color = (0, 0, 0, 0)

        self._label = Text()
        self.add(self._label, forward_signals=False)
        self._label.fg_color = (255, 255, 255, 255)
        self._label.bg_color = (0, 0, 0, 0)
        self._label.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self._label.alignment = pgm.TEXT_ALIGN_CENTER
        self._label.size = (1.0, 0.5)
        self._label.position = (0.0, 0.25, 0.0)
        self._label.visible = True

    def label__get(self):
        return self._label.label

    def label__set(self, value):
        self._label.label = value

    label = property(label__get, label__set)


class LoginResultController(PigmentController):

    def initialize(self, result, service, login_actions):
        dfr = super(LoginResultController, self).initialize()

        self.result = result
        self.service = service
        self.login_actions = login_actions

        if result:
            self.title = _('Login Successful')
            self.subtitle = _('Welcome to %s') % service
        else:
            self.title = _('Login Error')
            self.subtitle = _('Cannot Access %s') % service

        return dfr

    def set_frontend(self, frontend):
        super(LoginResultController, self).set_frontend(frontend)

        self.title_widget = Text()
        self.widget.add(self.title_widget)
        self.title_widget.weight = pgm.TEXT_WEIGHT_BOLD
        self.title_widget.alignment = pgm.TEXT_ALIGN_CENTER
        self.title_widget.label = self.title
        self.title_widget.x, self.title_widget.y = (0.0, 0.10)
        self.title_widget.width, self.title_widget.height = (1.0, 0.05)
        self.title_widget.bg_color = (0, 0, 0, 0)
        self.title_widget.visible = True

        self.subtitle_widget = Text()
        self.widget.add(self.subtitle_widget)
        self.subtitle_widget.weight = pgm.TEXT_WEIGHT_BOLD
        self.subtitle_widget.alignment = pgm.TEXT_ALIGN_CENTER
        self.subtitle_widget.label = self.subtitle
        self.subtitle_widget.x, self.subtitle_widget.y = (0.0, 0.16)
        self.subtitle_widget.width, self.subtitle_widget.height = (1.0, 0.04)
        self.subtitle_widget.bg_color = (0, 0, 0, 0)
        self.subtitle_widget.fg_color = (200, 200, 200, 255)
        self.subtitle_widget.visible = True

        if not self.result:
            return

        self.actions_widget = ListVertical(ReturnButton)
        self.widget.add(self.actions_widget)

        def renderer(item, widget):
            widget.label = item.label

        self.actions_widget.set_renderer(renderer)
        self.actions_widget.set_model(self.login_actions)
        self.actions_widget.visible_range_size = 6
        self.actions_widget.connect('item-clicked', self._action_clicked)

        self.actions_widget.position = (0.25, 0.40, 0.0)
        self.actions_widget.size = (0.50, 0.60)
        self.actions_widget.set_selector(self._create_selector())
        self.actions_widget.visible = True

        self.widget.connect('focus', self._on_focus)

    def _on_focus(self, widget, focus):
        if focus:
            self.actions_widget.focus = focus

    def _action_clicked(self, widget, action):
        action.run()

    def _create_selector(self):
        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)
        selector.visible = True

        return selector

    def handle_input(self, manager, input_event):
        if hasattr(self, 'actions_widget') and self.actions_widget.focus:
            if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
                index = self.actions_widget.selected_item_index
                item = self.actions_widget.model[index]
                self.actions_widget.emit("item-clicked", item)
                return True
            elif input_event.value == EventValue.KEY_GO_DOWN:
                if self.actions_widget.selected_item_index < (len(self.actions_widget.model) - 1):
                   self.actions_widget.selected_item_index += 1
                   return True
            elif input_event.value == EventValue.KEY_GO_UP:
                if self.actions_widget.selected_item_index > 0:
                   self.actions_widget.selected_item_index -= 1
                   return True
        return False
