# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.utils.cancellable_defer import CancelledError

from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.list_vertical import ListVertical

from elisa.plugins.poblesec.base.list import ListController
from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.menu_item import MenuItemWidget, \
                                                     DoubleLineMenuItemWidget

import pgm
from pgm.utils.image import cairo_gradient
from pgm.timing import implicit


class Shortcut(Widget):
    def __init__(self):
        super(Shortcut, self).__init__()

        self.letter = Text()
        self.add(self.letter)
        self.letter.width, self.letter.height = (1.0, 0.7)
        x = 0.0
        y = (1.0 - self.letter.height) / 2.0
        self.letter.x, self.letter.y = (x, y)
        self.letter.bg_color = (0, 0, 0, 0)
        self.letter.alignment = pgm.TEXT_ALIGN_CENTER
        self.letter.weight = pgm.TEXT_WEIGHT_BOLD
        self.letter.visible = True

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Shortcut, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'font-family':
                self.letter.font_family = value


class PreviewWidget(Widget):
    def __init__(self):
        super(PreviewWidget, self).__init__()

        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.x, self.image.y = (0.0, 0.0)
        self.image.width, self.image.height = (1.0, 0.5)
        self.image.visible = True

        # reflection
        self.reflection = Image()
        self.add(self.reflection)
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.x, self.reflection.y = (0.0, 0.5)
        self.reflection.width, self.reflection.height = (1.0, 0.5)
        self.reflection.opacity = 80
        self.reflection.visible = True


class PreviewListController(ListController):

    list_widget = ListVertical
    fastscroller = False
    fastscroller_threshold = 20
    shortcuts = list('#abcdefghijklmnopqrstuvwxyz')

    def _create_selector(self):
        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)
        return selector

    def nodes_setup(self):
        super(PreviewListController, self).nodes_setup()
        self.nodes = self.list_widget(self.node_widget, visible_range_size=9)
        self.widget.add(self.nodes)
        self.nodes.width, self.nodes.height = (0.5, 0.8)
        self.nodes.x, self.nodes.y = (0.05, (1.0 - self.nodes.height) / 2.0)
        self.nodes.visible = True
        self.nodes.focus = True
        self.nodes.set_selector(self._create_selector())

        if self.fastscroller:
            self._fastscroller_setup()

    def _fastscroller_setup(self):
        self.fast_scroller = ListVertical(Shortcut, visible_range_size=13)
        if len(self.model) < self.fastscroller_threshold:
            return
        self.fast_scroller.focus_on_click = False
        self.widget.add(self.fast_scroller)
        self.fast_scroller.width, self.fast_scroller.height = (0.05, 0.8)
        y = (1.0 - self.fast_scroller.height) / 2.0
        self.fast_scroller.x, self.fast_scroller.y = (0.0, y)
        self.fast_scroller.visible = True

        def shortcut_changed(widget, shortcut):
            self._shortcut_selected(shortcut)

        self.fast_scroller.connect('item-clicked', shortcut_changed)
        self.fast_scroller.set_model(self.shortcuts)
        self.fast_scroller.set_renderer(self._shortcut_renderer)
        self.fast_scroller.set_selector(self._create_selector())

    def item_to_letter(self, item):
        # item_to_label is the method subclasses should implement
        label = self.item_to_label(item)
        first_letter = label and label[0] or ''
        if first_letter.isalpha():
            return first_letter.lower()
        else:
            return '#'

    def _get_active_shortcuts(self):
        # FIXME: This call is very costly for a large list, the list of active
        # shortcuts should be cached somehow.
        def remove_duplicates(alist):
            # remove duplicates from a list preserving the order
            set = {}
            return [set.setdefault(e, e) for e in alist if e not in set]
 
        letters = [self.item_to_letter(item) for item in self.model]
        return remove_duplicates(letters)

    def _shortcut_renderer(self, shortcut, widget):
        if shortcut in self._get_active_shortcuts():
            widget.letter.fg_color = (100, 100, 100, 255)
        else:
            widget.letter.fg_color = (20, 20, 20, 255)
        widget.letter.label = shortcut.upper()

    def _get_current_shortcut(self):
        return self.shortcuts[self.fast_scroller.selected_item_index]

    def _set_current_shortcut(self, shortcut):
        self.fast_scroller.selected_item_index = self.shortcuts.index(shortcut)

    def _shortcut_selected(self, shortcut):
        # FIXME: stupid search, assumes that the model is already
        # alphabetically sorted
        for index, item in enumerate(self.model):
            if self.item_to_letter(item) == shortcut:
                self.nodes.selected_item_index = index
                break

    def set_frontend(self, frontend):
        super(PreviewListController, self).set_frontend(frontend)

        self.preview = PreviewWidget()
        self.widget.add(self.preview)
        list_width = self.nodes.x + self.nodes.width
        self.preview.width, self.preview.height = ((1.0 - list_width) * 0.7, 0.8)
        x = list_width + ((1.0 - list_width) - self.preview.width) / 2.0
        y = (1.0 - self.preview.height) / 2.0 + self.preview.height / 7.0
        self.preview.x, self.preview.y = (x, y)
        self.preview.visible = True

        self.animated_preview = implicit.AnimatedObject(self.preview)

        # FIXME: this should probably be done (and refactored) just after self.nodes
        #        is created in VerticalListController
        self.nodes.connect('selected-item-changed', self._initiate_load_preview)

        if self.nodes.selected_item_index != -1:
            self._initiate_load_preview(self.nodes, None, None)

    def image_from_item(self, item):
        return None

    def _initiate_load_preview(self, widget, item, previous_item):
        # do not do anything if it was already fading out
        if self.animated_preview.opacity == 0.0:
            return

        self.animated_preview.stop_animations()

        def load_new_preview(timer):
            # unload any previously loaded preview
            self.preview.image.clear()
            self.preview.reflection.clear()

            self.animated_preview.setup_next_animations(duration=200,
                                            transformation=implicit.DECELERATE,
                                            end_callback=None)
            self.animated_preview.opacity = 255
            selected_item = self.model[self.nodes.selected_item_index]
            self._load_preview(selected_item)

        self.animated_preview.setup_next_animations(duration=200,
                                            transformation=implicit.DECELERATE,
                                            end_callback=load_new_preview)
        self.animated_preview.opacity = 0.0

    def _load_preview(self, item):
        image_path = self.image_from_item(item)

        if image_path != None:
            self.preview.image.set_from_file(image_path)

            # FIXME: deactivated because of Python Cairo bindings not releasing
            # the GIL thus locking Pigment's rendering thread when the viewport
            # update-pass signal is emitted
            #try:
            #    cairo_gradient(image_path, self.preview.reflection, 0.4)
            #except:
            #    # FIXME: add support for JPG to the cairo gradient
            #    pass

    def handle_input(self, manager, input_event):
        if self.nothing_to_display_widget.visible:
            return False

        if input_event.value == EventValue.KEY_GO_LEFT and self.fastscroller:
            if self.nodes.focus:
                self.fast_scroller.focus = True
                # Select the letter corresponding to the current item in the
                # list
                current_item = self.model[self.nodes.selected_item_index]
                current_letter = self.item_to_letter(current_item)
                if current_letter in self.shortcuts:
                    self._set_current_shortcut(current_letter)
                return True
            else:
                return False

        elif input_event.value == EventValue.KEY_GO_RIGHT and self.fastscroller:
            if self.nodes.focus:
                return False
            else:
                self.nodes.focus = True
                return True

        elif input_event.value == EventValue.KEY_GO_UP:
            if self.nodes.focus:
                self.nodes.selected_item_index -= 1
                return True
            else:
                # Scroll to the previous shortcut that is activated
                current_shortcut = self._get_current_shortcut()
                active_shorcuts = self._get_active_shortcuts()
                try:
                    active_current_index = active_shorcuts.index(current_shortcut)
                except ValueError:
                    # Such a shortcut does not exist
                    pass
                try:
                    next_index = active_current_index - 1
                    if next_index < 0:
                        return True
                    next_shortcut = active_shorcuts[next_index]
                    if next_shortcut in self.shortcuts:
                        self._set_current_shortcut(next_shortcut)
                except IndexError:
                    # Such a shortcut does not exist
                    pass
                return True

        elif input_event.value == EventValue.KEY_GO_DOWN:
            if self.nodes.focus:
                self.nodes.selected_item_index += 1
                return True
            else:
                # Scroll to the next shortcut that is activated
                current_shortcut = self._get_current_shortcut()
                active_shorcuts = self._get_active_shortcuts()
                active_current_index = active_shorcuts.index(current_shortcut)
                try:
                    next_index = active_current_index + 1
                    next_shortcut = active_shorcuts[next_index]
                    if next_shortcut in self.shortcuts:
                        self._set_current_shortcut(next_shortcut)
                except IndexError:
                    # Such a shortcut does not exist
                    pass
                return True

        elif input_event.value == EventValue.KEY_OK:
            if self.fastscroller and self.fast_scroller.focus:
                shortcut = self._get_current_shortcut()
                self._shortcut_selected(shortcut)
                return True

        return super(PreviewListController, self).handle_input(manager,
                                                               input_event)


class MenuItemPreviewListController(PreviewListController):

    """
    Preview list controller tied to the menu item widget.
    """

    node_widget = MenuItemWidget

    def initialize(self, **kwargs):
        def create_view_mode(self):
            self._view_mode = self.view_mode()
            return self

        def insert_actions(self):
            self.model[0:0] = self.actions
            return self

        def connect_actions_to_model(self):
            notifier = self.actions.notifier
            self._actions_inserted_id = notifier.connect('items-inserted',
                                                         self._actions_inserted)
            self._actions_deleted_id = notifier.connect('items-deleted',
                                                        self._actions_deleted)
            self._actions_changed_id = notifier.connect('items-changed',
                                                        self._actions_changed)
            self._actions_reordered_id = notifier.connect('items-reordered',
                                                          self._actions_reordered)
            return self

        init_deferred = super(MenuItemPreviewListController, self).initialize(**kwargs)
        init_deferred.addCallback(create_view_mode)
        init_deferred.addCallback(insert_actions)
        init_deferred.addCallback(connect_actions_to_model)
        return init_deferred

    def clean(self):
        self.actions.notifier.disconnect(self._actions_inserted_id)
        self.actions.notifier.disconnect(self._actions_deleted_id)
        self.actions.notifier.disconnect(self._actions_changed_id)
        self.actions.notifier.disconnect(self._actions_reordered_id)
        dfr = super(MenuItemPreviewListController, self).clean()
        return dfr

    def _actions_inserted(self, notifier, index, actions):
        self.model[index:index] = actions

    def _actions_deleted(self, notifier, index, actions):
        self.model[index:index+len(actions)] = []

    def _actions_changed(self, notifier, index, actions):
        for i, action in enumerate(actions):
            self.model[index+i] = action

    def _actions_reordered(self, notifier, index, actions):
        self.model.notifier.emit('items-reordered')

    def node_renderer(self, item, widget):
        """
        Render a node using the common API methods defined by the
        L{elisa.plugins.poblesec.base.list.GenericListViewMode} class.
        """
        # Cancel previous deferred calls for this widget
        self.cancel_deferreds(widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        # Render the label of the widget
        widget.label.visible = False
        def got_label(text):
            widget.label.label = text
            widget.label.visible = True
        label_deferred = self._view_mode.get_label(item)
        label_deferred.addCallbacks(got_label, _failure)
        self.register_deferred(widget, label_deferred)

        # Set the default image of the widget
        widget.icon.clear()
        default_image = self._view_mode.get_default_image(item)
        self.frontend.load_from_theme(default_image, widget.icon)

        # Set the real image of the widget
        theme = self.frontend.get_theme()
        image_deferred = self._view_mode.get_image(item, theme)
        if image_deferred is not None:
            def got_thumbnail(thumbnail_file):
                if thumbnail_file is not None:
                    widget.icon.set_from_file(thumbnail_file)
                    selected_item = self.nodes.model[self.nodes.selected_item_index]
                    if selected_item == item:
                        self._load_preview(item)

            self.register_deferred(widget, image_deferred)
            image_deferred.addCallbacks(got_thumbnail, _failure)

    def image_from_item(self, item):
        """
        Display a preview image for an item using the common API methods
        defined by the L{elisa.plugins.poblesec.base.list.GenericListViewMode}
        class.
        """
        theme = self.frontend.get_theme()
        image = self._view_mode.get_preview_image(item, theme) or \
                theme.get_resource(self._view_mode.get_default_image(item))
        return image


class DoubleLineMenuItemPreviewListController(MenuItemPreviewListController):

    """
    Preview list controller tied to the double line menu item widget.
    """

    node_widget = DoubleLineMenuItemWidget

    def node_renderer(self, item, widget):
        super(DoubleLineMenuItemPreviewListController, self).node_renderer(item, widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        # Render the sublabel of the widget
        widget.sublabel.visible = False
        def got_sublabel(text):
            widget.sublabel.label = text
            widget.sublabel.visible = True
        sublabel_deferred = self._view_mode.get_sublabel(item)
        sublabel_deferred.addCallbacks(got_sublabel, _failure)
        self.register_deferred(widget, sublabel_deferred)
