# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import pgm
import math
import gobject

from node import Node
from drawable import Drawable

class NodeNotInGroup(Exception):
    pass

class Group(Node):

    # FIXME: add other Pigment signals: pressured, changed

    __gsignals__ = {
        'drag-begin': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'drag-motion': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'drag-end': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'clicked': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'pressed': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT, gobject.TYPE_UINT)),
        'released': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'double-clicked': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ButtonType, gobject.TYPE_UINT)),
        'scrolled': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN,
                (gobject.TYPE_FLOAT, gobject.TYPE_FLOAT, gobject.TYPE_FLOAT,
                pgm.ScrollDirection, gobject.TYPE_UINT)),
    }

    def __init__(self):
        self._canvas = None
        self._layer = pgm.DRAWABLE_MIDDLE
        self._children = []
        self.absolute_size = (1.0, 1.0)
        self.absolute_position = (0.0, 0.0, 0.0)
        super(Group, self).__init__()

    def canvas__set(self, canvas):
        old_canvas = self._canvas
        new_canvas = canvas

        self._canvas = canvas

        if old_canvas != None:
            for child in self._children:
                if isinstance(child, Drawable):
                    old_canvas.remove(child)
                elif isinstance(child, Group):
                    child.canvas = None

        if new_canvas != None:
            for child in self._children:
                if isinstance(child, Drawable):
                    new_canvas.add(self._layer, child)
                elif isinstance(child, Group):
                    child.canvas = new_canvas

    def canvas__get(self):
        return self._canvas

    canvas = property(canvas__get, canvas__set)

    def layer__set(self, layer):
        if layer == self._layer:
            return

        self._layer = layer

        for child in self._children:
            if isinstance(child, Drawable):
                self._canvas.remove(child)
                self._canvas.add(self._layer, child)
            elif isinstance(child, Group):
                child.layer = self._layer

    def layer__get(self):
        if self._canvas != None:
            return self._layer
        else:
            return pgm.DRAWABLE_UNBOUND

    layer = property(layer__get, layer__set)

    def add(self, child, forward_signals=True):
        self._children.append(child)
        child.parent = self

        if forward_signals:
            for signal_name in ('double-clicked', 'drag-end', 'released'):
                child.connect(signal_name,
                              self._proxy_child_signal, signal_name)

            for signal_name in ('clicked', 'pressed', 'drag-begin', 'drag-motion'):
                child.connect(signal_name,
                              self._proxy_child_signal_with_pressure, signal_name)
            child.connect('scrolled',
                          self._proxy_child_signal_scrolled, 'scrolled')

        if self._canvas != None:
            if isinstance(child, Drawable):
                self._canvas.add(self._layer, child)
            elif isinstance(child, Group):
                child.canvas = self._canvas
                child.layer = self._layer

    def _proxy_child_signal_scrolled(self, drawable, x, y, z, direction, time, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, direction, time)

    def _proxy_child_signal(self, drawable, x, y, z, button, time, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, button, time)

    def _proxy_child_signal_with_pressure(self, drawable, x, y, z, button, time, pressure, signal_name):
        if self.absolute_visible and self.absolute_opacity != 0:
            return self.emit(signal_name, x, y, z, button, time, pressure)

    def remove(self, child):
        try:
            self._children.remove(child)
        except:
            raise NodeNotInGroup

        child.parent = None

        if self._canvas != None:
            if isinstance(child, Drawable):
                self._canvas.remove(child)
            elif isinstance(child, Group):
                child.canvas = None

    def empty(self):
        for child in self._children:
            child.parent = None
            if isinstance(child, Drawable):
                self._canvas.remove(child)
            elif isinstance(child, Group):
                child.canvas = None

        self._children = []

    def __len__(self):
        return len(self._children)

    def __iter__(self):
        return self._children.__iter__()

    def update_absolute_x(self, value=None):
        super(Group, self).update_absolute_x(value)
        for child in self._children:
            child.update_absolute_x()

    def update_absolute_y(self, value=None):
        super(Group, self).update_absolute_y(value)
        for child in self._children:
            child.update_absolute_y()

    def update_absolute_z(self, value=None):
        super(Group, self).update_absolute_z(value)
        for child in self._children:
            child.update_absolute_z()

    def update_absolute_position(self, value=None):
        super(Group, self).update_absolute_position(value)
        for child in self._children:
            child.update_absolute_position()

    def update_absolute_width(self, value=None):
        super(Group, self).update_absolute_width(value)
        for child in self._children:
            child.update_absolute_width()
            # FIXME: write test before uncommenting
            child.update_absolute_x()

    def update_absolute_height(self, value=None):
        super(Group, self).update_absolute_height(value)
        for child in self._children:
            child.update_absolute_height()
            # FIXME: write test before uncommenting
            child.update_absolute_y()

    def update_absolute_size(self, value=None):
        super(Group, self).update_absolute_size(value)
        for child in self._children:
            child.update_absolute_size()
            # FIXME: write test before uncommenting
            child.update_absolute_position()

    def update_absolute_visible(self, value=None):
        super(Group, self).update_absolute_visible(value)
        for child in self._children:
            child.update_absolute_visible()

    def update_absolute_opacity(self, value=None):
        super(Group, self).update_absolute_opacity(value)
        for child in self._children:
            child.update_absolute_opacity()

    # FIXME: write tests
    def regenerate(self):
        for child in self._children:
            child.regenerate()

    def absolute_width__get(self):
        return self.absolute_size[0]

    def absolute_width__set(self, value):
        self.absolute_size = (value, self.absolute_size[1])

    absolute_width = property(absolute_width__get, absolute_width__set)

    def absolute_height__get(self):
        return self.absolute_size[1]

    def absolute_height__set(self, value):
        self.absolute_size = (self.absolute_size[0], value)

    absolute_height = property(absolute_height__get, absolute_height__set)

    def absolute_x__get(self):
        return self.absolute_position[0]

    def absolute_x__set(self, value):
        self.absolute_position = (value, self.absolute_position[1],
                                  self.absolute_position[2])

    absolute_x = property(absolute_x__get, absolute_x__set)

    def absolute_y__get(self):
        return self.absolute_position[1]

    def absolute_y__set(self, value):
        self.absolute_position = (self.absolute_position[0], value,
                                  self.absolute_position[2])

    absolute_y = property(absolute_y__get, absolute_y__set)

    def absolute_z__get(self):
        return self.absolute_position[2]

    def absolute_z__set(self, value):
        self.absolute_position = (self.absolute_position[0],
                                  self.absolute_position[1], value)

    absolute_z = property(absolute_z__get, absolute_z__set)
    
    def clean(self):
        for child in self._children:
            child.clean()
