# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
test the media scanning
"""

from twisted.trial.unittest import TestCase

from elisa.plugins.database.media_scanner import MediaScanner
from elisa.plugins.base.models.file import FileModel, DirectoryModel

from elisa.plugins.database.scanner_models import ScanResource 

from twisted.internet import defer, reactor

from elisa.core import common

from elisa.core.media_uri import MediaUri

from collections import deque

import os

class FakeResourceManager(object):
    tree = {"root": { 'usr' : '',
                   'home' : {'meeko' : '',
                             'chubaka' : '',
                             'ben' : '',
                            },
                   'lib': '',
                },
            "usr" : {'lib' : {'doc' : '',
                              'share' : '',
                              'python' : ''},
                     'share' : {'python' : '',
                                'usr' : '',
                                'test' : '42',
                                },
                    'log' : '',
                    'lock' : '',
                },
            "var" : {'log' : {'x11': '',
                              'gdm': '',
                              'test': '' },
                     'empty' : {},
                     'file1' : '',
                     'file3' : '',
                },   

            }

    def get(self, uri, context=None):
        if not context:
            context = DirectoryModel()
            context.uri = uri

        path = uri.path.split('/')[1:]
        if uri.filename == '':
            path.pop(-1)

        result_dict = self.tree
        for key in path:
            result_dict = result_dict[key]

        for key, children in result_dict.iteritems():
            cur_uri = MediaUri("%s%s" % (uri, key))
            if isinstance(children, dict):
                d = DirectoryModel()
                d.uri = cur_uri
                context.files.append(d)
            else:
                f = FileModel()
                f.uri = cur_uri
                context.files.append(f)
        return (context, defer.succeed(context))

class Application(object):
    pass 

class DummyParser(object):
    def query_model(self, *args):
        return defer.succeed(None)
    def mark_deleted(self, *args):
        return defer.succeed(None)

class DummyResult(object):
    result = None
    def get_all(self):
        return defer.succeed(self.result)

class DummyStore(object):
    def execute(self, *args):
        result = True
        if args[0].startswith('PRAGMA'):
            result = DummyResult()
            result.result = [(0,'playcount'),(1,'last_played'),\
                              (3,'shot_time'),(2,'release_date')]
        return defer.succeed(result)
    def commit(self, *args):
        return defer.succeed(True)

class TestMediaScanner(TestCase):
    """
    Test the media scanning part
    """
    stat_uri = MediaUri('media_scanner://localhost/statistic')
    queue_uri = MediaUri('media_scanner://localhost/queue')

    def setUp(self):
        def set_scanner(result):
            self.scanner = result
            # overwrite the Parser and stop the old one
            parser = self.scanner.parser
            self.scanner.parser = DummyParser()
            self.scanner._scan_directories = lambda: None
            self.scanner.local_resource = FakeResourceManager()
            return parser.clean()

        def create(result):
            return MediaScanner.create({'delay':0.2}).addCallback(set_scanner)
    
        return self.patch_application().addCallback(create)

    def patch_application(self):
        self.app = common.application
        common.application = Application()
#        common.application.resource_manager = FakeResourceManager()
        common.application.store = DummyStore()
        return defer.succeed(self)

    def tearDown(self):
        common.application = self.app
        return self.scanner.clean()

    def test_get_stats(self):
        model, dfr = self.scanner.get(MediaUri(self.stat_uri))
        self.assertEquals(model, self.scanner.scan_stat)

    def test_recursive_scan(self):
        s = ScanResource()
        s.root_uri = MediaUri("file:///root/")


        def check_result(result):
            stat, dfr = self.scanner.get(self.stat_uri)
            self.assertFalse(self.scanner.running)
            self.assertFalse(stat.running)

            # start the asnyc waiter
            dfr = defer.Deferred()
            async_wait(0, dfr)
            return dfr

        def async_wait(counter, dfr):
            if len(s.pending_dfrs):
                if counter == 5:
                    self.assertFalse(True, "Even after 5 Seconds it is peding")
                return reactor.callLater(1, async_wait, counter+1, dfr)

            self.assertEquals(s.files_scanned, 5)
            self.assertEquals(s.files_failed, [])

            dfr.callback(True)

        return self.scanner._scan_recursive(s).addCallback(check_result)

    def test_put_deleting(self):
        """
        Try to put some requests into the media scanner and check if it works
        """
        marked = []

        # we don't want them really start to scan, so overwrite the later
        # processes
        self.scanner._file_found = lambda model, stat: defer.succeed(model)
        self.scanner._count_files = lambda path: defer.succeed(19)

        stat, dfr = self.scanner.get(self.stat_uri)

        # overwrite the mark_deleted method
        self.scanner.parser.mark_deleted = lambda x: defer.succeed(marked.append(x))
        for name in ('root', 'usr', 'var'):
            dfr = self.scanner.put(MediaUri('file:///%s' % name), self.queue_uri)

        # as soon as they are put, they should be marked as deleted
        self.assertEquals(marked, [u'/root/', u'/usr/', u'/var/'])

        self.assertEquals(stat.currently_scanning.root_uri.path, u'/root/')
        return dfr


class TestFileCounting(TestCase):

    tree = {'test' : {'test' : None,
                      'drei' : {'vier' : None,
                                'fünf' : None,
                                'sechs': None,
                                },
                     'sieben' : None,
                     'acht' : { 'neun' : None,
                                'one' : None,
                                'two' : None,
                                'three' : None},
                    },
            'fuffzig' : {'four' : None,
                         'five' : None,
                         'six' : None,
                         'seven' : None,
            },
    }


    
    def setUp(self):
        self.counter_path = os.path.abspath('counter_dir')
        self._create_structure(self.counter_path, self.tree)

        # we only test on certain method, so it is enough to initiate
        self.scanner = MediaScanner()

    def _create_structure(self, path, values):
        if not isinstance(values, dict):
            # nothing. it is a file
            open(path,'w').close()
        else:
            os.makedirs(path)
            for key, value in values.iteritems():
                new_path = os.path.join(path, key)
                self._create_structure(new_path, value)

    def test_counter(self):
        dfr = self.scanner._count_files(self.counter_path)
        dfr.addCallback(self.assertEquals, 13)
        return dfr
        

