# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Alessandro Decina <alessandro@fluendo.com>

import dbus
from dbus.exceptions import DBusException
from twisted.trial.unittest import TestCase
from twisted.internet import defer, task, reactor
from twisted.python.reflect import qual

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.extern.storm_wrapper import store
from storm.locals import create_database
from elisa.plugins.database.dbus_service import DatabaseDBusServiceProvider, \
        ArtistNotFoundError, MusicAlbumNotFoundError, MusicTrackNotFoundError, \
        PhotoAlbumNotFoundError, PictureNotFoundError
from elisa.plugins.database.media_scanner import SCHEMA
from elisa.plugins.database.models import Artist, MusicAlbum, MusicTrack, \
        PhotoAlbum, Image, File

def defer_dbus(method, *args, **kw):
    dfr = defer.Deferred()
    def callback(*args):
        dfr.callback(*args)
    
    def errback(*args):
        dfr.errback(*args)

    kw['reply_handler'] = callback
    kw['error_handler'] = errback

    method(*args, **kw)

    return dfr

class ArtistName(unicode):
    def __new__(cls, name):
        return unicode.__new__(cls, 'Artist %s' % name)

class MusicAlbumName(unicode):
    def __new__(cls, artist_name, name):
        return unicode.__new__(cls, '%s - MusicAlbum %s' % (artist_name, name))

class MusicAlbumCoverUri(unicode):
    def __new__(cls, album_name):
        return unicode.__new__(cls, '%s - Cover.png' % album_name)

class MusicTrackTitle(unicode):
    def __new__(cls, album_name, name):
        return unicode.__new__(cls, '%s - Track %s' % (album_name, name))

class MusicTrackFilePath(unicode):
    def __new__(cls, music_track_name):
        return unicode.__new__(cls, '/' + music_track_name.replace(' ', '_') + '.ogg')

class PhotoAlbumName(unicode):
    def __new__(cls, name):
        return unicode.__new__(cls, 'PhotoAlbum %s' % name)

class ImageFilePath(unicode):
    def __new__(cls, album_name, name):
        return unicode.__new__(cls, '/%s - %s.png' % (album_name, name))

class TestDatabaseDBusServiceProvider(DatabaseDBusServiceProvider):
    _last_played_track_tuple = None
    _last_displayed_picture_tuple = None

    def _register_player_status_changes(self):
        pass

    def _emit_last_played_track(self, last_played_track_tuple):
        self._last_played_track_tuple = last_played_track_tuple
    
    def _emit_last_displayed_picture(self, last_displayed_picture_tuple):
        self._last_displayed_picture_tuple = last_displayed_picture_tuple

class TestDBMixin(object):
    def setUp(self):
        # create and start the service 
        def service_created_cb(service):
            self.service = service
            return service.start()

        def create_dbus_proxy(service):
            bus = dbus.SessionBus()
            self.proxy = bus.get_object(self.connection, self.object_path)

        dfr1 = TestDatabaseDBusServiceProvider.create({})
        dfr1.addCallback(service_created_cb)
        dfr1.addCallback(create_dbus_proxy)

        # create and start the database
        self.db = create_database('sqlite:')
        self.store = store.DeferredStore(self.db, False)
        self.patch_application()
        dfr2 = self.store.start()
        dfr2.addCallback(self._populate_database)

        return defer.DeferredList([dfr1, dfr2])

    def tearDown(self):
        self.unpatch_application()
        dfr1 = self.store.stop()
        dfr2 = self.service.stop()
        del self.proxy

        return defer.DeferredList([dfr1, dfr2])

    def patch_application(self):
        self.application = common.application

        class Dummy(object):
            pass

        common.application = Dummy()
        common.application.store = self.store

    def unpatch_application(self):
        common.application = self.application

    def failUnlessDBusException(self, dfr, *expected_failures):
        expected_failure_names = []
        for expected_failure in expected_failures:
            name = qual(expected_failure)
            if name.startswith('exceptions.'):
                name = name[11:]

            name = 'org.freedesktop.DBus.Python.' + name
            expected_failure_names.append(name)

        def _cb(ignore):
            raise self.failureException(
                "did not catch an error, instead got %r" % (ignore,))

        def _eb(failure):
            if failure.check(DBusException) and \
                failure.value.get_dbus_name() in expected_failure_names:
                return failure.value
            else:
                output = ('\nExpected DBusException: %r\nGot:\n%s'
                          % (expected_failure_names, str(failure)))
                raise self.failureException(output)
        return dfr.addCallbacks(_cb, _eb)

class TestMusic(TestDBMixin, TestCase):
    number_artists = 3
    number_albums = 5
    number_tracks = 16

    connection = 'com.fluendo.Elisa.Plugins.Database'
    object_path = '/com/fluendo/Elisa/Plugins/Database/Music'

    def _populate_database(self, result):
        store = self.store

        def iterator():
            for statement in SCHEMA:
                yield store.execute(statement)
            yield store.commit()

            for a in xrange(self.number_artists):
                artist = Artist()
                artist.name = ArtistName(a)
                yield store.add(artist)

                for i in xrange(self.number_albums):
                    album = MusicAlbum()
                    album.name = MusicAlbumName(artist.name, i)
                    album.cover_uri = MusicAlbumCoverUri(album.name)
                    yield store.add(album)

                    for x in xrange(self.number_tracks):
                        track = MusicTrack()
                        track.title = MusicTrackTitle(album.name, x)
                        track.file_path = MusicTrackFilePath(track.title)
                        track.album_name = album.name
                        yield store.add(track)
                        yield track.artists.add(artist)

                        dbfile = File()
                        dbfile.path = track.file_path
                        dbfile.last_played = (a * 100) + (i * 10) + x
                        yield store.add(dbfile)

            yield store.commit()

            self._last_played_track_path = dbfile.path

        dfr = task.coiterate(iterator())
        return dfr

    def test_get_albums_wrong_arguments(self):
        deferreds = []

        dfr = defer_dbus(self.proxy.get_albums, 0, 0)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)

        dfr = defer_dbus(self.proxy.get_albums, -1, 10)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)
        
        dfr = defer_dbus(self.proxy.get_albums, 1, -1)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)

        return defer.DeferredList(deferreds)

    def test_get_albums(self):
        total_albums = self.number_artists * self.number_albums

        def get_full_range(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), 10)

            dfr = defer_dbus(self.proxy.get_albums, 0, 10)
            dfr.addCallback(get_albums_cb)
        
            return dfr

        def get_non_full_range(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), total_albums)

            dfr = defer_dbus(self.proxy.get_albums, 0, total_albums + 3)
            dfr.addCallback(get_albums_cb)
            
            return dfr

        def get_non_full_range_offset(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), 7)

            dfr = defer_dbus(self.proxy.get_albums, total_albums - 7, 18)
            dfr.addCallback(get_albums_cb)
            
            return dfr

        dfr = defer.Deferred()
        dfr.addCallback(get_full_range)
        dfr.addCallback(get_non_full_range)
        dfr.addCallback(get_non_full_range_offset)

        dfr.callback(None)

        return dfr

    def test_get_album_artwork_wrong_arguments(self):
        artist_name = ArtistName(0)
        album_name = MusicAlbumName(artist_name, 0)
        cover_uri = MusicAlbumCoverUri(album_name) 

        def test_wrong_artist_name(result):
            dfr = defer_dbus(self.proxy.get_album_artwork,
                    'not in the library', album_name)
            self.failUnlessDBusException(dfr, ArtistNotFoundError)
            return dfr

        def test_wrong_album_name(result):
            dfr = defer_dbus(self.proxy.get_album_artwork,
                    artist_name, 'not in the library')
            self.failUnlessDBusException(dfr, MusicAlbumNotFoundError)
            return dfr

        dfr = defer.Deferred()
        dfr.addCallback(test_wrong_artist_name)
        dfr.addCallback(test_wrong_album_name)
        dfr.callback(None)

        return dfr

    def test_get_album_artwork(self):
        def get_album_artwork_cb(result, album_name):
            cover_uri = MusicAlbumCoverUri(album_name)
            self.failUnlessEqual(result, cover_uri)
        
        def iterator():
            for n_artist in xrange(self.number_artists):
                artist_name = ArtistName(n_artist)

                for n_album in xrange(self.number_albums):
                    album_name = MusicAlbumName(artist_name, n_album)

                    dfr = defer_dbus(self.proxy.get_album_artwork,
                            artist_name, album_name)
                    dfr.addCallback(get_album_artwork_cb, album_name)

                    yield dfr

        dfr = task.coiterate(iterator())
        return dfr
    
    def test_get_album_tracks_wrong_arguments(self):
        artist_name = ArtistName(0)
        album_name = MusicAlbumName(artist_name, 0)
        cover_uri = MusicAlbumCoverUri(album_name) 

        def test_wrong_artist_name(result):
            dfr = defer_dbus(self.proxy.get_album_tracks,
                    'not in the library', album_name)
            self.failUnlessDBusException(dfr, ArtistNotFoundError)
            return dfr

        def test_wrong_album_name(result):
            dfr = defer_dbus(self.proxy.get_album_tracks,
                    artist_name, 'not in the library')
            self.failUnlessDBusException(dfr, MusicAlbumNotFoundError)
            return dfr

        dfr = defer.Deferred()
        dfr.addCallback(test_wrong_artist_name)
        dfr.addCallback(test_wrong_album_name)
        dfr.callback(None)

        return dfr

    def test_get_album_tracks(self):
        def get_album_tracks_cb(result, artist_name, album_name):
            self.failUnlessEqual(len(result), self.number_tracks)
            
            res = []
            for title, path in result:
                res.append((unicode(title), unicode(path)))
            
            for i in xrange(self.number_tracks):
                expected_title = MusicTrackTitle(album_name, i)
                expected_path = MusicTrackFilePath(expected_title)
                self.failUnlessIn((expected_title, expected_path), res)

        def iterator():
            for n_artist in xrange(self.number_artists):
                artist_name = ArtistName(n_artist)

                for n_album in xrange(self.number_albums):
                    album_name = MusicAlbumName(artist_name, n_album)

                    dfr = defer_dbus(self.proxy.get_album_tracks,
                            artist_name, album_name)
                    dfr.addCallback(get_album_tracks_cb,
                            artist_name, album_name)

                    yield dfr

        dfr = task.coiterate(iterator())
        return dfr

    def test_get_last_played_track(self):
        def check_last_played(last_played_tuple):
            self.failUnlessEqual(len(last_played_tuple), 4)
            
            artist_name, album_name, title, path = last_played_tuple
            self.failUnlessEqual(self._last_played_track_path, path)

        dfr = defer_dbus(self.proxy.get_last_played_track)
        dfr.addCallback(check_last_played)
        return dfr

    def test_last_played_track_signal(self):
        # create a fake PlayableModel
        class Model(object):
            def __init__(self, filename):
                self.uri = MediaUri('file://' + filename)

        # create a fake Player
        class FakePlayer(object):
            STOPPED = 1
            PLAY = 2
            BUFFERING = 3
            PLAYING = 4
            PAUSED = 5
            
            def __init__(self):
                self.playlist = []
                self.current_index = 0

        player = FakePlayer()
        tuples = []
        for i in xrange(2):
            artist_name = ArtistName(i)
            album_name = MusicAlbumName(artist_name, i+1)
            track_title = MusicTrackTitle(album_name, i+2)
            track_path = MusicTrackFilePath(track_title)
            tuples.append((artist_name, album_name, track_title, track_path))
            player.playlist.append(Model(track_path))

        def simulate_playing(result):
            return self.service._music_player_status_cb(player, FakePlayer.PLAYING)

        def pop_and_check_tuple(result, expected):
            tup, self.service._last_played_track_tuple = \
                    self.service._last_played_track_tuple, None
            self.failUnlessEqual(tup, expected)

            return result

        def set_current_index(result, index):
            player.current_index = index

            return result

        dfr = defer.Deferred()
        # simulate the first playing event, last == None, emit
        dfr.addCallback(simulate_playing)
        dfr.addCallback(pop_and_check_tuple, tuples[0])
        # simulate another playing event, last == current, don't emit
        dfr.addCallback(simulate_playing)
        dfr.addCallback(pop_and_check_tuple, None)
        # change track, emit event, last != current, emit
        dfr.addCallback(set_current_index, 1)
        dfr.addCallback(simulate_playing)
        dfr.addCallback(pop_and_check_tuple, tuples[1])
        # who-hoo
        dfr.callback(None)

        return dfr

class TestPhoto(TestDBMixin, TestCase):
    number_albums = 12
    number_images = 17

    connection = 'com.fluendo.Elisa.Plugins.Database'
    object_path = '/com/fluendo/Elisa/Plugins/Database/Photo'

    def _populate_database(self, result):
        store = self.store

        def iterator():
            for statement in SCHEMA:
                yield store.execute(statement)
            yield store.commit()

            for i in xrange(self.number_albums):
                album = PhotoAlbum()
                album.name = PhotoAlbumName(i)
                yield store.add(album)

                for x in xrange(self.number_images):
                    image = Image()
                    image.file_path = ImageFilePath(album.name, x)
                    image.album_name = album.name
                    yield store.add(image)
                    yield album.photos.add(image)
                    
                    dbfile = File()
                    dbfile.path = image.file_path
                    dbfile.last_played = (i * 10) + x
                    yield store.add(dbfile)

            yield store.commit()
            
            self._last_displayed_picture_path = dbfile.path
        

        dfr = task.coiterate(iterator())
        return dfr

    def test_get_albums_wrong_arguments(self):
        deferreds = []

        dfr = defer_dbus(self.proxy.get_albums, 0, 0)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)

        dfr = defer_dbus(self.proxy.get_albums, -1, 10)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)
        
        dfr = defer_dbus(self.proxy.get_albums, 1, -1)
        self.failUnlessDBusException(dfr, IndexError)
        deferreds.append(dfr)

        return defer.DeferredList(deferreds)

    def test_get_albums(self):
        total_albums = self.number_albums

        def get_full_range(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), 10)

            dfr = defer_dbus(self.proxy.get_albums, 0, 10)
            dfr.addCallback(get_albums_cb)
        
            return dfr

        def get_non_full_range(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), total_albums)

            dfr = defer_dbus(self.proxy.get_albums, 0, total_albums + 3)
            dfr.addCallback(get_albums_cb)
            
            return dfr

        def get_non_full_range_offset(result):
            def get_albums_cb(albums):
                self.failUnlessEqual(len(albums), 7)

            dfr = defer_dbus(self.proxy.get_albums, total_albums - 7, 18)
            dfr.addCallback(get_albums_cb)
            
            return dfr

        dfr = defer.Deferred()
        dfr.addCallback(get_full_range)
        dfr.addCallback(get_non_full_range)
        dfr.addCallback(get_non_full_range_offset)

        dfr.callback(None)

        return dfr
    
    def test_get_album_pictures_wrong_arguments(self):
        def test_wrong_album(result):
            dfr = defer_dbus(self.proxy.get_album_pictures,
                    'not in the library')
            self.failUnlessDBusException(dfr, PhotoAlbumNotFoundError)
            return dfr

        dfr = defer.Deferred()
        dfr.addCallback(test_wrong_album)
        dfr.callback(None)

        return dfr

    def test_get_album_pictures(self):
        def iterator():
            for n_album in xrange(self.number_albums):
                album_name = PhotoAlbumName(n_album)

                def get_album_pictures_cb(result):
                    self.failUnlessEqual(len(result), self.number_images)

                dfr = defer_dbus(self.proxy.get_album_pictures, album_name)
                dfr.addCallback(get_album_pictures_cb)

                yield dfr

        dfr = task.coiterate(iterator())
        return dfr
    
    def test_get_last_displayed_picture(self):
        def check_last_played(last_played_tuple):
            self.failUnlessEqual(len(last_played_tuple), 2)

            album_name, path = last_played_tuple
            self.failUnlessEqual(self._last_displayed_picture_path, path)

        dfr = defer_dbus(self.proxy.get_last_displayed_picture)
        dfr.addCallback(check_last_played)
        return dfr

    def test_last_displayed_picture_signal(self):
        class ImageModel(object):
            def __init__(self, filename):
                self.references = [MediaUri('file://' + filename)]

        # create a fake Player
        class FakePlayer(object):
            def __init__(self):
                self.playlist = []

        player = FakePlayer()
        tuples = []
        for i in xrange(2):
            album_name = PhotoAlbumName(i)
            picture_path = ImageFilePath(album_name, i+1)
            tuples.append((album_name, picture_path))
            player.playlist.append(ImageModel(picture_path))

        def simulate_playing(result, index):
            return self.service._slideshow_player_current_index_changed_cb(player,
                    index, None)

        def pop_and_check_tuple(result, expected):
            tup, self.service._last_displayed_picture_tuple = \
                    self.service._last_displayed_picture_tuple, None
            self.failUnlessEqual(tup, expected)

            return result

        dfr = defer.Deferred()
        # simulate the first playing event, last == None, emit
        dfr.addCallback(simulate_playing, 0)
        dfr.addCallback(pop_and_check_tuple, tuples[0])
        # simulate another playing event, last == current, don't emit
        dfr.addCallback(simulate_playing, 0)
        dfr.addCallback(pop_and_check_tuple, None)
        # change picture, emit event, last != current, emit
        dfr.addCallback(simulate_playing, 1)
        dfr.addCallback(pop_and_check_tuple, tuples[1])
        # who-hoo
        dfr.callback(None)

        return dfr
