# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

import pgm, math

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text

from pgm.timing import implicit


from elisa.core.resource_manager import NoMatchingResourceProvider
from elisa.core.media_uri import MediaUri
from elisa.core import common

# i18n
from elisa.core.utils.i18n import install_translation
_ = install_translation('database')

from twisted.internet import reactor, defer

class ScannerWidget(Widget):
    def __init__(self):
        super(ScannerWidget, self).__init__()
        
        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.visible = True
        self.image.x = 0.85
        self.image.y = 0.0
        self.image.width = 0.189
        self.image.height = 1

        self.text = Text()
        self.add(self.text)
        self.text.bg_a = 0
        self.text.font_family = "Liberation Sans"
        self.text.font_height = 1
        self.text.font_weight = pgm.TEXT_WEIGHT_BOLD
        self.text.alignment = pgm.TEXT_ALIGN_RIGHT
        self.text.y = 0.0
        self.text.width = 0.80
        self.text.height = 0.5
        self.text.visible = True
        self.text.fg_color = (130, 130, 130, 255)

        self.subtext = Text()
        self.add(self.subtext)
        self.subtext.bg_a = 0
        self.subtext.font_family = "Liberation Sans"
        self.subtext.font_height = 1
        self.subtext.font_weight = pgm.TEXT_WEIGHT_LIGHT
        self.subtext.alignment = pgm.TEXT_ALIGN_RIGHT
        self.subtext.y = 0.5
        self.subtext.width = 0.80
        self.subtext.height = 0.42
        self.subtext.visible = True
        self.subtext.fg_color = (102, 102, 102, 255)
       
        # create animation stuff
        self._loop_call = None
        self.animated = implicit.AnimatedObject(self)
        self._rotation_matrix = pgm.mat4x4_new_identity()
        self._rotation_matrix.translate(0.5, 0.5, 0.0)
        self._rotation_matrix.rotate_z(-math.pi / 30.0)
        self._rotation_matrix.translate(-0.5, -0.5, 0.0)
        
        self._stopping_animation = False

        # to be able to be more verbose on statistics
        self._currently_scanning = None
        self._percent = None

    def running__set(self, running):
        if running:
            self.opacity = 255
            if not self._loop_call:
                self._transform_mapping_matrix_cb()
        else:
            self.opacity = 0

    running = property(fset=running__set)

    def currently_scanning__set(self, scan_resource):
        if self._currently_scanning:
            # we have to unbind from the old one
            self.files_scanned = 0
            self._currently_scanning.unbind_object(self)
            self._currently_scanning = None
            self.text.label = ''
            self.subtext.label = ''

        if scan_resource is None:
            # scanning stopped
            self.running = False
        else:
            self.running = True
            self._currently_scanning = scan_resource

            self.text.label = _('Scanning %s') % scan_resource.name
            self.subtext.label = _('estimating')
            scan_resource.bind('files_scanned', self, 'files_scanned')

    currently_scanning = property(fset=currently_scanning__set)

    def files_scanned__set(self, counter):
        if not self._currently_scanning.files_total:
            # there seem to be no total amount. we can't calculate
            return

        percent = int(counter * 100. / self._currently_scanning.files_total)

        if self._percent == percent:
            # only update if the percent changed
            return

        self._percent = percent
        self.subtext.label = _('%d%% Complete') % self._percent

    files_scanned = property(fset=files_scanned__set)
    
    def clean(self):
        super(ScannerWidget, self).clean()
        self._stopping_animation = True

    def _transform_mapping_matrix_cb(self):
        self._loop_call = None
        if self.opacity and self.visible and not self._stopping_animation:
            # only do the animation as long as we are shown
            self.image.mapping_matrix *= self._rotation_matrix
            self._loop_call = reactor.callLater(0.017, \
                    self._transform_mapping_matrix_cb)

def scanning_decorator(controller):
    uri = MediaUri("media_scanner://localhost/statistic")
    try:
        model, dfr = common.application.resource_manager.get(uri)
    except NoMatchingResourceProvider, e:
        # Media Scanner not installed
        return defer.fail(e)

    widget = ScannerWidget()
    controller.widget.add(widget)

    theme = controller.frontend.get_theme()

    wait_icon = theme.get_resource("elisa.plugins.poblesec.waiting_icon")
    widget.opacity = 0
    widget.visible = True
    widget.image.set_from_file(wait_icon)
    widget.width = 0.35
    widget.height = 0.08
    widget.x = 0.62
    widget.y = 0.9

    model.bind('currently_scanning', widget, 'currently_scanning')

    return defer.succeed(widget)

