# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>


from elisa.core import common

from elisa.plugins.base.models.audio import TrackModel

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.music_library import ArtistsViewMode, \
                                                 AlbumsViewMode, \
                                                 TracksViewMode

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController


class ArtistsDaapController(HierarchyController):

    def initialize(self, uri):
        deferred = super(ArtistsDaapController, self).initialize()
        self.uri = uri

        def get_artists(self):
            model, dfr = common.application.resource_manager.get(self.uri)
            return dfr

        def got_artists(model):
            artists = model.artists
            artists = sorted(artists, key=lambda a: a.name.lower())
            self.model.extend(artists)
            return self

        deferred.addCallback(get_artists)
        deferred.addCallback(got_artists)
        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item

            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            controller_path = '/poblesec/music/daap/albums'
            dfr = browser.history.append_controller(controller_path,
                                                    item.name, artist=item)

class ArtistsDaapPreviewListController(ArtistsDaapController, MenuItemPreviewListController):
    view_mode = ArtistsViewMode
    fastscroller = True

    def item_to_label(self, item):
        return item.name

class ArtistsDaapCoverflowController(ArtistsDaapController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsViewMode

class ArtistsDaapGridController(ArtistsDaapController, GridItemGridController):
    view_mode = ArtistsViewMode

class ArtistsDaapListSwitcherController(ListSwitcherController):
    modes = [ArtistsDaapPreviewListController,
             ArtistsDaapCoverflowController,
             ArtistsDaapGridController]
    default_mode = ArtistsDaapPreviewListController


class AlbumsDaapController(HierarchyController):

    def initialize(self, artist):
        deferred = super(AlbumsDaapController, self).initialize()
        self.artist = artist

        def get_albums(self):
            model, dfr = common.application.resource_manager.get(self.artist.elisa_uri)
            return dfr

        def got_albums(albums_list_model):
            albums = albums_list_model.albums

            albums = sorted(albums, key=lambda a: a.name.lower())
            self.model.extend(albums)
            return self

        deferred.addCallback(get_albums)
        deferred.addCallback(got_albums)
        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item

            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            controller_path = '/poblesec/music/daap/tracks'
            dfr = browser.history.append_controller(controller_path,
                                                    item.name, album=item)

class AlbumsDaapPreviewListController(AlbumsDaapController, MenuItemPreviewListController):
    view_mode = AlbumsViewMode

class AlbumsDaapCoverflowController(AlbumsDaapController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsViewMode

class AlbumsDaapGridController(AlbumsDaapController, GridItemGridController):
    view_mode = AlbumsViewMode

class AlbumsDaapListSwitcherController(ListSwitcherController):
    modes = [AlbumsDaapPreviewListController,
             AlbumsDaapCoverflowController,
             AlbumsDaapGridController]
    default_mode = AlbumsDaapCoverflowController


class TracksDaapController(HierarchyController):

    def initialize(self, album):
        deferred = super(TracksDaapController, self).initialize()
        self.album = album

        def get_tracks(self):
            model, dfr = common.application.resource_manager.get(self.album.elisa_uri)
            return dfr

        def got_tracks(tracks_list_model):
            tracks = sorted(tracks_list_model.tracks,
                            key=lambda i: i.track_number)
            self.model.extend(tracks)
            return self

        deferred.addCallback(get_tracks)
        deferred.addCallback(got_tracks)
        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            self.play_audio(item)

    def play_audio(self, item):
        player = self.frontend.retrieve_controllers('/poblesec/music_player')[0]

        # Enqueue and play the clicked item
        player.player.play_model(item.playable_model)

        main = self.frontend.retrieve_controllers('/poblesec')[0]
        main.show_music_player()
        self.stop_loading_animation()

        # Enqueue all the following items of the container
        index = self.model.index(item)
        for item in self.model[index+1:]:
            if not isinstance(item, TrackModel):
                continue

            player.player.enqueue_to_playlist(item.playable_model)


class TracksDaapViewMode(TracksViewMode):

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        return None



class TracksDaapPreviewListController(TracksDaapController, MenuItemPreviewListController):
    view_mode = TracksDaapViewMode

class TracksDaapCoverflowController(TracksDaapController, ImageWithReflectionCoverflowController):
    view_mode = TracksDaapViewMode

class TracksDaapGridController(TracksDaapController, GridItemGridController):
    view_mode = TracksDaapViewMode

class TracksDaapListSwitcherController(ListSwitcherController):
    modes = [TracksDaapPreviewListController,
             TracksDaapCoverflowController,
             TracksDaapGridController]
    default_mode = TracksDaapPreviewListController
