# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampann <benjamin@fluendo.com>



from elisa.plugins.search.search_metaresource_provider import SearchMetaresourceProvider
from elisa.plugins.search.searcher import Searcher

from elisa.core.media_uri import MediaUri
from twisted.internet import task, reactor
from elisa.core.utils import defer

from elisa.core.tests.resource_providers import UriRegExpTester, GenericSetup
from twisted.trial.unittest import TestCase

class TestUris(GenericSetup, UriRegExpTester, TestCase):
    resource_class = SearchMetaresourceProvider
    working_uris = [ 'elisa://search/music',
                     'elisa://search/music/madonna?filter=artist',
                     'elisa://search/music/music?filter=albums',
                     'elisa://search/music/water?filter=albums,artists,tracks']
    failing_uris = ['elisa:///', 'elisa://', 'elisa://search_something/',
                    'elisas://search/', 'elisa://search/pictures',
                    'elisa://search/someone']

class BitterSearcher(Searcher):
    """
    This dummy searcher gets angry when he is unreferenced without being cleaned
    up
    """
    def __init__(self):
        super(BitterSearcher, self).__init__()
        self._cleaned = False

    def clean(self):
        self._cleaned = True
        return super(BitterSearcher, self).clean()

    def __del__(self):
        assert self._cleaned

class DummySearcher(Searcher):
    def __init__(self):
        self.searches = []
    def search(self, *args):
        self.searches.append(args)
        return defer.succeed(self)

class HangingSearcher(Searcher):
    def __init__(self):
        self.cancelled = False
    def cancel(self, dfr):
        self.cancelled = True
    def search(self, *args):
        return defer.Deferred(self.cancel)

class UnloadMixin(object):

    def _unload_searchers(self, old_result):
        searchers = self.resource._searchers_by_path.values()
        self.resource._searchers_by_path = {}

        full_list = []
        for searcher_list in searchers:
            full_list.extend(searcher_list)

        dfrs = []
        # filter them: each only once
        for searcher in set(full_list):
            dfrs.append(searcher.clean())

        return defer.DeferredList(dfrs)

    def _patch_default(self, old_result):
        # we overwrite the default_searcher function to speed up the process
        self.resource._default_searcher = None


class TestSearchers(GenericSetup, UnloadMixin, TestCase):
    resource_class = SearchMetaresourceProvider
    config = {'default_searcher' : ''}

    def setUp(self):
        # don't load the pkg_resources
        SearchMetaresourceProvider._load_searchers = lambda x,y: x
        dfr = super(TestSearchers, self).setUp()
        # Whatever is automatically loaded, unload it to have a
        # clean component.
        dfr.addCallback(self._unload_searchers)
        dfr.addCallback(self._patch_default)
        return dfr

    def test_searcher_cleaned(self):
        """
        Adds the L{BitterSearcher} into the searchers list and tears down to
        see if the resource provider is cleaning it up properly.
        """
        def created(searcher):
            self.resource._searchers_by_path['test'] = [searcher]

        dfr = BitterSearcher.create({})
        dfr.addCallback(created)
        return dfr

    def test_searchers_called(self):

        uri = MediaUri('elisa://search/music/madonna')
        self.resource._searchers_by_path['music'] = []
        searchers = []

        def created(searcher):
            self.resource._searchers_by_path['music'].append(searcher)
            searchers.append(searcher)

        def iterate():
            for i in xrange(3):
                dfr = DummySearcher.create({})
                dfr.addCallback(created)
                yield dfr

        def run_test(result):
            model, dfr = self.resource.get(uri)
            return dfr

        def check(result_model):
            for searcher in searchers:
                self.assertEquals(len(searcher.searches), 1)
                req_uri, model = searcher.searches[0]
                self.assertTrue(uri is req_uri)
                self.assertTrue(model is result_model)

        dfr = task.coiterate(iterate())
        dfr.addCallback(run_test)
        dfr.addCallback(check)
        return dfr

    def test_only_default_searchers_called(self):

        uri = MediaUri('elisa://search/music/madonna?only_default=True')
        self.resource._searchers_by_path['music'] = []
        searchers = []

        def created(searcher):
            self.resource._searchers_by_path['music'].append(searcher)
            searchers.append(searcher)

        def iterate():
            for i in xrange(3):
                dfr = DummySearcher.create({})
                dfr.addCallback(created)
                yield dfr

        def run_test(result):
            model, dfr = self.resource.get(uri)
            return dfr

        def set_default_searcher(result):
            def set_def(default):
                self.resource._default_searcher = default

            dfr = DummySearcher.create({})
            dfr.addCallback(set_def)
            return dfr

        def check(result_model):
            # only the default searcher was asked
            default = self.resource._default_searcher
            self.assertEquals(len(default.searches), 1)

            # and no one else
            for searcher in searchers:
                self.assertEquals(len(searcher.searches), 0)

        dfr = task.coiterate(iterate())
        dfr.addCallback(set_default_searcher)
        dfr.addCallback(run_test)
        dfr.addCallback(check)
        return dfr

    def test_only_selected_searchers(self):

        uri = MediaUri('elisa://search/music/madonna?' \
                'only_searchers=searcher0,searcher2')
        self.resource._searchers_by_path['music'] = []
        searchers = []

        def created(searcher, name):
            self.resource._searchers_by_path['music'].append(searcher)
            searcher.name = name
            searchers.append(searcher)

        def iterate():
            for i in xrange(3):
                dfr = DummySearcher.create({})
                dfr.addCallback(created, "searcher%s" % i)
                yield dfr

        def run_test(result):
            model, dfr = self.resource.get(uri)
            return dfr

        def set_default_searcher(result):
            def set_def(default):
                default.name = "searcher-1"
                self.resource._default_searcher = default

            dfr = DummySearcher.create({})
            dfr.addCallback(set_def)
            return dfr

        def check(result_model):
            # the default searcher was not asked
            default = self.resource._default_searcher
            self.assertEquals(len(default.searches), 0)

            # because only searcher0 and searcher2 are asked
            s0, s1, s2 = searchers    
            self.assertEquals(len(s0.searches), 1)
            self.assertEquals(len(s1.searches), 0)
            self.assertEquals(len(s2.searches), 1)

        dfr = task.coiterate(iterate())
        dfr.addCallback(set_default_searcher)
        dfr.addCallback(run_test)
        dfr.addCallback(check)
        return dfr

    def test_only_selected_searchers_with_default(self):

        uri = MediaUri('elisa://search/music/madonna?' \
                'only_searchers=searcher0,searcher2,searcher-1')
        self.resource._searchers_by_path['music'] = []
        searchers = []

        def created(searcher, name):
            self.resource._searchers_by_path['music'].append(searcher)
            searcher.name = name
            searchers.append(searcher)

        def iterate():
            for i in xrange(3):
                dfr = DummySearcher.create({})
                dfr.addCallback(created, "searcher%s" % i)
                yield dfr

        def run_test(result):
            model, dfr = self.resource.get(uri)
            return dfr

        def set_default_searcher(result):
            def set_def(default):
                default.name = "searcher-1"
                self.resource._default_searcher = default

            dfr = DummySearcher.create({})
            dfr.addCallback(set_def)
            return dfr

        def check(result_model):
            # the default searcher was asked, because we requested it
            default = self.resource._default_searcher
            self.assertEquals(len(default.searches), 1)

            # because only searcher0 and searcher2 are asked
            s0, s1, s2 = searchers    
            self.assertEquals(len(s0.searches), 1)
            self.assertEquals(len(s1.searches), 0)
            self.assertEquals(len(s2.searches), 1)

        dfr = task.coiterate(iterate())
        dfr.addCallback(set_default_searcher)
        dfr.addCallback(run_test)
        dfr.addCallback(check)
        return dfr

    def test_only_selected_and_only_default(self):

        uri = MediaUri('elisa://search/music/madonna?' \
                'only_searchers=a,b,c&only_default=True')
        model, dfr = self.resource.get(uri)
        self.assertFailure(dfr, TypeError)
        return dfr

    def test_cancellable(self):

        uri = MediaUri('elisa://search/music/coldplay')
        self.resource._searchers_by_path['music'] = []
        searchers = []

        def created(searcher):
            self.resource._searchers_by_path['music'].append(searcher)
            searchers.append(searcher)

        def iterate():
            for i in xrange(3):
                dfr = HangingSearcher.create({})
                dfr.addCallback(created)
                yield dfr

        def run_test(result):
            model, dfr = self.resource.get(uri)
            # wait some time before cancelling the deferred
            reactor.callLater(1.0, dfr.cancel)
            return dfr

        def check(failure):
            for searcher in searchers:
                self.assertTrue(searcher.cancelled)
            return None

        dfr = task.coiterate(iterate())
        dfr.addCallback(run_test)
        dfr.addErrback(check)
        return dfr
    test_cancellable.timeout = 5


class SimpleSearcher(object):
    def clean(self):
        return defer.succeed(None)

class TestSearchersAdding(GenericSetup, UnloadMixin, TestCase):
    resource_class = SearchMetaresourceProvider
    config = {'default_searcher' : 'default'}

    def setUp(self):
        dfr = super(TestSearchersAdding, self).setUp()
        # what ever there is automatically loaded, we want a clear component, so
        # we unload it
        dfr.addCallback(self._unload_searchers)
        dfr.addCallback(self._patch_default)
        return dfr

    def test_add_searcher(self):
        """
        Test if the usual adding of a normal searcher works as expected
        """
        this_searcher = SimpleSearcher()
        this_searcher.paths = ['test', 42]

        self.resource._add_searcher(this_searcher, 'cool_name')

        searchers = self.resource._searchers_by_path
        self.assertEquals(set(searchers.keys()), set(['test', 42]))
        self.assertEquals(len(searchers['test']), 1)
        self.assertEquals(len(searchers[42]), 1)
        self.assertTrue(searchers['test'][0] is this_searcher)
        self.assertTrue(searchers[42][0] is this_searcher)

    def test_add_searcher_with_uppercase_name(self):
        """
        As the searcher filtering is using lowercase only, the name has to be
        lowercased
        """
        this_searcher = SimpleSearcher()
        this_searcher.paths = ['test', 42]

        self.resource._add_searcher(this_searcher, 'UpperName')
        self.assertEquals(this_searcher.name, 'uppername')

    def test_add_another_searcher(self):

        first_searcher = SimpleSearcher()
        another_searcher = SimpleSearcher()
        self.assertEquals(self.resource._searchers_by_path, {})
        self.resource._searchers_by_path['33'] = [first_searcher]
        self.resource._searchers_by_path['88'] =  \
                    [another_searcher, first_searcher]

        new_searcher = SimpleSearcher()
        new_searcher.paths = ['33', 'test']

        self.resource._add_searcher(new_searcher, 'another_name')

        searchers = self.resource._searchers_by_path
        self.assertEquals(set(searchers.keys()), set(['33','88','test']))
        self.assertEquals(len(searchers['test']), 1)
        self.assertEquals(len(searchers['88']), 2)
        self.assertEquals(len(searchers['33']), 2)
        self.assertTrue(searchers['test'][0] is new_searcher)

        self.assertEquals(searchers['33'], [first_searcher, new_searcher])
        self.assertEquals(searchers['88'],
                [another_searcher, first_searcher])

    def test_adding_default_searcher(self):
        """
        Test if adding the default searcher works as expected
        """
        default = SimpleSearcher()
        default.paths = ['test', 42]

        self.resource._add_searcher(default, 'default')

        searchers = self.resource._searchers_by_path
        self.assertEqual(searchers, {})
        self.assertTrue(self.resource._default_searcher is default)

