# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import component
from elisa.plugins.poblesec.widgets.player.status_display import VideoStatusDisplay,\
                                                        BARS
from elisa.plugins.poblesec.widgets.player.control_ribbon import ControlRibbon,\
                                                                 Control
from elisa.plugins.poblesec.widgets.player.button import Button
from elisa.core.utils.cancellable_defer import CancelledError

from elisa.plugins.poblesec.music_library import AlbumsViewMode
from elisa.plugins.database.models import MusicTrack
from elisa.plugins.base.models.audio import TrackModel as BaseTrackModel
from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal

from elisa.plugins.poblesec.videoplayer_controls import PlayPauseControl, \
                                                        StopControl, \
                                                        SkipPreviousControl, \
                                                        SkipNextControl, \
                                                        VolumeUpControl, \
                                                        VolumeDownControl

from elisa.core.input_event import *
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation
import platform

try:
    from elisa.plugins.database.models import File as DBFile
except ImportError:
    # database missing
    DBFile = None

from elisa.core import common

_ = install_translation('poblesec')

from elisa.plugins.base.models.media import PlayableModel

from elisa.core.log import Loggable
import gobject

from twisted.internet import reactor, defer
from pgm.timing import implicit
from pgm import imaging
import pgm, time
import gst
from gtk import gdk
from gst import pbutils
import math
import os
import platform

class VisualisationError(Exception):
    pass

# The Visualisation Fake
class ElisaVisualisationBin(gst.Bin):

    def __init__(self, visualisation, size, pgm_image):
        """
        The ElisaVisualisationBin is a wrapper for the different visualisation
        elements of GStreamer.

        @param visualisation:   the GStreamer Element to use
        @type visualisation:    string
        @param size:            a size in pixel to calculate the visualisation
                                size for
        @type size:             int
        @param pgm_image:       the pgm_image that is used for sinking
        @type pgm_image:         L{pgm.graph.image.Image}
        """

        gst.Bin.__init__(self)

        self._visualisation = visualisation

        self._capsfilter = gst.element_factory_make('capsfilter')
        self.add(self._capsfilter)

        caps_pad = self._capsfilter.get_pad('src')

        self._visu = gst.element_factory_make(self._visualisation)
        self.add(self._visu)

        height = int(round(size * pgm_image.absolute_height / pgm_image.absolute_width))
        height = (height + 3) & ~3

        visu_sink = self._visu.get_pad('sink')
        visu_src = self._visu.get_pad('src')

        caps = visu_src.get_pad_template().caps

        if caps == None:
            raise VisualisationError("The Visualisation element %s does not" \
                                     " have any Caps." % self._visualisation)

        caps = caps.copy()

        if caps.is_fixed() == False:
            caps.make_writable()
            for i in xrange(0, caps.get_size()):
                cur_struct = caps.get_structure(i)

                cur_struct.fixate_field_nearest_int('width', size)

                cur_struct.fixate_field_nearest_int('height', height)
        else:
            raise VisualisationError("The Caps of the GstElement are fixed.")

        self._capsfilter.set_property('caps', caps)

        self._ghost_sink = gst.GhostPad('sink', visu_sink )
        self._ghost_src = gst.GhostPad('src', caps_pad)

        self._visu.link(self._capsfilter)
        self.add_pad(self._ghost_sink)
        self.add_pad(self._ghost_src)


class MouseOsd(Widget):

    def __init__(self):
        super(MouseOsd, self).__init__()

        # back button to leave the player
        self.back_button = Button()
        self.add(self.back_button)
        self.back_button.set_glyphs(
                          "elisa.plugins.poblesec.player.back_button.normal",
                          "elisa.plugins.poblesec.player.back_button.selected",
                          "elisa.plugins.poblesec.player.back_button.active")
        self.back_button.bg_a = 0
        self.back_button.position = (0.03, 0.02, 0.0)
        self.back_button.size = (0.09, 0.05)
        self.back_button.visible = True

        # animation facilities
        self.animated = implicit.AnimatedObject(self)
        settings = {'duration': 350,
                    'transformation': implicit.DECELERATE}
        self.animated.setup_next_animations(**settings)
        self.animated.mode = implicit.REPLACE

    def hide(self):
        """
        DOCME
        """
        self.animated.opacity = 0

    def show(self):
        """
        DOCME
        """
        self.animated.opacity = 255

class PlayerOsd(Widget):
    """
    On screen display widget for the player.

    Signals:
      - 'shown': emitted when the widget is displayed.
      - 'hidden': emitted when the widget is hidden.
    """

    __gsignals__ = {'shown':
                    (gobject.SIGNAL_RUN_LAST,
                     gobject.TYPE_BOOLEAN,
                     ()),
                    'hidden':
                    (gobject.SIGNAL_RUN_LAST,
                     gobject.TYPE_BOOLEAN,
                     ())}

    status_widget = VideoStatusDisplay

    def __init__(self):
        super(PlayerOsd, self).__init__()

        self.status = self.status_widget()
        self.add(self.status)
        self.status.visible = True

        pbc = self.status.status_display_details.progress_bars
        pbc.buffering_bar_label.label = _('Buffering...')

        self.control_ribbon = ControlRibbon()
        self.add(self.control_ribbon)
        self.control_ribbon.opacity = 0
        self.control_ribbon.visible = True

        # mouse on screen display only shown on mouse move
        self.mouse_osd = MouseOsd()
        self.add(self.mouse_osd)
        self.mouse_osd.position = (0.0, 0.0, 0.0)
        self.mouse_osd.size = (1.0, 1.0)
        self.mouse_osd.opacity = 0
        self.mouse_osd.visible = True

        # cover overlay
        self.cover_overlay = Image()
        self.cover_overlay.bg_color = (0, 0, 0, 120)
        self.cover_overlay.layout = pgm.IMAGE_ZOOMED
        self.add(self.cover_overlay)
        self.cover_overlay.size = (1.0, 1.0)
        self.cover_overlay.position = (0.0, 0.0, -0.1)
        self.cover_overlay.visible = True

        #create animation stuff
        self.animated = implicit.AnimatedObject \
                                                (self)
        settings = {'duration': 350,
                    'transformation': implicit.DECELERATE}
        self.animated.setup_next_animations(**settings)
        self.animated.mode = implicit.REPLACE

        self.opacity = 0
        # FIXME: for some reason the opacity is not taken into account for the
        # ribbon: it is always visible. The following hack fixes it for an
        # unknown reason.
        self.animated.opacity = 0

        self.time_before_hiding = 3.0
        self._hiding_dfr = None

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(PlayerOsd, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'status-display-x':
                self.status.x = value
            elif key == 'status-display-y':
                self.status.y = value
            elif key == 'status-display-height':
                self.status.height = value
            elif key == 'status-display-width':
                self.status.width = value
            elif key == 'control-ribbon-display-x':
                self.control_ribbon.x = value
            elif key == 'control-ribbon-display-y':
                self.control_ribbon.y = value
            elif key == 'control-ribbon-display-height':
                self.control_ribbon.height = value
            elif key == 'control-ribbon-display-width':
                self.control_ribbon.width = value

    def hide(self):
        """
        Hide the OSD.
        """
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()
        self.status.status_display_details.progress_bars.set_sensitive(False)
        self.animated.opacity = 0
        self.control_ribbon.hide()
        self.mouse_osd.hide()
        self.emit('hidden')

    def show(self, timeout=True):
        """
        Show the OSD.

        @param timeout: whether or not the OSD will hide automatically after
                        L{time_before_hiding} seconds
        @type timeout:  C{bool}
        """
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()
        self.status.status_display_details.progress_bars.set_sensitive(True)
        self.animated.opacity = 255
        if self.time_before_hiding and timeout:
            self._hiding_dfr = reactor.callLater(self.time_before_hiding, self.hide)

        self.emit('shown')

class Player(gobject.GObject, Loggable):

    STOPPED = 0
    PLAYING = 1
    PAUSED = 2
    BUFFERING = 3

    __gsignals__ = {'status-changed':
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_INT,)),
                    'volume-changed':
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_FLOAT,)),
                    'playback-ended' :
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_NONE,
                              ()),
                    'player-missing-decoder':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               (gobject.TYPE_STRING, gobject.TYPE_STRING)),
                    'player-codec-error':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               ()),
                    'player-unknown-error':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               (gobject.TYPE_STRING,))
                              }

    def __init__(self, config):
        super(Player, self).__init__()
        # GStreamer pipeline setup

        self.config = config

        # make sure typefindfunctions is loaded so we can remove the typefinders
        if platform.system() == 'Windows':
            gst.plugin_load_by_name('typefindfunctions')
            registry = gst.registry_get_default()
            typefinders = gst.type_find_factory_get_list()
            for typefinder in typefinders:
                if typefinder.get_name() in ('application/x-ape',
                    'application/x-apetag'):
                    registry.remove_feature(typefinder)

        self.pgm_sink = gst.element_factory_make('pgmimagesink')
        self.pipeline = gst.element_factory_make('playbin')
        self.pipeline.set_property('video-sink', self.pgm_sink)

        self.status = self.STOPPED
        self.last_error = None
        self.current_index = -1
        self.filename = None
        self.image = None
        self.volume_max = 2.0
        self.volume_increment = 0.10
        self.volume_decrement = 0.30
        self.seek_backward_seconds = 60
        self.seek_forward_seconds = 60
        self.key_repeat_count_second = 0.150
        self._last_key_time = time.time()
        self.playlist = []

        pbus = self.pipeline.get_bus()
        pbus.connect('message::eos', self._message_eos_cb)
        pbus.connect('message::error', self._message_error_cb)
        pbus.connect('message::state-changed', self._message_state_changed_cb)
        pbus.connect('message::buffering', self._message_buffering_cb)
        pbus.connect('message::element', self._message_element_cb)
        pbus.add_signal_watch()

        self._play_deferreds = []

    def _message_eos_cb(self, bus, message):
        self.pipeline.set_state(gst.STATE_NULL)
        self.play_next()

    def _message_buffering_cb(self, bus, message):
        self.status = self.BUFFERING
        self.emit('status-changed', self.status)

    def _message_error_cb(self, bus, message):
        err, msg = message.parse_error()
        self.last_error = (err, msg)
        self.warning("Gstreamer %s:%s" % (err, msg))
        self.pipeline.set_state(gst.STATE_NULL)
        code = message.structure['gerror'].code
        if code == gst.STREAM_ERROR_CODEC_NOT_FOUND:
            self.emit('player-codec-error')
        elif code != gst.STREAM_ERROR_FAILED:
            self.emit('player-unknown-error', err)
        else:
            self.emit('player-codec-error')

    def _message_state_changed_cb(self, bus, message):
        old_state, new_state, pending = message.parse_state_changed()

        if message.src != self.pipeline:
            return
        if new_state == gst.STATE_PLAYING:
            if self.status != self.PLAYING:
                self.status = self.PLAYING
                self.emit('status-changed', self.status)
        elif new_state == gst.STATE_PAUSED:
            if self.status != self.PAUSED:
                self.status = self.PAUSED
                self.emit('status-changed', self.status)
        elif self.status != self.STOPPED:
            self.status = self.STOPPED
            self.emit('status-changed', self.status)

    def _message_element_cb(self, bus, message):
        if pbutils.is_missing_plugin_message(message):
            caps =  message.structure['detail']
            details = caps.to_string().split(', ')
            mimetype = details[0]
            decoder = pbutils.get_decoder_description(mimetype)
            self.emit('player-missing-decoder', mimetype, decoder)

    def _load_subs(self, uri):
        found = False
        if uri.scheme == 'file':
            basename, ext = os.path.splitext(uri.filename)
            for sub_ext in ('srt', 'ssa', 'sub', 'txt'):
                sub_file = os.path.join(os.path.dirname(uri.path),
                                        "%s.%s" % (basename, sub_ext))
                if os.path.exists(sub_file):
                    sub_uri = MediaUri("file://%s" % sub_file)
                    self.pipeline.set_property('suburi', sub_uri)
                    self.pipeline.set_property('subtitle-font-desc', "Liberation Sans")
                    self.info("Loaded subtitles at %r", sub_uri)
                    found = True
                    break
        if not found:
            self.info("No subtitles found for %r", uri)
            self.pipeline.set_property('suburi', '')


    def set_drawable(self, value):
        self.image = value
        self.pgm_sink.set_property('image', value)

    def enqueue_to_playlist(self, playable_model):
        self.playlist.append(playable_model)

    def play_next(self):
        playlist_length = len(self.playlist)
        if self.current_index < playlist_length-1:
            self.play_at_index(self.current_index+1)
            return True
        else:
            if self.status != self.STOPPED:
                self.stop()
            self.emit('playback-ended')
            self.stop()
            return False

    def play_previous(self):
        if self.current_index > 0 and len(self.playlist) > 0:
            self.play_at_index(self.current_index-1)
            return True
        return False

    def play_model(self, playable_model):
        # clear all the media that have never been played from the playlist
        self.playlist[self.current_index+1:] = []
        self.playlist.append(playable_model)
        self.play_at_index(len(self.playlist)-1)

    def _update_playcount_and_time(self, model):
        if model.uri.scheme != 'file':
            # database only support files
            return

        def update(result):
            if not result:
                return

            self.debug("updating playcount")
            result.playcount += 1
            result.last_played = time.time()

        store = common.application.store

        if not store or not DBFile:
            return

        dfr = store.get(DBFile, model.uri.path)
        dfr.addCallback(update)
        return dfr
    
    def get_current_model(self):
        return self.playlist[self.current_index]

    def play_at_index(self, index):
        if self.current_index == index:
            if self.status != self.PLAYING:
                self.play()
        else:
            for dfr in self._play_deferreds:
                if hasattr(dfr, 'cancel'):
                    dfr.cancel()

            def remove_from_play_deferreds(result, deferred):
                self._play_deferreds.remove(deferred)
                return result

            model = self.playlist[index]
            if not isinstance(model, PlayableModel):
                dfr = model.get_playable_model()
                self._play_deferreds.append(dfr)
                dfr.addBoth(remove_from_play_deferreds, dfr)
            else:
                dfr = defer.succeed(model)

            dfr.addCallback(self._play_playable_model_at_index, index)

    def _play_playable_model_at_index(self, model, index):
        self.filename = model.title or model.uri.filename
        self.stop()
        self.pipeline.set_property('uri', model.uri)
        self._load_subs(model.uri)

        self.update_visualization()
        self.update_audiosink()

        self.play()
        self._update_playcount_and_time(model)
        self.current_index = index

    def update_visualization(self):
        # retrieve infos from config and configure the pipeline
        vis_plugin_name = self.config['visualization']
        if vis_plugin_name:
            try:
                visu = ElisaVisualisationBin(vis_plugin_name, 400, self.image)
            except gst.ElementNotFoundError:
                self.warning("Failed to set audio visualization %r", vis_plugin_name)
            else:
                self.pipeline.set_property('vis-plugin', visu)
        # TODO: code to disable the vis-plugin on-the-fly

    def update_audiosink(self):   
        audio_sink = self.config.get('audio_sink')
        if audio_sink:
            try:
                self.audio_sink = gst.element_factory_make(audio_sink)
            except gst.ElementNotFoundError:
                self.warning("Can't find audio sink '%s'" % audio_sink)
            else:
                self.pipeline.set_property('audio-sink', self.audio_sink)

    def stop(self):
        current = self.pipeline.get_state(0)[1]
        self.pipeline.set_state(gst.STATE_NULL)
        # when setting the state to gst.STATE_NULL the pipeline doesn't emit a
        # "message::state-changed" signal.
        if current != gst.STATE_NULL:
            self.status = self.STOPPED
            self.emit('status-changed', self.status)
        if self.image:
            self.image.clear()

    def pause(self):
        self.pipeline.set_state(gst.STATE_PAUSED)

    def play(self):
        current = self.pipeline.get_state(0)[1]
        if current != gst.STATE_PLAYING:
            if current != gst.STATE_PAUSED:
                self.status = self.BUFFERING
                self.emit('status-changed', self.status)
            self.pipeline.set_state(gst.STATE_PLAYING)

    def get_position(self):
        try:
            position, format = self.pipeline.query_position(gst.FORMAT_TIME)
            return position
        except gst.QueryError:
            return -1

    def set_position(self, value):
        if self.status not in (self.PLAYING, self.PAUSED):
            return

        event = self.pipeline.seek(1.0, gst.FORMAT_TIME,
                            gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_KEY_UNIT,
                            gst.SEEK_TYPE_SET, value,
                            gst.SEEK_TYPE_SET, -1 )

        duration = self.get_duration()
        if duration != -1 and value >= duration:
            self.play_next()

    def get_duration(self):
        try:
            duration, format = self.pipeline.query_duration(gst.FORMAT_TIME)
            return duration
        except gst.QueryError:
            return -1

    def _can_handle_key(self):
        #FIXME Florian will not like that
        t = time.time()
        if (t - self._last_key_time) > self.key_repeat_count_second:
            self._last_key_time = t
            return True
        return False

    def get_volume(self):
        return self.pipeline.get_property('volume')

    def set_volume(self, volume):
        self.pipeline.set_property('volume', volume)
        self.emit('volume-changed', volume)

    def volume_up(self):
        volume = self.get_volume()
        if self._can_handle_key():
            volume += self.volume_increment
            if volume > self.volume_max:
                volume = self.volume_max
            self.set_volume(volume)
        return volume

    def volume_down(self):
        volume = self.get_volume()
        if self._can_handle_key():
            volume -= self.volume_decrement
            if volume < 0:
                volume = 0
            self.set_volume(volume)
        return volume

    def seek_backward(self):
        duration = self.get_duration()
        position = self.get_position()
        if position != -1 and duration > 0 and self._can_handle_key():
            position -= self.seek_backward_seconds * gst.SECOND
            if position < 0:
                # the user wants to seek to the previous track
                position = 0
                if self.play_previous():
                    return position
            self.set_position(position)
        return position

    def seek_forward(self):
        duration = self.get_duration()
        position = self.get_position()
        if position != -1 and duration > 0 and self._can_handle_key():
            position += self.seek_forward_seconds * gst.SECOND
            self.set_position(position)
            return position
        return -1

def player_default_config():
    if platform.system() == 'Windows':
        visualization = 'goom2k1'
    else:
        visualization = 'libvisual_jess'
    default_config = {'visualization': visualization,
                      'osd_timeout': '3',
                      'audio_sink' : '',
                      }
    return default_config

class PlayerController(PigmentController):
    
    PlayerClass = Player

    config_doc = {'visualization': 'GStreamer visualization element to use for '\
                                   'audio animations display',
                  'osd_timeout': 'Time in seconds after which the OSD should '\
                                 'auto-hide. Put 0 to deactive the OSD auto-hide '\
                                 'feature',
                  'audio_sink' : 'give the name of the gstreamer audio sink' \
                                 'elisa should use. If not set autoaudiosink'\
                                 'will be used.'
                  }
    default_config = player_default_config()
    
    def __init__(self):
        super(PlayerController, self).__init__()

        self._current_osd_dfr = None

        # FIXME: mute support should probably be moved to Player
        # volume before muting
        self.mute_volume = -1
        self.volume_display_factor = 10.0
        
        self.seek_interval_seconds = 0.200
        self.seek_last_time = time.time()

        self._delayed_call = None

        self.background = Image()
        self.widget.add(self.background)
        self.background.bg_color = (0, 0, 0, 0)
        # FIXME: is that -1.0 necessary?
        self.background.z = -1.0
        self.background.visible = True

        self.player_osd = PlayerOsd()

        self.widget.add(self.player_osd)
        self.player_osd.size = (1.0, 1.0)
        self.player_osd.position = (0.0, 0.0, 0.0)
        self.player_osd.bg_a = 0
        self.player_osd.opacity = 0
        self.player_osd.visible = True
        self.player_osd.connect('hidden', self._stop_monitoring)


        self.player = self.PlayerClass(None)
        self.player.set_drawable(self.background)
        self.player.connect('status-changed',self._player_status_cb)
        
        self.codec_installer = None

        #Initialise the volume control initial values
        self.player_osd.status.set_volume_max \
                        (self.player.volume_max * self.volume_display_factor)
        self.player_osd.status.set_volume_position \
                        (self.player.get_volume() * self.volume_display_factor)
        
        #connect the seeking bar and volume bar
        self.player_osd.status.connect \
                                ('seeking-changed', self._seeking_bar_cb)
        self.player_osd.status.connect \
                                ('volume-changed', self._volume_bar_cb)


        # connect to the back button
        self.player_osd.mouse_osd.back_button.connect('clicked',
                                                      self._exit_clicked_cb)

        self._missing_decoders = set()
        self.player.connect('player-missing-decoder', self._player_missing_decoder_cb)
        self.player.connect('player-codec-error', self._check_missing_decoders_cb)

        self.player.connect('player-unknown-error', self._player_unknown_error_cb)

        self.player.connect('playback-ended', self._playback_ended)

        # video controls
        ribbon = self.player_osd.control_ribbon
        ribbon.add_control(SkipPreviousControl(self))
        ribbon.add_control(PlayPauseControl(self))
        ribbon.add_control(StopControl(self))
        ribbon.add_control(SkipNextControl(self))
        ribbon.add_control(VolumeDownControl(self))
        ribbon.add_control(VolumeUpControl(self))

        self.album_helper = AlbumsViewMode()

        self._widget_focus_id = self.widget.connect('focus',
                                                    self._on_focus_changed)

    def initialize(self):
        dfr = super(PlayerController, self).initialize()

        config = self.config
        if config is None:
           config = {}

        self.player.config = self.config

        # we want OSD dock always there if no visualization is displayed
        osd_timeout = int(config.get('osd_timeout', '3'))

        self.player_osd.time_before_hiding = osd_timeout

        return dfr


    def clean(self):
        dfr = super(PlayerController, self).clean()
        self.widget.disconnect(self._widget_focus_id)
        if self._current_osd_dfr is not None:
            self._current_osd_dfr.cancel()
        return dfr

    def _on_focus_changed(self, widget, focus):
        if focus and self.player.status in (self.player.PAUSED, self.player.PLAYING):
            self.show_on_screen_display()
        else:
            self.hide_on_screen_display()

    def _exit_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.exit()
        return True

    def exit(self):
        """
        Exit the player by asking poblesec's main controller to hide it.
        """
        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        self.hide_on_screen_display()
        main.hide_current_player()

    def _play_pause_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_play_pause()
        return True

    def _seeking_bar_cb(self, widget, position):
        
        current_time = time.time()
        if current_time - self.seek_last_time < self.seek_interval_seconds:
            return
        
        self.seek_last_time = current_time
        self.player.set_position(position * gst.SECOND)

    def _volume_bar_cb(self, widget, position):
        self.mute_volume = -1
        self.player.set_volume(position / self.volume_display_factor)

    def _volume_mute_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_mute()
        return True

    def _clear_osd(self):
        osd_details = self.player_osd.status.status_display_details
        osd_details.title.label = ""
        osd_details.details.label = ""
        osd_details.details_2.label = ""
        icon = 'elisa.plugins.poblesec.file'
        self.frontend.load_from_theme(icon, self.player_osd.status.preview)
        self.player_osd.cover_overlay.clear()
        self.background.clear()

    def _player_status_cb(self, player, status):
        if status != player.PLAYING:
            self._stop_monitoring()

        # show buffering even if focus is not set
        if status == player.BUFFERING:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.BUFFERING_BAR)
            self.show_on_screen_display()
        elif status == player.STOPPED:
            # just clean up
            self._clear_osd()
            if self._current_osd_dfr is not None:
                self._current_osd_dfr.cancel()
            return

        else:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)

        if not self.has_focus():
            return
        
        osd_details = self.player_osd.status.status_display_details
        if self.player.filename != None:
            
            model  = self.player.get_current_model()
                
            #Current file is a music
            if isinstance(model, MusicTrack) or isinstance(model, BaseTrackModel):
                if osd_details.title.label != model.title:

                    osd_details.title.label = model.title
                    
                    def dfr_error(failure, source):
                        if failure.type != CancelledError:
                            self.warning("failure in %s : %s" % (source, failure))
                        self._current_osd_dfr = None
                    
                    def got_artwork(file_path):
                        if file_path:
                            self.player_osd.status.preview.set_from_file(file_path)
                            src = gdk.pixbuf_new_from_file(file_path)
                            dst = imaging.linear_alpha_gradient(src, 0.25, 0.0, 1.3,
                                                                0.75, 0.75, 0.0)
                            self.player_osd.cover_overlay.set_from_pixbuf(dst)

                        self._current_osd_dfr = None

                    def got_artist(artist, album):
                        osd_details.details_2.label = artist
                        dfr = self.album_helper.get_image(album, None)
                        dfr.addCallback(got_artwork)
                        dfr.addErrback(dfr_error, "in got_artwork")
                        return dfr
    
                    def got_album(album):
                        if album:
                            osd_details.details.label = album.name
                            dfr = album.get_artist_name()
                            dfr.addCallback(got_artist, album)
                            dfr.addErrback(dfr_error, "in got_artist")
                            return dfr

                        self._current_osd_dfr = None

                    icon = 'elisa.plugins.poblesec.player.thumbnail.default_album'
                    self.frontend.load_from_theme \
                                        (icon, self.player_osd.status.preview)
                    self.player_osd.cover_overlay.clear()
                    osd_details.details.label = ""
                    osd_details.details_2.label = ""
                    album_dfr = model.get_album()
                    album_dfr.addCallback(got_album)
                    album_dfr.addErrback(dfr_error, "in got_album")
                    self._current_osd_dfr = album_dfr

            #Current file is a movie
            elif osd_details.title.label != self.player.filename:
                                
                osd_details.title.label = self.player.filename
                
                duration = self.player.get_duration()
                if duration == -1:
                    osd_details.details.label = ""
                elif osd_details.details.label == "":
                    duration_in_min = duration / (gst.SECOND * 60)
                    osd_details.details.label = "%s %s" % \
                                                 (duration_in_min, _('minutes'))

                osd_details.details_2.label = ""
                icon = 'elisa.plugins.poblesec.player.thumbnail.default_movie'
                self.frontend.load_from_theme \
                                        (icon, self.player_osd.status.preview)
                
        if status == player.PLAYING:
            #self.player_osd.status.pause()
            self.show_on_screen_display()

            self._monitor()
            self.player_osd.status.set_seeking_duration \
                                        (self.player.get_duration()/gst.SECOND)

            self._check_missing_decoders_cb(self.player, partly_playable=True)
        elif status == player.PAUSED:
            #self.player_osd.status.play()
            self.show_on_screen_display()
        return True

    def _stop_monitoring(self, *args):
        self.debug('Stop monitoring')
        # stop the monitoring instantly
        if self._delayed_call and self._delayed_call.called == 0 and \
            self._delayed_call.cancelled == 0:
            self._delayed_call.cancel()

    def _monitor(self):
        if self._delayed_call and self._delayed_call.called == 0 and \
            self._delayed_call.cancelled == 0:
            # still a pending one
            return

        self.debug('Monitoring')

        progress_bars = self.player_osd.status.status_display_details.progress_bars
        if not progress_bars.seeking_bar.seek_drag:
            # no update during drag seeking to not flicker around
            player_position = self.player.get_position()
            self.player_osd.status.set_seeking_position \
                                                (player_position / gst.SECOND)

        self._delayed_call = reactor.callLater(0.10, self._monitor)

    def _player_missing_decoder_cb(self, player, mimetype, decoder):
        self._missing_decoders.add(decoder)
            
    def _check_missing_decoders_cb(self, player, partly_playable=False):
        if self._missing_decoders:
            missing_decoders = list(self._missing_decoders)
            self._missing_decoders.clear()
            main = self.frontend.retrieve_controllers('/poblesec')[0]
            if not partly_playable:
                self.hide_on_screen_display()
                main.hide_current_player()

            def missing_decoders_popup():
                icon = 'elisa.plugins.poblesec.warning'
                title = _('Unable To Play Selected File')

                if partly_playable:
                    text = _('Unfortunately Elisa can only partly play the selected file.')
                else:
                    text = _('Unfortunately Elisa cannot play the selected file.')
                if missing_decoders:
                    text += _(' Please download and install: ')
                    # Localization of the names of the missing decoders is
                    # automatically taken care of by gstreamer.
                    text += _(', ').join(missing_decoders) + _('.')

                buttons = [(_('Close'), main.hide_popup)]

                main.enqueue_popup(icon, title, text, buttons)

            if platform.system().lower() == 'windows':
                if self._codec_installer_setup():
                    self._show_codec_installer_popup(self.codec_installer, player.playlist[0].uri)
                else:
                    missing_decoders_popup()
            else:
                missing_decoders_popup()

    def _player_unknown_error_cb(self, player, message):
        if platform.system().lower() == 'windows':
            def missing_decoders_popup():
                main = self.frontend.retrieve_controllers('/poblesec')[0]
                icon = 'elisa.plugins.poblesec.warning'
                title = _('Unable To Play Selected File')
                text = _('Unfortunately Elisa cannot play the selected file.')
                buttons = [(_('Close'), main.hide_popup)]

                main.enqueue_popup(icon, title, text, buttons)

            if self._codec_installer_setup():
                self._show_codec_installer_popup(self.codec_installer, player.playlist[0].uri)
            else:
                missing_decoders_popup()
            return
            
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        self.hide_on_screen_display()
        main.hide_current_player()

        icon = 'elisa.plugins.poblesec.warning'
        title = _('Unable To Play Selected File')

        text = _('Unfortunately Elisa cannot play the selected file.')
        text += _(' Gstreamer error: ') + message

        buttons = [(_('Close'), main.hide_popup)]

        main.enqueue_popup(icon, title, text, buttons)

    def _playback_ended(self, player):
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        self.hide_on_screen_display()
        self.background.clear()
        main.hide_current_player()

    def set_frontend(self, frontend):
        super(PlayerController, self).set_frontend(frontend)
        self.frontend.viewport.connect('motion-notify-event', \
                                                        self._mouse_motion_cb)

    def _mouse_motion_cb(self, viewport, event):
        if self.has_focus():
            if self.player_osd.opacity == 0:
                pbc = self.player_osd.status.status_display_details.progress_bars
                if pbc.get_visible_bar() != BARS.SEEKING_BAR:
                    pbc.set_visible_bar(BARS.SEEKING_BAR)
            self.show_on_screen_display(with_ribbon=True, with_mouse=True)

    def toggle_play_pause(self):
        if self.player.status == self.player.PLAYING:
            self.go_to_pause()
        else:
            self.go_to_play()

    def go_to_pause(self):
        self.player.pause()

    def go_to_play(self):
        self.player.play()

    def toggle_mute(self):
        if self.mute_volume >= 0:
            self.show_on_screen_display()
            self.player.set_volume(self.mute_volume)
            self.mute_volume = -1
        else:
            self.show_on_screen_display()
            self.mute_volume = self.player.get_volume()
            self.player.set_volume(0)

    def volume_up(self):
        pbc = self.player_osd.status.status_display_details.progress_bars
        pbc.set_visible_bar(BARS.VOLUME_BAR)
        self.show_on_screen_display()
        volume = self.player.volume_up() * self.volume_display_factor
        self.player_osd.status.set_volume_position(int(volume))

    def volume_down(self):
        pbc = self.player_osd.status.status_display_details.progress_bars
        pbc.set_visible_bar(BARS.VOLUME_BAR)
        self.show_on_screen_display()
        volume = self.player.volume_down() * self.volume_display_factor
        self.player_osd.status.set_volume_position(int(volume))

    def show_on_screen_display(self, with_ribbon=False, with_mouse=False):
        """
        Display player's on screen display. Optionally displays ribbon
        controls and mouse controls.

        @param with_ribbon: whether or not to display ribbon controls
        @type with_ribbon:  C{bool}
        @param with_mouse:  whether or not to display mouse controls
        @type with_mouse:   C{bool}
        """
        if self.player.status == self.player.PAUSED:
            self.player_osd.show(timeout=False)
        elif self.player.status == self.player.PLAYING:
            self._monitor()
            self.player_osd.show(timeout=True)
        else:
            self.player_osd.show(timeout=True)

        if with_ribbon:
            self.player_osd.control_ribbon.show()
        if with_mouse:
            self.player_osd.mouse_osd.show()

    def hide_on_screen_display(self):
        """
        Hide player's on screen display including ribbon controls and mouse
        controls.
        """
        self.player_osd.hide()

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_MENU:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            if self.player_osd.animated.opacity == 0:
                self.show_on_screen_display(with_ribbon=True)
                return True
            elif self.player_osd.control_ribbon.animated.opacity == 0:
                self.show_on_screen_display(with_ribbon=True)
                return True
            else:
                self.hide_on_screen_display()
                return False
        elif input_event.value == EventValue.KEY_OK or \
             input_event.value == EventValue.KEY_SPACE:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.control_ribbon.selected_control.activate()
            else:
                self.toggle_play_pause()
            return True
        elif input_event.value == EventValue.KEY_PLAY:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            if self.player.status == self.player.PAUSED:
                self.go_to_play()
            return True
        elif input_event.value == EventValue.KEY_PAUSE:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            if self.player.status == self.player.PLAYING:
                self.go_to_pause()
            return True
        elif input_event.value == EventValue.KEY_GO_UP:
            self.volume_up()
            return True
        elif input_event.value == EventValue.KEY_GO_DOWN:
            self.volume_down()
            return True
        elif input_event.value == EventValue.KEY_GO_LEFT:
            if self.player_osd.control_ribbon.opacity == 0:
                pbc = self.player_osd.status.status_display_details.progress_bars
                pbc.set_visible_bar(BARS.SEEKING_BAR)
            self.show_on_screen_display()
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.control_ribbon.select_previous_control()
            else:
                self.player.seek_backward()
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            if self.player_osd.control_ribbon.opacity == 0:
                pbc = self.player_osd.status.status_display_details.progress_bars
                pbc.set_visible_bar(BARS.SEEKING_BAR)
            self.show_on_screen_display()
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.control_ribbon.select_next_control()
            else:
                self.player.seek_forward()
        elif input_event.value == EventValue.KEY_x or \
             input_event.value == EventValue.KEY_PREVIOUS:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            self.show_on_screen_display()
            self.player.play_previous()
        elif input_event.value == EventValue.KEY_c or \
             input_event.value == EventValue.KEY_NEXT:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.SEEKING_BAR)
            self.show_on_screen_display()
            self.player.play_next()
        elif input_event.value == EventValue.KEY_m:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.VOLUME_BAR)
            self.show_on_screen_display()
            self.toggle_mute()

        return False
        
    def _codec_installer_setup(self):
        from mswin32 import tools
        if tools.get_windows_codecs_installed_value() != 1:
            if not self.codec_installer:
                codec_installer = []
                codec_installer.append(tools.get_elisa_installation_path())
                if len(codec_installer[0]) != 0:
                    codec_installer.append('\\')
                codec_installer.append('codec_installer.exe')
                self.codec_installer = ''.join(codec_installer)
            return True
        return False
        
    def _show_codec_installer_popup(self, codec_installer_executable, file):
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        icon = 'elisa.plugins.poblesec.warning'
        title = _('Unable To Play Selected File')
        text = _("Elisa is not able to play '%s' due to a missing codec. To install this playback codec please click on 'Install'.")
        text = text % file

        def ok_clicked():
            from win32api import ShellExecute, CloseHandle
            import pywintypes

            try:
                handle = ShellExecute(0, 'open', codec_installer_executable, '', None, 1)
                CloseHandle(handle)
            except pywintypes.error, error:
                main.warning(error)
                main.hide_popup()
            
            main.hide_popup()
            main.frontend.viewport.iconified = True

        buttons = [(_('Install'), ok_clicked), (_('Cancel'), main.hide_popup)]
        main.enqueue_popup(icon, title, text, buttons)

class VideoPlayerController(PlayerController):
    pass
