# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
A configurable on-screen keyboard.
"""

from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.group import Group

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.widgets.theme import Theme

import gobject

import pgm
from pgm.timing import implicit

from xml.dom import minidom

import os.path

import logging


class Key(object):

    """
    One given key of a virtual keyboard.
    A key has a type that determines its behaviour when activated:
     - L{Key.CHAR}: a normal character
     - L{Key.SWITCH}: switch between caps
     - L{Key.DELETE}: backspace
     - L{Key.PLACEHOLDER}: an invisible key
     - L{Key.MISC}: a key with a miscellaneous behaviour
    """

    CHAR = 'char'
    SWITCH = 'switch'
    DELETE = 'delete'
    PLACEHOLDER = 'placeholder'
    MISC = 'misc'

    def __init__(self, ktype, kwidth, kvalues=None, label=None, image=None):
        """
        Constructor.

        @param ktype:   the type of the key (one of (L{Key.CHAR},
                        L{Key.SWITCH}, L{Key.DELETE}, L{Key.PLACEHOLDER},
                        L{Key.MISC}))
        @type ktype:    C{str}
        @param kwidth:  the relative width of the key on the row (must be
                        between 0.0 and 1.0, the sum of the widths of the key
                        on one row should not exceed 1.0)
        @type kwidth:   C{float}
        @param kvalues: the value of the key (depends on its type).
                         - if ktype is L{Key.CHAR}, then kvalues is a C{dict}
                           associating each cap to a string value
                         - if ktype is L{Key.SWITCH}, then kvalues is a C{list}
                           of caps to cycle through when switching
                         - if ktype is L{Key.DELETE}, then kvalues is ignored
                         - if ktype is L{Key.PLACEHOLDER}, then kvalues is
                           ignored
                         - if ktype is L{Key.MISC}, then kvalues is a
                           miscellaneous string value
        @type kvalues:  C{dict} of C{str}, or C{list} of C{str}, or C{None}, or
                        C{str}
        @param label:   the label to display on the graphical representation of
                        the key
        @type label:    C{str}
        @param image:   an optional image resource to display on the graphical
                        representation of the key
        @type image:    C{str}
        """
        self.ktype = ktype
        self.kwidth = kwidth # 0.0 < kwidth < 1.0
        self.kvalues = kvalues
        self.label = label
        self.image = image

    def to_xml(self, document):
        """
        Dump the key to an XML node.

        @param document: the XML document for which to generate the node
        @type document:  L{xml.dom.minidom.Document}

        @return: an XML node representing the key
        @rtype:  L{xml.dom.minidom.Element}
        """
        node = document.createElement('key')
        node.setAttribute('type', self.ktype)
        node.setAttribute('width', '%.2f' % self.kwidth)
        if self.kvalues is not None:
            if self.ktype == Key.CHAR:
                for key, value in self.kvalues.iteritems():
                    node.setAttribute(key, value)
            elif self.ktype == Key.SWITCH:
                node.setAttribute('caps', ', '.join(self.kvalues))
            elif self.ktype == Key.MISC:
                node.setAttribute('value', self.kvalues)
        if self.label is not None:
            node.setAttribute('label', self.label)
        if self.image is not None:
            node.setAttribute('image', self.image)
        return node

    @classmethod
    def from_xml(cls, key_node, caps):
        """
        Instantiate a key from its XML representation.

        @param key_node: the XML representation of the key
        @type key_node:  L{xml.dom.minidom.Element}
        @param caps:     the list of caps of the keyboard
        @type caps:      C{list} of C{str}

        @return:         a key
        @rtype:          L{Key}
        """
        attr = key_node.attributes
        ktype = attr['type'].nodeValue
        kwidth = float(attr['width'].nodeValue)
        kvalues = None
        if ktype == cls.CHAR:
            kvalues = {}
            for cap in caps:
                kvalues[cap] = attr[cap].nodeValue
        elif ktype == cls.SWITCH:
            kvalues = attr['caps'].nodeValue.split(', ')
        elif ktype == cls.MISC:
            kvalues = attr['value'].nodeValue
        label = None
        if attr.has_key('label'):
            label = attr['label'].nodeValue
        image = None
        if attr.has_key('image'):
            image = attr['image'].nodeValue
        key = cls(ktype, kwidth, kvalues, label, image)
        return key


class Button(Widget):

    """
    A button widget used to represent a given key in a keyboard.
    """

    def __init__(self, text='', image_resource=None):
        """
        Constructor.

        @param text:           the optional contents of the label
        @type text:            C{str}
        @param image_resource: an optional image resource
        @type image_resource:  C{str}
        """
        super(Button, self).__init__()

        # Add a background drawable to receive events
        self._background = Image()
        self.add(self._background)
        self._background.visible = True
        self._background.bg_color = (0, 0, 0, 0)

        self._label = Text()
        self.add(self._label, forward_signals=False)
        self._label.fg_color = (255, 255, 255, 255)
        self._label.bg_color = (0, 0, 0, 0)
        self._label.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._label.alignment = pgm.TEXT_ALIGN_CENTER
        self._label.weight = pgm.TEXT_WEIGHT_BOLD
        self._label.height = 0.25

        if image_resource is None:
            self._label.width = 0.7
            self._label.position = (0.15, 0.375, 0.0)
            self._image = None
        else:
            self._image = Image()
            self.add(self._image, forward_signals=False)
            self._image.bg_color = (0, 0, 0, 0)
            self._image.layout = pgm.IMAGE_SCALED
            self._image.alignment = pgm.IMAGE_LEFT
            theme = Theme.get_default()
            image_file = theme.get_resource(image_resource)
            self._image.set_from_file(image_file)

            def image_loaded(image):
                self._image.disconnect(self._loaded_id)
                self._image.height = 0.25
                real_aspect_ratio = self._image.aspect_ratio[0] / float(self._image.aspect_ratio[1])
                current_aspect_ratio = self._image.absolute_width / self._image.absolute_height
                deformation = real_aspect_ratio / current_aspect_ratio
                self._image.width *= deformation
                self._image.position = (0.15, 0.375, 0.0)
                self._image.visible = True
                self._label.width = 0.7 - self._image.width
                self._label.position = (0.15 + self._image.width, 0.375, 0.0)

            self._loaded_id = self._image.connect('file-loaded', image_loaded)

        self._label.label = text
        self._label.visible = True

    def label__get(self):
        if self._label:
            return self._label.label
        return ''

    def label__set(self, value):
        self._label.label = value

    label = property(label__get, label__set)

    def clean(self):
        if self._background is not None:
            self._background.clean()
            self._background = None
        if self._label is not None:
            self._label.clean()
            self._label = None

        if self._image is not None:
            self._image.clean()
            self._image = None

        return super(Button, self).clean()


class OnScreenKeyboard(Widget):

    """
    A configurable on-screen keyboard widget.

    Its layout is defined in an XML file.
    It defines several caps that can be cycled through, changing the value
    emitted by each key.
    """

    __gsignals__ = {'key-press-char': (gobject.SIGNAL_RUN_LAST,
                                       gobject.TYPE_BOOLEAN,
                                       (str,)),
                    'key-press-special': (gobject.SIGNAL_RUN_LAST,
                                          gobject.TYPE_BOOLEAN,
                                          (str,)),
                   }

    def __init__(self, xml_file):
        """
        Constructor.

        @param xml_file: the path to an XML file containing the keyboard layout
        @type xml_file:  C{str}
        """
        super(OnScreenKeyboard, self).__init__()
        self.keys = []
        self.current_key = None
        self.caps = []
        self.current_caps = None

        self.keys_widget = Group()
        self.add(self.keys_widget)
        self.keys_widget.position = (0.0, 0.0, 0.0)
        self.keys_widget.size = (1.0, 1.0)
        self.keys_widget.visible = True

        self.load_layout(xml_file)

        self._signal_handler_ids = []
        self._render()
        self.set_selector()

        # Store the motions in an internal cache to avoid computing them every
        # time.
        self._motion_cache = {}
        for direction in (LEFT, RIGHT, TOP, BOTTOM):
            self._motion_cache[direction] = {}

    def clean(self):
        for widget, signal_id in self._signal_handler_ids:
            widget.disconnect(signal_id)
        self._signal_handler_ids = []

        if self._animated_selector:
            self._animated_selector.stop_animations()
            self._animated_selector = None

        if self._selector:
            self._selector.clean()
            self._selector = None

        self.keys_widget.clean()
        self.keys_widget = None
        if self.current_key:
            self.current_key = None
        self.keys = []
        return super(OnScreenKeyboard, self).clean()

    def to_xml(self):
        """
        Dump the keyboard to an XML document.

        @return: an XML document representating the keyboard
        @rtype:  L{xml.dom.minidom.Document}
        """
        implementation = minidom.getDOMImplementation()
        document = implementation.createDocument(None, 'osk', None)
        document.documentElement.setAttribute('caps', ', '.join(self.caps))
        for row in self.keys:
            row_node = document.createElement('row')
            document.documentElement.appendChild(row_node)
            for key in row:
                key_node = key.to_xml(document)
                row_node.appendChild(key_node)
        return document

    def load_layout(self, xml_file):
        """
        Load the layout of the keyboard from an XML file.

        @param xml_file: the path to an XML file containing the keyboard layout
        @type xml_file:  C{str}
        """
        dom = minidom.parse(xml_file)
        osk = dom.firstChild
        self.caps = osk.attributes['caps'].nodeValue.split(', ')
        self.current_caps = self.caps[0]
        rows = osk.getElementsByTagName('row')
        for row in rows:
            self.keys.append([])
            keys = row.getElementsByTagName('key')
            for key_node in keys:
                key = Key.from_xml(key_node, self.caps)
                self.keys[-1].append(key)
        self.current_key = self.keys[0][0]

    def _render(self):
        """
        Render the layout of the keyboard.
        """
        for row in self.keys:
            bx = 0.0
            for key in row:
                if key.ktype == Key.PLACEHOLDER:
                    bx += self.keys_widget.width * key.kwidth
                    continue
                if not hasattr(key, 'widget') or key.widget is None:
                    button = Button(image_resource=key.image)
                    self.keys_widget.add(button)
                    key.widget = button
                    button_clicked_id = button.connect('clicked', self._button_clicked_cb, key)
                    self._signal_handler_ids.append((button, button_clicked_id))
                button = key.widget
                label = ''
                if key.label is not None:
                    label = key.label
                elif key.ktype == Key.CHAR:
                    label = key.kvalues[self.current_caps]
                button.label = label
                bwidth = self.keys_widget.width * key.kwidth
                bheight = self.keys_widget.height / len(self.keys)
                button.size = (bwidth, bheight)
                by = bheight * self.keys.index(row)
                button.position = (bx, by, 0.0)
                button.visible = True
                bx += bwidth

    def set_selector(self):
        """
        Set the selector widget of the keyboard.

        This one is a default ugly selector. This method should be overwritten
        by subclasses to provide a better looking selector.
        """
        self._selector = Image()
        self.add(self._selector)
        self._selector.bg_color = (100, 200, 135, 150)
        self._selector.position = self.current_key.widget.position
        self._selector.size = self.current_key.widget.size
        self._animated_selector = implicit.AnimatedObject(self._selector)
        settings = {'duration': 300,
                    'transformation': implicit.DECELERATE,
                    'resolution': 5}
        self._animated_selector.setup_next_animations(**settings)

    def do_focus(self, focus):
        self._animated_selector.visible = focus

    def _switch(self):
        index = (self.caps.index(self.current_caps) + 1) % len(self.caps)
        self.current_caps = self.caps[index]
        self._render()

    def _button_clicked_cb(self, button, x, y, z, mbutton, time, data, key):
        self.focus = True
        if key != self.current_key:
            self.select_key(key)
        self.activate_key(key)

    def move_selector(self, direction):
        """
        Move the selector in the specified direction.

        @param direction: where to move the cursor
        @type direction:  one of (L{elisa.plugins.pigment.widgets.const.LEFT},
                                  L{elisa.plugins.pigment.widgets.const.RIGHT},
                                  L{elisa.plugins.pigment.widgets.const.TOP},
                                  L{elisa.plugins.pigment.widgets.const.BOTTOM})
        """
        if self._motion_cache[direction].has_key(self.current_key):
            # The motion to perform is already in the cache
            self.select_key(self._motion_cache[direction][self.current_key])
            return

        # FIXME: store the coordinates of the currently selected key?
        current_row = None
        for row in self.keys:
            for key in row:
                if key == self.current_key:
                    current_row = row
                    break
            if current_row is not None:
                break

        real_row = [key for key in current_row if key.ktype != Key.PLACEHOLDER]

        if direction in (LEFT, TOP):
            delta = -1
        elif direction in (RIGHT, BOTTOM):
            delta = 1

        if direction in (LEFT, RIGHT):
            index = (real_row.index(self.current_key) + delta) % len(real_row)
            new_key = real_row[index]
        elif direction in (TOP, BOTTOM):
            # Compute the new key in the new row based on the X coordinates of
            # the keys in the rows.
            current_x = (self.current_key.widget.x + self.current_key.widget.width / 2) / self.keys_widget.width
            new_row = self.keys[(self.keys.index(current_row) + delta) % len(self.keys)]
            while len([k for k in new_row if k.ktype != Key.PLACEHOLDER]) == 0:
                # The row is just one big placeholder, jump to the next one
                new_row = self.keys[(self.keys.index(new_row) + delta) % len(self.keys)]
            if new_row is current_row:
                # The keyboard has only one row containing real keys
                return
            bx = 0.0
            for key in new_row:
                if (bx <= current_x) and ((bx + key.kwidth) >= current_x):
                    if key.ktype == Key.PLACEHOLDER:
                        index = new_row.index(key)
                        left = [k for k in new_row[:index] if k.ktype != Key.PLACEHOLDER]
                        right = [k for k in new_row[index:] if k.ktype != Key.PLACEHOLDER]
                        if len(left) == 0:
                            new_key = right[0]
                        elif len(right) == 0:
                            new_key = left[-1]
                        else:
                            lkey = left[-1]
                            lkey_x = sum([k.kwidth for k in new_row[new_row.index(lkey):]])
                            ldelta = current_x - (lkey_x + lkey.kwidth)
                            rkey = right[0]
                            rkey_x = sum([k.kwidth for k in new_row[new_row.index(rkey):]])
                            rdelta = rkey_x - current_x
                            if ldelta <= rdelta:
                                new_key = lkey
                            else:
                                new_key = rkey
                    else:
                        new_key = key
                    break
                bx += key.kwidth

        self._motion_cache[direction][self.current_key] = new_key
        self.select_key(new_key)

    def select_key(self, key):
        """
        Move the selector to a given key on the keyboard.

        @param key: the new key to select
        @type key:  L{Key}
        """
        self.current_key = key
        self._animated_selector.size = key.widget.size
        self._animated_selector.position = key.widget.position

    def activate_key(self, key):
        """
        Activate a key on the keyboard.

        @param key: the key to activate
        @type key:  L{Key}
        """
        if key.ktype == Key.CHAR:
            self.emit('key-press-char', key.kvalues[self.current_caps])
        elif key.ktype == Key.DELETE:
            self.emit('key-press-special', 'delete')
        elif key.ktype == Key.MISC:
            self.emit('key-press-special', key.kvalues)
        elif key.ktype == Key.SWITCH:
            self._switch()

    def do_key_press_event(self, viewport, event, widget):
        """
        Default handler for the key-press event.

        Can be overwritten in subclasses for specific behaviours.
        """
        if event.keyval == pgm.keysyms.Return:
            self.activate_key(self.current_key)
        elif event.keyval == pgm.keysyms.Left:
            self.move_selector(LEFT)
        elif event.keyval == pgm.keysyms.Right:
            self.move_selector(RIGHT)
        elif event.keyval == pgm.keysyms.Up:
            self.move_selector(TOP)
        elif event.keyval == pgm.keysyms.Down:
            self.move_selector(BOTTOM)

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        xml_file = os.path.join(os.path.dirname(__file__),
                                'data', 'osk_alphabetic.xml')
        widget = cls(xml_file)
        widget.visible = True
        widget.focus = True

        def on_key_press(osk, value):
            logging.info('OSK key-press: %s' % value)

        widget.connect('key-press-char', on_key_press)
        widget.connect('key-press-special', on_key_press)

        return widget

    @classmethod
    def _set_demo_widget_defaults(cls, widget, canvas, viewport):
        Widget._set_demo_widget_defaults(widget, canvas, viewport)
        widget.size = (4.0, 3.0)
        widget.regenerate()


if __name__ == '__main__':
    gobject.threads_init()
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)

    osk = OnScreenKeyboard.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()
    osk.clean()
    #FIXME: why on earth do we need this?
    import time
    time.sleep(0.1)
