# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>

"""
Box widgets.
"""

from elisa.extern import enum
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.group import Group
from elisa.plugins.pigment.graph.drawable import Drawable

import pgm

from twisted.internet import reactor

PACKING = enum.Enum("START", "END")
ALIGNMENT = enum.Enum("START", "CENTER", "END")


class WidgetNotBoxedError(Exception):
    """
    Raised when trying to deal with a widget as a child of the box, even if
    it is not.
    """


class Child(object):
    """
    Child encapsulates all the necessary information for a L{Box} to render
    one of its children.

    @ivar widget:  widget to render in the parent box
    @type widget:  L{elisa.plugins.pigment.widgets.widget.Widget}
    @ivar packing: START if child was packed using pack_start, END if child
                   was packed using pack_end
    @type packing: L{PACKING}
    @ivar expand:  value of expand passed at packing time
    @type expand:  bool
    """

    def __init__(self, widget, packing, expand):
        self.widget = widget
        self.packing = packing
        self.expand = expand

class Box(Widget):
    """
    Box is an abstract widget that defines a specific kind of container that
    layouts a variable number of widgets into a rectangular area. The former
    is organized into either a single row or a single column of child widgets
    depending upon whether the box is of type L{HBox} or L{VBox}, respectively.

    Use repeated calls to gtk_box_pack_start to pack widgets from start to end.
    Use gtk_box_pack_end to add widgets from end to start.

    @ivar spacing:   amount of space between children
    @type spacing:   int
    @ivar alignment: defines where the children are positioned in the
                     rectangular area of the box
    @type alignment: L{ALIGNMENT}
    """

    def __init__(self):
        super(Box, self).__init__()
        self._layout_call = None
        self._spacing = 0
        self._start_packed_children = []
        self._end_packed_children = []
        self._alignment = ALIGNMENT.CENTER

        # list of signal ids to be disconnected on cleanup
        self._signals_id = []

        id = self.connect("resized", self._resized_callback)
        self._signals_id.append(id)
        id = self.connect("repositioned", self._repositioned_callback)
        self._signals_id.append(id)

        # dictionary of signal ids per child to be disconnected on cleanup
        # key:  child widget
        # valu: list of signal ids
        self._children_signal_ids = {}

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Box, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'spacing':
                self.spacing = value
            elif key == 'alignment':
                style_to_enum = {"start": ALIGNMENT.START,
                                 "center": ALIGNMENT.CENTER,
                                 "end": ALIGNMENT.END}
                self.alignment = style_to_enum[value]

    def _queue_layout(self):
        if self._layout_call is not None:
            return

        def layout():
            self._layout_call = None
            self._layout()

        self._layout_call = reactor.callLater(0, layout)

    def _resized_callback(self, notifier, width, height):
        self._queue_layout()

    def _repositioned_callback(self, notifier, x, y, z):
        self._queue_layout()

    def _child_resized_callback(self, notifier, width, height):
        self._queue_layout()

    def _child_changed_callback(self, notifier, property):
        # FIXME: I could not find the enum value pgm.DRAWABLE_SIZE of
        # PgmDrawableProperty; it is therefore hardcoded here
        # As to revision 1296, Pigment Python has the value and we are waiting
        # for a release before making the modification here.
        PGM_DRAWABLE_SIZE = 1
        if property == PGM_DRAWABLE_SIZE:
            self._queue_layout()

    def _disconnect_child(self, child):
        for id in self._children_signal_ids.get(child, []):
            child.widget.disconnect(id)

    def clean(self):
        super(Box, self).clean()

        # disconnect from signals of the box
        for id in self._signals_id:
            self.disconnect(id)

        # disconnect from signals of the children
        for child in self._children_signal_ids:
            self._disconnect_child(child)

    def pack_start(self, widget, expand=False):
        """
        Add L{widget} to the box packed after any other widget packed using
        pack_start. Visually L{widget} will be positioned after any other
        widget packed that way.

        @param widget: widget to pack in the box
        @type widget:  L{elisa.plugins.pigment.widgets.widget.Widget}
        @param expand: True if widget is to be given extra space allocated to box.
                       The extra space will be divided evenly between all
                       widgets of box that use this option.
        @type expand:  bool
        """
        child = Child(widget, PACKING.START, expand)
        self._start_packed_children.append(child)
        self._insert_new_child(child)

    def pack_end(self, widget, expand=False):
        """
        Add L{widget} to the box packed after any other widget packed using
        pack_end. Visually L{widget} will be positioned before any other
        widget packed that way.

        @param widget: widget to pack in the box
        @type widget:  L{elisa.plugins.pigment.widgets.widget.Widget}
        @param expand: True if widget is to be given extra space allocated to box.
                       The extra space will be divided evenly between all
                       widgets of box that use this option.
        @type expand:  bool
        """
        child = Child(widget, PACKING.END, expand)
        self._end_packed_children.insert(0, child)
        self._insert_new_child(child)

    def remove(self, widget):
        super(Box, self).remove(widget)

        for i, child in enumerate(self._start_packed_children):
            if child.widget == widget:
                self._remove_packed_child(i, self._start_packed_children)
                self._queue_layout()
                return

        for i, child in enumerate(self._end_packed_children):
            if child.widget == widget:
                self._remove_packed_child(i, self._end_packed_children)
                self._queue_layout()
                return

        raise WidgetNotBoxedError()

    def pop(self):
         child = None

         if self._end_packed_children:
             child = self._remove_packed_child(0, self._end_packed_children)
         elif self._start_packed_children:
             child = self._remove_packed_child(-1, self._start_packed_children)

         if child:
             super(Box, self).remove(child.widget)
             self._queue_layout()

         return child

    def _remove_packed_child(self, idx, child_list):
        child = child_list.pop(idx)

        if child in self._children_signal_ids:
            self._disconnect_child(child)
            del self._children_signal_ids[child]

        return child

    def __len__(self):
        return len(self._start_packed_children) + len(self._end_packed_children)

    def __contains__(self, widget):
        for child in self._start_packed_children:
            if child.widget == widget:
                return True

        for child in self._end_packed_children:
            if child.widget == widget:
                return True

        return False

    def _insert_new_child(self, child):
        self.add(child.widget)
        self._queue_layout()

        # connect to 'resized' signal of the child in order to relayout when
        # the child's size changes
        # FIXME: deals with two different cases because signals API are
        # different for Groups and Drawables
        if not child.expand:
            widget = child.widget
            if isinstance(widget, Group):
                id = widget.connect("resized", self._child_resized_callback)
                self._children_signal_ids[child] = [id]
            elif isinstance(widget, Drawable):
                id = widget.connect("changed", self._child_changed_callback)
                self._children_signal_ids[child] = [id]

    def spacing__get(self):
        return self._spacing

    def spacing__set(self, spacing):
        self._spacing = spacing
        self._queue_layout()

    spacing = property(spacing__get, spacing__set)

    def alignment__get(self):
        return self._alignment

    def alignment__set(self, alignment):
        self._alignment = alignment
        self._queue_layout()

    alignment = property(alignment__get, alignment__set)

    def _layout(self):
        raise NotImplementedError("To be implemented by classes inheriting of \
                                   Box")

    def _prelayout_children(self, children, property):
        """
        Compute the size of children packed with expand set to True and the
        coordinate at which the first widget should be positioned depending on
        the number of children, their expand mode and the box alignment.

        @param children: children for which to compute the value 
        @type children:  list of L{Child}
        @param property: one of 'width', 'height'
        @type property:  str

        @rtype: tuple of float
        """
        # total amount of spacing needed
        total_spacing = self.spacing*(len(children))

        # compute size of children in expand mode = remaining space divided
        # equally among them after taking off other children and spacing
        non_expand_children = []
        non_expand_children_size = 0
        for child in children:
            if child.expand:
                continue

            non_expand_children.append(child)
            non_expand_children_size += getattr(child.widget, property)

        len_expand_children = len(children)-len(non_expand_children)
        if len_expand_children > 0:
            expand_children_size = 1.0-non_expand_children_size-total_spacing
            expand_child_size = expand_children_size/len_expand_children
        else:
            expand_children_size = 0.0
            expand_child_size = 0.0

        # compute 'start_coordinate' initial value depending on the alignment of the
        # box
        if self.alignment == ALIGNMENT.START:
            start_coordinate = self.spacing
        else:
            children_size = non_expand_children_size+expand_children_size

            if self.alignment == ALIGNMENT.END:
                start_coordinate = 1.0-children_size-total_spacing
            elif self.alignment == ALIGNMENT.CENTER:
                start_coordinate = (1.0-children_size-total_spacing)/2.0

        return start_coordinate, expand_child_size


    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        from elisa.plugins.pigment.graph.image import Image

        widget = cls()
        widget.spacing = 0.01
        widget.alignment = ALIGNMENT.END
        widget.visible = True

        widget.position = (1.0, 1.0, 0.0)
        widget.size = (1.0, 1.0)

        # red debugging background
        background = Image()
        background.bg_color = (255, 0, 0, 255)
        widget.add(background)
        background.position = (0.0, 0.0, 0.0)
        background.size = (1.0, 1.0)
        background.visible = True

        # children start packed with expand set to True
        # their background is green
        for i in xrange(2):
            image = Image()
            image.bg_color = (55, 255, 55, 255)
            image.visible = True
            widget.pack_start(image, expand=True)

        # children end packed with expand set to True
        # their background is blue
        for i in xrange(1):
            image = Image()
            image.bg_color = (55, 55, 255, 255)
            image.visible = True
            widget.pack_end(image, expand=True)

        # children start packed with expand set to False
        # their background is white
        for i in xrange(4):
            image = Image()
            image.bg_color = (255, 255, 255, 255)
            image.width /= 10.0
            image.height /= 10.0
            image.visible = True
            widget.pack_start(image)

        return widget

class HBox(Box):
    """
    A L{Box} layouting its children horizontally.
    """
    def _layout(self):
        # merge all children in layout order
        children = self._start_packed_children + self._end_packed_children
        accumulator, expand_child_width = self._prelayout_children(children,
                                                                   "width")

        # traverse the children layouting each of them
        # 'accumulator' is used to store the horizontal translation applied to
        # the children as we are traversing them
        for index, child in enumerate(children):
            if child.widget.height != 1.0:
                child.widget.height = 1.0

            child.widget.position = (accumulator, 0.0, 0.0)
            if child.expand:
                child.widget.width = expand_child_width
            accumulator += child.widget.width + self.spacing


class VBox(Box):
    """
    A L{Box} layouting its children vertically.
    """

    def _layout(self):
        # merge all children in layout order
        children = self._start_packed_children + self._end_packed_children
        accumulator, expand_child_width = self._prelayout_children(children,
                                                                   "height")
        # traverse the children layouting each of them
        # 'accumulator' is used to store the vertical translation applied to
        # the children as we are traversing them
        for index, child in enumerate(children):
            if child.widget.width != 1.0:
                child.widget.width = 1.0
            child.widget.position = (0.0, accumulator, 0.0)
            if child.expand:
                child.widget.height = expand_child_width
            accumulator += child.widget.height + self.spacing

if __name__ == '__main__':
    import logging

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    widget = HBox.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()
