# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Gernot Klimscha <gernot@fluendo.com>

"""
Retrieve artist pictures from Discogs.com
"""

from elisa.core.components.resource_provider import ResourceProvider
from elisa.core.media_uri import MediaUri

from elisa.plugins.http_client.http_client import ElisaAdvancedHttpClient, \
                                                  ElisaHttpClientNotOpenedError
from twisted.web2.stream import BufferedStream

# models
from elisa.plugins.base.models.audio import ArtistModel
from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.models.media import RawDataModel

from twisted.internet import task

from elisa.plugins.http_client.extern.client_http import ClientRequest
from elisa.extern.coherence.et import parse_xml

import gzip, cStringIO

class DiscogsResource(ResourceProvider):

    # comment to the API - every request has to be made with API key
    api_key = '2d1c349d44'
    supported_uri = "^http://www.discogs.com"

    def __init__(self):
        super(DiscogsResource, self).__init__()
        self._client = ElisaAdvancedHttpClient('www.discogs.com', 80)

    def get(self, uri, model):
        if not model:
            if uri.path.startswith("/image"):
                model = RawDataModel()
            else:
                model = ArtistModel()

        # comment to the API - has to be set to get a correct response
        uri.set_param('f', 'xml')
        uri.set_param('api_key', self.api_key)

        def got_data(response, model):
            if not response.code == 200:
                raise Exception(response.code)

            dfr = BufferedStream(response.stream).readExactly()

            # comment to the API - if gzip encoded unzip it before parsing
            encoding = response.headers.getRawHeaders('Content-Encoding')
            if encoding and encoding[0] == 'gzip':
                dfr.addCallback(self.unzip)

            return dfr.addCallback(self.parse, model)

        request = ClientRequest('GET', str(uri), None, None)
        # comment to the API - most of the time the response will be gzipped
        request.headers.setRawHeaders('Accept-Encoding', ['gzip'])

        return model, self._client.request_full(request).addCallback(got_data, model)

    def clean(self):
        def eater(failure):
            if failure.type == ElisaHttpClientNotOpenedError:
                return None
            return failure
        dfr = self._client.close()
        dfr.addErrback(eater)

        def parent_clean(result):
            return super(DiscogsResource, self).clean()

        dfr.addCallback(parent_clean)

        return dfr
        

    def unzip(self, data):
        """
        Unzip the data from the response to be further processed in parse.

        @param data:       gzip compressed data from response
        @type data:        C{str}
        @return:           decompressed data
        @rtype:            C{str}
        """
        return gzip.GzipFile(fileobj=cStringIO.StringIO(data)).read()

    def parse(self, data, model):
        """
        Parse xml data into model.
        At the moment only the artist name and the artist pictures,
        with thumbnails, are used from the xml data.
        The parsed data is then stored in a ArtistModel which holds a
        C{list} of L{elisa.plugins.base.models.image.ImageModel}s.

        @param data:       plain xml data from response
        @type data:        C{str}
        @param model:      the ArtistModel that gets filled
        @type model:       L{elisa.plugins.base.models.audio.ArtistModel}
        @return:           a filled ArtistModel
        @rtype:            L{elisa.plugins.base.models.audio.ArtistModel}
        """
        if isinstance(model, RawDataModel):
            # raw data request
            model.data = data
            model.size = len(data)
            return model

        # parse the xml data
        et = parse_xml(data)
        if et:
            # get artist name
            artist = et.findtext(".//name")
            # clear the image list if request was made for a new artist
            if model.name != artist:
                del model.images[:]
                model.name = artist
            for images in et.getiterator("image"):
                d = ImageModel()
                d.references = [MediaUri(images.get("uri")),
                                 MediaUri(images.get("uri150"))]
                model.images.append(d)

        return model

