/*
 * Device Control Service
 * Copyright (C) 2008  Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, Intel gives permission to link
 * the code of portions of this program with the OpenSSL project's
 * "OpenSSL" library (or with modified versions of it that use the same
 * license as the "OpenSSL" library), and distribute the linked
 * executables.  You must obey the GNU General Public License in all
 * respects for all of the code used other than "OpenSSL".  If you modify
 * this file, you may extend this exception to your version of the file,
 * but you are not obligated to do so.  If you do not wish to do so,
 * delete this exception statement from your version.
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/resource.h>
#include <stdio.h>
#include <pthread.h>
#include <signal.h>
#include <stdarg.h>
#include "ConnMgmt.h"
#include "SupplierManager.h"

extern void DebugString(const char* format, ...);

extern void * ProcessRequest(void *data);

#ifdef NDEBUG
///Ensure that only one copy of a daemon is running

#define LOCKFILE "/var/run/daemon.pid"
#define LOCKMODE (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)

int lockfile(int fd)
{
    struct flock fl;

    fl.l_type = F_WRLCK;
    fl.l_start = 0;
    fl.l_whence = SEEK_SET;
    fl.l_len = 0;
    return(fcntl(fd, F_SETLK, &fl));
}

/* print a message and return to caller. caller specifies "errnoflag". */
static void err_doit(int errnoflag, const char *fmt, va_list ap)
{
   int errno_save;
   char buf[MAX_STRING_LEN];

   errno_save = errno;  /* value caller might want printed */
   vsprintf(buf, fmt, ap);
   
   if (errnoflag)
     sprintf(buf+strlen(buf),": %s", strerror(errno_save));
     
   strcat(buf, "\n");
   fflush(stdout); /* in case stdout and stderr are the same */
   fputs(buf, stderr);
   fflush(NULL); /*flushes and stdio  output streams */
   
   return;
}
/* Fatal error related to a system call . print a message and terminate. */
void err_quit(const char *fmt,...)
{
   va_list ap;
   va_start(ap, fmt);
   err_doit(0, fmt, ap);
   va_end(ap);
   exit(1);
}
int already_running(void)
{
    int     fd;
    char    buf[16];

    fd = open(LOCKFILE, O_RDWR|O_CREAT, LOCKMODE);
    if (fd < 0) {
        syslog(LOG_ERR, "can't open %s: %s", LOCKFILE, strerror(errno));
        exit(1);
    }
    if (lockfile(fd) < 0) {
        if (errno == EACCES || errno == EAGAIN) {
            close(fd);
            return(1);
        }
        syslog(LOG_ERR, "can't lock %s: %s", LOCKFILE, strerror(errno));
        exit(1);
    }
    ftruncate(fd, 0);
    sprintf(buf, "%ld", (long)getpid());
    write(fd, buf, strlen(buf)+1);
    return(0);
}

void
daemonize(const char *cmd)
{
    int                 i, fd0, fd1, fd2;
    pid_t               pid;
    struct rlimit       rl;
    struct sigaction    sa;
    /*
     * Clear file creation mask.
     */
    umask(0);

    /*
     * Get maximum number of file descriptors.
     */
    if (getrlimit(RLIMIT_NOFILE, &rl) < 0)
        err_quit("%s: can't get file limit", cmd);

    /*
     * Become a session leader to lose controlling TTY.
     */
    if ((pid = fork()) < 0)
        err_quit("%s: can't fork", cmd);
    else if (pid != 0) /* parent */
        exit(0);
    setsid();

    /*
     * Ensure future opens won't allocate controlling TTYs.
     */
    sa.sa_handler = SIG_IGN;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = 0;
    if (sigaction(SIGHUP, &sa, NULL) < 0)
        err_quit("%s: can't ignore SIGHUP");
    if ((pid = fork()) < 0)
        err_quit("%s: can't fork", cmd);
    else if (pid != 0) /* parent */
        exit(0);

    /*
     * Change the current working directory to the root so
     * we won't prevent file systems from being unmounted.
     */
    if (chdir("/") < 0)
        err_quit("%s: can't change directory to /");

    /*
     * Close all open file descriptors.
     */
    if (rl.rlim_max == RLIM_INFINITY)
        rl.rlim_max = 1024;
    for (i = 0; i < rl.rlim_max; i++)
        close(i);

    /*
     * Attach file descriptors 0, 1, and 2 to /dev/null.
     */
    fd0 = open("/dev/null", O_RDWR);
    fd1 = dup(0);
    fd2 = dup(0);

    /*
     * Initialize the log file.
     */
    openlog(cmd, LOG_PID, LOG_DAEMON);
    if (fd0 != 0 || fd1 != 1 || fd2 != 2) {
        syslog(LOG_ERR, "unexpected file descriptors %d %d %d",
          fd0, fd1, fd2);
        exit(1);
    }
}
#endif

void DelPip(int sgn)
{
        /* ignore all other signals */
        if ((SIGHUP == sgn) || (SIGINT == sgn) || (SIGQUIT == sgn) ||
                (SIGILL == sgn) || (SIGABRT == sgn) || (SIGSEGV == sgn) ||
                (SIGTERM == sgn) || (SIGTSTP == sgn) || (SIGPIPE == sgn)) {
                DebugString("the process is terminated and delete pipe file\n");
                system("rm -f /var/lock/dcs");
                exit(1);
        }
}

int main()
{
	int ret = 0;
	 /* install signal handlers to delete named pipe file */
        if ((SIG_ERR == signal(SIGHUP, DelPip)) ||
                (SIG_ERR == signal(SIGINT, DelPip)) ||
                (SIG_ERR == signal(SIGQUIT, DelPip)) ||
                (SIG_ERR == signal(SIGILL, DelPip)) ||
                (SIG_ERR == signal(SIGABRT, DelPip)) ||
                (SIG_ERR == signal(SIGFPE, DelPip)) ||
                (SIG_ERR == signal(SIGSEGV, DelPip)) ||
                (SIG_ERR == signal(SIGPIPE, DelPip)) ||
                (SIG_ERR == signal(SIGALRM, DelPip)) ||
                (SIG_ERR == signal(SIGTERM, DelPip)) ||
                (SIG_ERR == signal(SIGUSR1, DelPip)) ||
                (SIG_ERR == signal(SIGUSR2, DelPip)) ||
                (SIG_ERR == signal(SIGTSTP, DelPip)) ||
                (SIG_ERR == signal(SIGTTIN, DelPip)) ||
                (SIG_ERR == signal(SIGTTOU, DelPip))) {
                DebugString("Install signal handlers failed\n");
                return 1;
        }

	ret = mkfifo("/var/lock/dcs", S_IRUSR | S_IWUSR);
	if (0 != ret)
	{
		DebugString("there is lock file under /var/lock/, please check if dcs is already started!\n");
		return 1;
	}
#ifdef NDEBUG
	daemonize("dcs");
#endif
	SupplierManager::Instance();

	/* sleep a few second to wait plugin thread startup */
	sleep(5);

	ConnMgmt* pConnMgmt = ConnMgmt::Instance();
	int clifd;
	if (pConnMgmt == NULL)  //the socket  fails listening
		return -1;
//ignore SIGPIPE

	struct sigaction sa;
	sa.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &sa, 0);

	struct sigaction sa2;
	sa2.sa_handler = SIG_IGN;

	sigaction(SIGILL, &sa2, 0);
	
	while (1) 
	{	
		clifd = pConnMgmt->AcceptConnection();
		DebugString("server: got connection \n");

		if (clifd < 0)
		{
			DebugString("Server Accept Failed!\n");
            break;
		}

		pthread_t child_thread;
        pthread_attr_t child_thread_attr;
        pthread_attr_init(&child_thread_attr);
        pthread_attr_setdetachstate(&child_thread_attr,PTHREAD_CREATE_DETACHED);
        if( pthread_create(&child_thread,&child_thread_attr,ProcessRequest, (void *)clifd) < 0 )
            DebugString("pthread_create Failed \n");
		/* sleep a second to avoid conflict on create Policy instance */
		sleep(1);		
	}        	
	return 0;
}

