/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gio/gio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <clutter/clutter.h>

#include "launcher-config.h"
#include "launcher-defines.h"

G_DEFINE_TYPE (LauncherConfig, launcher_config, G_TYPE_OBJECT)

#define LAUNCHER_CONFIG_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_CONFIG, LauncherConfigPrivate))

#define FONT_DIR              "/desktop/gnome/interface"
#define FONT_PATH             "/desktop/gnome/interface/font_name"
#define DPI_DIR               "/desktop/gnome/font_rendering"
#define FONT_DPI              "/desktop/gnome/font_rendering/dpi"
#define LAUNCHER_DIR          "/apps/dell-launcher"
#define NEW_THEME             LAUNCHER_DIR"/clarity_theme"
#define CAT_FONT_COLOR        "/apps/dell-launcher/cat_font_color"
#define CAT_FONT_ACTIVE_COLOR "/apps/dell-launcher/cat_font_active_color"
#define ICN_FONT_COLOR        "/apps/dell-launcher/icon_font_color"
#define GRID_LINE_COLOR       "/apps/dell-launcher/grid_line_color"
#define DISABLE_SINGLE_INSTANCE "/apps/dell-launcher/disable_single_instance"

#define POULSBO "/var/tmp/poulsbo"

struct _LauncherConfigPrivate
{
  GdkScreen *screen;  

  GFile *watch_dir;
  GFileMonitor *dir_monitor;

  DBusGConnection *connection;
  DBusGProxy      *proxy;
};

enum
{
  CONFIG_CHANGED,
  RESUME_EVENT,

  LAST_SIGNAL
};
static guint _config_signals[LAST_SIGNAL] = { 0 };

static void
calculate_sizes (LauncherConfig *cfg)
{
  gint          width;
  gint          height;

  width = CSW();
  height = CSH();
  
  /* First the number of categories */
  if (!cfg->n_cats)
  {
    if (cfg->is_poulsbo)
      cfg->n_cats = 7;
    else
      cfg->n_cats = 7;
  }

  cfg->is_tiger = FALSE;
  cfg->cat_width = width / 8; /* 0.5 is for the + button */
  cfg->cat_height = cfg->cat_width * 0.80;
  cfg->cat_padding = 0; /* FIXME: is this really needed now? */
  
  cfg->iconview_cols = 6;
  cfg->iconview_rows = 3;
  cfg->iconview_padding = 0;
  cfg->iconview_width = (width/8) * 6;
  cfg->iconview_height = (width/8) * 3;

  cfg->icon_width = cfg->iconview_width / cfg->iconview_cols;
  cfg->icon_height = cfg->iconview_height / cfg->iconview_rows;
  
  if (height == 576 && cfg->is_poulsbo)
  {
    /* Tiger */
    cfg->icon_size = 52; //cfg->icon_height * 0.5;
    cfg->is_tiger = TRUE;
  }
  else if (height == 800)
  {
    cfg->icon_size = cfg->icon_height * 0.45;
  }
  else
  {
    cfg->icon_size = 52;//cfg->icon_height * 0.6;
  }
}

static int
getdec(char hexchar)
{
  if ((hexchar >= '0') && (hexchar <= '9')) return hexchar - '0';

  if ((hexchar >= 'A') && (hexchar <= 'F')) return hexchar - 'A' + 10;

  if ((hexchar >= 'a') && (hexchar <= 'f')) return hexchar - 'a' + 10;

  return -1; // Wrong character

}

static void
hex2float(const char *HexColor, gint *FloatColor)
{
  const char *HexColorPtr = HexColor;

  int i = 0;

  for (i = 0; i < 4; i++)
  {
    int IntColor = (getdec(HexColorPtr[0]) * 16) +
                   getdec(HexColorPtr[1]);

    FloatColor[i] = IntColor;
    HexColorPtr += 2;
  }
}

static gboolean
load_color (const gchar *string, ClutterColor *color)
{
  gint colors[4];

  if (!string)
  {
    color->red = 0x00;
    color->green = 0x00;
    color->blue = 0x00;
    color->alpha = 0xff;
    return FALSE;
  }

  hex2float(string, colors);
  color->red = colors[0];
  color->green = colors[1];
  color->blue = colors[2];
  color->alpha = colors[3];

  g_debug ("%s =%d %d %d %d", string, color->red, color->green, color->blue, color->alpha);

  return TRUE;
}

static void
set_dpi (gfloat dpi, LauncherConfig *config)
{
  ClutterBackend *backend = clutter_get_default_backend ();

  clutter_backend_set_resolution (backend, dpi);

  g_debug ("Font dpi: %f", dpi);
}

static void
set_font (const gchar *font_name, LauncherConfig *config)
{
  PangoFontDescription *desc;
  gchar *temp;

  desc = pango_font_description_from_string (font_name);

  pango_font_description_set_weight (desc, PANGO_WEIGHT_BOLD);

  temp = pango_font_description_to_string (desc);

  memcpy (config->font_name, temp, strlen (temp)+1);

  g_debug ("Font changed: %s\n", config->font_name);

  pango_font_description_free (desc);
}

/* Callbacks */
static void
on_font_changed (GConfClient    *client,
                 guint           cid,
                 GConfEntry     *entry,
                 LauncherConfig *config)
{
  GConfValue *value = NULL;

  value = gconf_entry_get_value (entry);

  set_font (gconf_value_get_string (value), config);

  g_signal_emit (config, _config_signals[CONFIG_CHANGED], 0);
}

static void
on_font_dpi_changed (GConfClient    *client,
                     guint           cid,
                     GConfEntry     *entry,
                    LauncherConfig *config)
{
  GConfValue *value = NULL;

  value = gconf_entry_get_value (entry);

  set_dpi (gconf_value_get_float (value), config);

  g_signal_emit (config, _config_signals[CONFIG_CHANGED], 0);
}


/*
 * Watch for resume
 */
static gboolean
clean_up_dbus (LauncherConfig *cfg)
{
  g_object_unref (cfg->priv->proxy);
  dbus_g_connection_unref (cfg->priv->connection);

  return FALSE;
}

static void
on_session_active_changed (DBusGProxy     *proxy, 
                           gboolean        active, 
                           LauncherConfig *cfg)
{
  g_debug ("ActiveChanged: %s", active ? "Active" : "Inactive");
  
  if (active)
  {
    dbus_g_proxy_disconnect_signal (proxy, "ActiveChanged", 
                                    G_CALLBACK (on_session_active_changed), 
                                    cfg);
    g_signal_emit (cfg, _config_signals[RESUME_EVENT], 0);
    g_idle_add ((GSourceFunc)clean_up_dbus, cfg);
  }
}
/* GObject functions */
static void
launcher_config_dispose (GObject *object)
{
  LauncherConfig *config = LAUNCHER_CONFIG (object);
  LauncherConfigPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_CONFIG (config));
  priv = LAUNCHER_CONFIG (config)->priv;

  g_object_unref (priv->dir_monitor);
  g_object_unref (priv->watch_dir);

  g_free (config->add); config->add = NULL;
  g_free (config->catbg); config->catbg = NULL;
  g_free (config->catbgv); config->catbgv = NULL;
  g_free (config->catbarbg); config->catbarbg = NULL;
  g_free (config->iconviewbg); config->iconviewbg = NULL;
  g_free (config->options); config->options = NULL;
  g_free (config->info); config->info = NULL;
  g_free (config->sep); config->sep = NULL;
  g_free (config->scrollbar); config->scrollbar = NULL;
  g_free (config->handle); config->handle = NULL;

  G_OBJECT_CLASS (launcher_config_parent_class)->dispose (object);
}

static void
launcher_config_class_init (LauncherConfigClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->dispose = launcher_config_dispose;

	_config_signals[CONFIG_CHANGED] =
		g_signal_new ("config-changed",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, config_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);

	_config_signals[RESUME_EVENT] =
		g_signal_new ("resume-event",
        G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, resume_event),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);


 
  g_type_class_add_private (obj_class, sizeof (LauncherConfigPrivate)); 
}

static void
launcher_config_init (LauncherConfig *config)
{
  LauncherConfigPrivate *priv;
  GConfClient           *client = gconf_client_get_default ();
  const gchar           *env;
  gint                   on_poulbo = 0;
  DBusGConnection       *connection;
  DBusGProxy            *proxy;
  GError                *error = NULL;
  const gchar           *restartenv;
  gboolean               norestart = FALSE;
  gboolean               new_theme = FALSE;
         
  priv = config->priv = LAUNCHER_CONFIG_GET_PRIVATE (config);
 
  /* Check if we're running on Poulsbo */
  env = g_getenv ("LAUNCHER_POULSBO");
  if (env)
    on_poulbo = atoi (env);
  config->is_poulsbo = on_poulbo;
  
  if (g_file_test (POULSBO, G_FILE_TEST_EXISTS))
    config->is_poulsbo = TRUE;
  if (on_poulbo)
    config->is_poulsbo = TRUE;
  if (config->is_poulsbo)
    g_debug ("Running on Poulsbo");
  
  /* Load in gconf values first */
  gconf_client_add_dir (client, FONT_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
  set_font (gconf_client_get_string (client, FONT_PATH, NULL), 
            config);
  gconf_client_notify_add (client, FONT_PATH,
                           (GConfClientNotifyFunc)on_font_changed,
                           config, NULL, NULL);

  gconf_client_add_dir (client, DPI_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
  set_dpi (gconf_client_get_float (client, FONT_DPI, NULL), config);
  gconf_client_notify_add (client, FONT_DPI, 
                           (GConfClientNotifyFunc)on_font_dpi_changed, config,
                           NULL, NULL);

  new_theme = config->new_theme=gconf_client_get_bool (client,NEW_THEME,&error);
  load_color (gconf_client_get_string (client, CAT_FONT_COLOR, NULL),
              &config->cat_font_color);
  load_color (gconf_client_get_string (client, CAT_FONT_ACTIVE_COLOR, NULL),
              &config->cat_font_active_color);

  load_color (gconf_client_get_string (client, ICN_FONT_COLOR, NULL),
              &config->icon_font_color);

  if (!load_color (gconf_client_get_string (client, GRID_LINE_COLOR, NULL),
                   &config->grid_line_color))
  {
    config->grid_line_color.red = 0;
    config->grid_line_color.green = 0;
    config->grid_line_color.blue = 0;
    config->grid_line_color.alpha = 50;
  }

  config->disable_single_instance = gconf_client_get_bool (client,
                                                    DISABLE_SINGLE_INSTANCE, 
                                                           NULL);
  /* Override colour settings for new theme */
  if (new_theme)
  {
    config->cat_font_color.red = 0xff;
    config->cat_font_color.green = 0xff;
    config->cat_font_color.blue = 0xff;
    config->cat_font_color.alpha = 0xff;

    config->cat_font_active_color.red = 0xff;
    config->cat_font_active_color.green = 0xff;
    config->cat_font_active_color.blue = 0xff;
    config->cat_font_active_color.alpha = 0xff;

    config->icon_font_color.red = 0xff;
    config->icon_font_color.green = 0xff;
    config->icon_font_color.blue = 0xff;
    config->icon_font_color.alpha = 0xff;

    config->grid_line_color.red = 0xff;
    config->grid_line_color.green = 0xff;
    config->grid_line_color.blue = 0xff;
    config->grid_line_color.alpha = 0x10;

    config->bg_padding = 0;
    config->bg_active = 130;
    config->bg_inactive = 0;
  }
  else
  {
    config->bg_padding = 4;
    config->bg_active = 170;
    config->bg_inactive = 100;
  }

  config->add = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                              PKGDATADIR,
                                  "add.svg", NULL);
  config->catbg = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                PKGDATADIR,
                                    "catbg.svg", NULL);
  config->catbgv = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                 PKGDATADIR,
                                     "catbgv.svg", NULL);  
  config->catbarbg  = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                    PKGDATADIR,
                                        "catbarbg.svg", NULL); 
  config->iconviewbg = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                     PKGDATADIR,
                                         "iconviewbg.svg", NULL); 
  config->catbarbg = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                   PKGDATADIR,
                                       "catbarbg.svg", NULL); 
  config->options = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                  PKGDATADIR,
                                      "options.svg", NULL); 
  config->info =  g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                PKGDATADIR,
                                    "info.svg", NULL);
  config->sep =  g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                               PKGDATADIR,
                                   "sep.svg", NULL);  
  config->scrollbar =  g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                    PKGDATADIR,
                                        "scrollbar.svg", NULL);  
  config->handle = g_build_filename (new_theme ? PKGDATADIR"/clarity-theme" : 
                                                 PKGDATADIR,
                                     "handle.svg", NULL);  


 
  /* Now load in all the calculations */
  calculate_sizes (config);

  restartenv = g_getenv ("LAUNCHER_NORESTART");
  norestart = restartenv ? atoi (restartenv) : FALSE;

  if (norestart || config->is_poulsbo)
  {
    g_debug ("Not connecting to ConsoleKit as \"--no-restart\" requested");
    g_object_unref (client);
    return;
  }
  g_debug ("Connecting to ConsoleKit for suspend/resume restarting");

  /* Session switch watching */
  priv->connection = connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);

  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free(error);
    return;
  }
  /* Get the current session object */
  proxy = priv->proxy = dbus_g_proxy_new_for_name(connection,
                                    "org.freedesktop.ConsoleKit",
                                    "/org/freedesktop/ConsoleKit/Manager",
                                    "org.freedesktop.ConsoleKit.Manager");

  if (!proxy)
  {
    g_warning ("Unable to get the ConsoleKit manager.");
    dbus_g_connection_unref (connection);
    return;
  }
  if (!dbus_g_proxy_call (proxy, "GetCurrentSession", &error, 
                          G_TYPE_INVALID, 
                          DBUS_TYPE_G_PROXY, &priv->proxy,
                          G_TYPE_INVALID))
  {
    g_warning ("Unable to GetCurrentSession from ConsoleKit.Manager");
    g_object_unref (proxy);
    dbus_g_connection_unref (connection);
    return;
  }
  
  dbus_g_proxy_set_interface (priv->proxy, 
                              "org.freedesktop.ConsoleKit.Session");
  dbus_g_proxy_add_signal (priv->proxy, "ActiveChanged",
                           G_TYPE_BOOLEAN, G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (priv->proxy, "ActiveChanged", 
                               G_CALLBACK (on_session_active_changed), config, 
                               NULL);

  g_object_unref (proxy);

  g_object_unref (client);
}

LauncherConfig*
launcher_config_get_default (void)
{
  static LauncherConfig *config = NULL;
  
  if (config == NULL)
    config = g_object_new (LAUNCHER_TYPE_CONFIG, 
                            NULL);

  return config;
}
