/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <tidy/tidy.h>

#include "launcher-add.h"

#include "clutter-drag-server.h"
#include "clutter-focus-group.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-startup.h"
#include "launcher-util.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherAdd,
                         launcher_add, 
                         CLUTTER_TYPE_TEXTURE,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));


#define LAUNCHER_ADD_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ADD, \
  LauncherAddPrivate))

struct _LauncherAddPrivate
{
  ClutterTimeline       *time;
  ClutterEffectTemplate *temp;
  
  ClutterTimeline *current;
};

enum
{
  CLICKED,

  LAST_SIGNAL
};
static guint _add_signals[LAST_SIGNAL] = { 0 };


static gboolean
on_enter (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherAdd          *add)
{
  LauncherAddPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ADD (add), FALSE);
  priv = add->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }
  clutter_actor_set_anchor_point_from_gravity (CLUTTER_ACTOR (add),
                                               CLUTTER_GRAVITY_CENTER); 
  priv->current = clutter_effect_scale (priv->temp, actor, 1.3, 1.3,NULL, NULL);
  return TRUE;
}


static gboolean
on_leave (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherAdd          *add)
{
  LauncherAddPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ADD (add), FALSE);
  priv = add->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }
  clutter_actor_set_anchor_point_from_gravity (CLUTTER_ACTOR (add),
                                               CLUTTER_GRAVITY_CENTER);
  priv->current = clutter_effect_scale (priv->temp, actor, 1.0,1.0,NULL, NULL);

  return TRUE;
}

static gboolean
on_click (ClutterActor *actor, ClutterButtonEvent *event)
{
  if (event->button == 1)
  {
    g_signal_emit (actor, _add_signals[CLICKED], 0);
    return TRUE;
  }
  return FALSE;
}

/*
 * FOCUS STUFF
 */
static void
launcher_iconview_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  ClutterActor *actor = CLUTTER_ACTOR (group);
  LauncherAddPrivate *priv = LAUNCHER_ADD (group)->priv;

  if (has_focus)
  {
   priv->current = clutter_effect_scale (priv->temp, actor, 
                                         1.3, 1.3,NULL, NULL);

  }
  else
  {
   priv->current = clutter_effect_scale (priv->temp, actor, 
                                         1.0, 1.0,NULL, NULL);  
  }
}

static gboolean
launcher_iconview_direction_event (ClutterFocusGroup     *group,
                                 ClutterFocusDirection  dir)
{
  /* There is no-where to move focus to */
  return FALSE;
}

static gboolean
launcher_iconview_action_event (ClutterFocusGroup *group)
{
  g_signal_emit (group, _add_signals[CLICKED], 0);
  return TRUE;
}

static gboolean
launcher_iconview_key_event (ClutterFocusGroup *group, const gchar *string)
{
  g_debug ("String event: %s", string);
  return TRUE;
}                        

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus       = launcher_iconview_set_focus;
  iface->direction_event = launcher_iconview_direction_event;
  iface->key_event       = launcher_iconview_key_event;
  iface->action_event    = launcher_iconview_action_event;
}

/*
 * /FOCUS STUFF
 */

/* GObject stuff */
static void
launcher_add_finalize (GObject *add)
{
  LauncherAddPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ADD (add));
  priv = LAUNCHER_ADD_GET_PRIVATE (add);

  G_OBJECT_CLASS (launcher_add_parent_class)->finalize (add);
}

static void
launcher_add_class_init (LauncherAddClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_add_finalize;

	_add_signals[CLICKED] =
		g_signal_new ("clicked",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherAddClass, clicked),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);
  
  g_type_class_add_private (obj_class, sizeof (LauncherAddPrivate));
}

      
static void
launcher_add_init (LauncherAdd *add)
{
  LauncherAddPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GdkPixbuf *temp;

  priv = add->priv = LAUNCHER_ADD_GET_PRIVATE (add);

  temp = gdk_pixbuf_new_from_file (cfg->add, NULL);
  g_object_set (add, "pixbuf", temp, NULL);
  g_object_unref (temp);
    
  clutter_actor_set_reactive (CLUTTER_ACTOR (add), TRUE);
  g_signal_connect (add, "enter-event", 
                    G_CALLBACK (on_enter), add);
  g_signal_connect (add, "leave-event", 
                    G_CALLBACK (on_leave), add);
  g_signal_connect (add, "button-release-event", 
                    G_CALLBACK (on_click), add);

  priv->time = launcher_util_timeline_new_fast ();
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);
}

ClutterActor *
launcher_add_new (void)

{
  ClutterActor *add = NULL;

  add = g_object_new (LAUNCHER_TYPE_ADD, 
                           NULL);

  return add;
}

void
launcher_set_enabled (LauncherAdd *add, gboolean enabled)
{
  LauncherAddPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ADD (add));
  priv = add->priv;

  //clutter_actor_set_reactive (CLUTTER_ACTOR (add), enabled);
  clutter_actor_set_opacity (CLUTTER_ACTOR (add), enabled ? 255 : 50);
}
