/*******************************************************************************
*
*   clutter/gtk+-based offscreen-widget test
*
*   Copyright (C) 2008 Canonical Ltd.
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 3,
*   as published by the Free Software Foundation.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*   author: Mirco Müller <mirco.mueller@ubuntu.com>
*
*   compile with:
*      gcc -g `pkg-config --cflags --libs gtk+-2.0 clutter-0.6 clutter-gtk-0.6`\
*      -lm main.c cairo-utils.c -o clutter-widget-redirection
*
*******************************************************************************/

#include "cairo-utils.h"

gdouble*
_kernel_1d_new (gint    radius,
                gdouble deviation)
{
  gdouble* kernel = NULL;
  gdouble  sum    = 0.0f;
  gdouble  value  = 0.0f;
  gint     i;
  gint     size = 2 * radius + 1;
  gdouble  radiusf;

  if (radius <= 0)
    return NULL;

  kernel = (gdouble*) g_malloc0 ((size + 1) * sizeof (gdouble));
  if (!kernel)
    return NULL;

  radiusf = fabs (radius) + 1.0f;
  if (deviation == 0.0f)
    deviation = sqrt (-(radiusf * radiusf) / (2.0f * log (1.0f / 255.0f)));

  kernel[0] = size;
  value = (gdouble) -radius;
  for (i = 0; i < size; i++)
    {
    kernel[1 + i] = 1.0f / (2.506628275f * deviation) *
                    expf (-((value * value) /
                    (2.0f * deviation * deviation)));
    sum += kernel[1 + i];
    value += 1.0f;
    }

  for (i = 0; i < size; i++)
    kernel[1 + i] /= sum;

  return kernel;
}

void
_kernel_1d_delete (gdouble* kernel)
{
  g_assert (kernel != NULL);
  g_free ((gpointer) kernel);
}

void
cairo_image_surface_blur (cairo_surface_t* surface,
			  gint             horzRadius,
			  gint             vertRadius)
{
  gint     iX;
  gint     iY;
  gint     i;
  gint     x;
  gint     y;
  gint     stride;
  gint     offset;
  gint     baseOffset;
  gdouble* horzBlur;
  gdouble* vertBlur;
  gdouble* horzKernel;
  gdouble* vertKernel;
  guchar*  src;
  gint     width;
  gint     height;
  gint     channels;

  /* sanity checks */
  if (!surface || horzRadius == 0 || vertRadius == 0)
    return;

  if (cairo_surface_get_type (surface) != CAIRO_SURFACE_TYPE_IMAGE)
    return;

  cairo_surface_flush (surface);

  src  = cairo_image_surface_get_data (surface);
  width  = cairo_image_surface_get_width (surface);
  height = cairo_image_surface_get_height (surface);

   if (cairo_image_surface_get_format (surface) != CAIRO_FORMAT_ARGB32 &&
       cairo_image_surface_get_format (surface) != CAIRO_FORMAT_RGB24)
     return;

   channels = 4;

  stride = width * channels;

  /* create buffers to hold the blur-passes */
  horzBlur = (gdouble*) g_malloc0 (height * stride * sizeof (gdouble));
  vertBlur = (gdouble*) g_malloc0 (height * stride * sizeof (gdouble));
  if (!horzBlur || !vertBlur)
    {
    g_print ("cairo_image_surface_blur(): ");
    g_print ("Could not allocate the temporary blur-buffers!\n");

    if (horzBlur)
      g_free ((gpointer) horzBlur);

    if (vertBlur)
      g_free ((gpointer) vertBlur);

    return;
    }

  horzKernel = _kernel_1d_new (horzRadius, 0.0f);
  vertKernel = _kernel_1d_new (vertRadius, 0.0f);

  if (!horzKernel || !vertKernel)
    {
    g_print ("cairo_image_surface_blur(): ");
    g_print ("Could create blur-kernels!\n");

    g_free ((gpointer) horzBlur);
    g_free ((gpointer) vertBlur);

    if (horzKernel)
      _kernel_1d_delete (horzKernel);

    if (vertKernel)
      _kernel_1d_delete (vertKernel);

    return;
    }

  /* horizontal pass */
  for (iY = 0; iY < height; iY++)
    {
    for (iX = 0; iX < width; iX++)
      {
      gdouble red   = 0.0f;
      gdouble green = 0.0f;
      gdouble blue  = 0.0f;
      gdouble alpha = 0.0f;

      offset = ((gint) horzKernel[0]) / -2;
      for (i = 0; i < (gint) horzKernel[0]; i++)
        {
        x = iX + offset;
        if (x >= 0 && x <= width)
          {
          baseOffset = iY * stride + x * channels;

          if (channels == 4)
            alpha += (horzKernel[1+i] * (gdouble) src[baseOffset + 3]);

          red   += (horzKernel[1+i] * (gdouble) src[baseOffset + 2]);
          green += (horzKernel[1+i] * (gdouble) src[baseOffset + 1]);
          blue  += (horzKernel[1+i] * (gdouble) src[baseOffset + 0]);
          }

          offset++;
        }

      baseOffset = iY * stride + iX * channels;

      if (channels == 4)
        horzBlur[baseOffset + 3] = alpha;

      horzBlur[baseOffset + 2] = red;
      horzBlur[baseOffset + 1] = green;
      horzBlur[baseOffset + 0] = blue;
      }
    }

  /* vertical pass */
  for (iY = 0; iY < height; iY++)
    {
    for (iX = 0; iX < width; iX++)
      {
      gdouble red   = 0.0f;
      gdouble green = 0.0f;
      gdouble blue  = 0.0f;
      gdouble alpha = 0.0f;

      offset = ((gint) vertKernel[0]) / -2;
      for (i = 0; i < (gint) vertKernel[0]; i++)
        {
        y = iY + offset;
        if (y >= 0 && y <= height)
          {
          baseOffset = y * stride + iX * channels;

          if (channels == 4)
            alpha += (vertKernel[1+i] * horzBlur[baseOffset + 3]);

          red   += (vertKernel[1+i] * horzBlur[baseOffset + 2]);
          green += (vertKernel[1+i] * horzBlur[baseOffset + 1]);
          blue  += (vertKernel[1+i] * horzBlur[baseOffset + 0]);
          }

        offset++;
        }

      baseOffset = iY * stride + iX * channels;

      if (channels == 4)
        vertBlur[baseOffset + 3] = alpha;

      vertBlur[baseOffset + 2] = red;
      vertBlur[baseOffset + 1] = green;
      vertBlur[baseOffset + 0] = blue;
      }
    }

  _kernel_1d_delete (horzKernel);
  _kernel_1d_delete (vertKernel);

  for (iY = 0; iY < height; iY++)
    {
    for (iX = 0; iX < width; iX++)
      {
      offset = iY * stride + iX * channels;

      if (channels == 4)
        src[offset + 3] = (guchar) vertBlur[offset + 3];

      src[offset + 2] = (guchar) vertBlur[offset + 2];
      src[offset + 1] = (guchar) vertBlur[offset + 1];
      src[offset + 0] = (guchar) vertBlur[offset + 0];
      }
    }

  cairo_surface_mark_dirty (surface);

  g_free ((gpointer) horzBlur);
  g_free ((gpointer) vertBlur);
}

void
demultiply_alpha (guint32* buffer,
                  guint    width,
                  guint    height)
{
  guint  size = width * height;
  guint8 a;
  guint8 r;
  guint8 g;
  guint8 b;
  guint  i;

  for (i = 0; i < size; i++)
    {
    a = (buffer[i] & ARGB_A_MASK) >> ARGB_A_OFFSET;
    if (a == 0)
      continue;
    else if (a == 255)
      buffer[i] = GUINT32_TO_LE (buffer[i]);
    else
      {
      r = (((buffer[i] & ARGB_R_MASK) >> ARGB_R_OFFSET) * 255 + a / 2) / a;
      g = (((buffer[i] & ARGB_G_MASK) >> ARGB_G_OFFSET) * 255 + a / 2) / a;
      b = (((buffer[i] & ARGB_B_MASK) >> ARGB_B_OFFSET) * 255 + a / 2) / a;
      buffer[i] = b << BYTE_1_OFFSET
                | g << BYTE_2_OFFSET
                | r << BYTE_3_OFFSET
                | a << BYTE_4_OFFSET;
      }
    }
}

void
cairo_round_rect (cairo_t* cr,
                  gdouble  aspect,
                  gdouble  x,
                  gdouble  y,
                  gdouble  corner,
                  gdouble  width,
                  gdouble  height)
{
  gdouble radius = corner / aspect;

  /* top-left, right of the corner */
  cairo_move_to (cr, x + radius, y);

  /* top-right, left of the corner */
  cairo_line_to (cr,
                 x + width - radius,
                 y);

  /* top-right, below the corner */
  cairo_arc (cr,
             x + width - radius,
             y + radius,
             radius,
             -90.0f * G_PI / 180.0f,
             0.0f * G_PI / 180.0f);

  /* bottom-right, above the corner */
  cairo_line_to (cr,
                 x + width,
                 y + height - radius);

  /* bottom-right, left of the corner */
  cairo_arc (cr,
             x + width - radius,
             y + height - radius,
             radius,
             0.0f * G_PI / 180.0f,
             90.0f * G_PI / 180.0f);

  /* bottom-left, right of the corner */
  cairo_line_to (cr,
                 x + radius,
                 y + height);

  /* bottom-left, above the corner */
  cairo_arc (cr,
             x + radius,
             y + height - radius,
             radius,
             90.0f * G_PI / 180.0f,
             180.0f * G_PI / 180.0f);

  /* top-left, below the corner */
  cairo_line_to (cr,
                 x,
                 y + radius);

  /* top-left, right of the corner */
  cairo_arc (cr,
             x + radius,
             y + radius,
             radius,
             180.0f * G_PI / 180.0f,
             270.0f * G_PI / 180.0f);
}

