/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License version 3 as published by the Free Software Foundation;
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib.h>
#include <gio/gio.h>
#include <gtk/gtk.h>

#include <panel-applet.h>

#define ICON_SIZE 24
#define DEFAULT_ICON_NAME "gtk-help"

typedef struct 
{
  GtkWidget    *image;
  GtkWidget    *applet;
  
} MenuApp;

/* Forwards */
static void cw_panel_background_changed (PanelApplet               *applet,
                                         PanelAppletBackgroundType  type,
				                                 GdkColor                  *colour,
				                                 GdkPixmap                 *pixmap,
                                         gpointer                   user_data);
static void display_about_dialog (BonoboUIComponent *component, 
                                  gpointer           user_data, 
                                  const gchar       *verb);

static const char Context_menu_xml [] =
   "<popup name=\"button3\">\n"
  "   <menuitem name=\"About Item\" "
   "             verb=\"MenuAbout\" "
   "           _label=\"_About\"\n"
   "          pixtype=\"stock\" "
   "          pixname=\"gnome-stock-about\"/>\n"
   "</popup>\n";

static const BonoboUIVerb _menu_verbs [] = 
{
        BONOBO_UI_VERB ("MenuAbout", display_about_dialog),
        BONOBO_UI_VERB_END
}; 

static const gchar *close_window_authors [] = 
{
	"Neil J. Patel <neil.patel@canonical.com>",
	NULL
};

static const gchar *topics [] =
{
  NULL,
  "yelp ghelp:newtoubuntu",
  NULL,
  "yelp ghelp:internet#connect",
  "yelp ghelp:config-desktop",
  "yelp ghelp:files-and-docs",
  "yelp ghelp:musicvideophotos",
  NULL,
  "yelp ghelp:keeping-safe#updates",
  "yelp ghelp:users-admin#tool-introduction",
  "yelp ghelp:printing",
  NULL,
  "yelp",
  "xdg-open http://www.ubuntu.com/support/communitysupport",
  NULL,
  "xdg-open http://support.dell.com"
};

static const gchar *ui [][2] =
{
  { "face-smile", N_("Welcome!") }, 
  { "distributor-logo", N_("New to Ubuntu?"), },
  { NULL, NULL },
  { "web-browser", N_("Connecting to the Internet") },
  { "user-desktop", N_("Customizing Your Desktop") },
  { "folder", N_("Files, Folders and Documents") },
  { "applications-multimedia", N_("Music, Videos and Photos") },
  { NULL, NULL }, 
  { "system-software-update", N_("Software Updates") },
  { "system-users", N_("Adding Users") },
  { "printer", N_("Adding a printer") },
  { NULL, NULL },
  { "gtk-help", N_("Ubuntu Help Center") },
  { "distributor-logo", N_("Ubuntu Community Support") },
  { NULL, NULL },
  { PKGDATADIR"/DellFavicon.png", N_("Dell Support") }
};

enum
{
  TOPIC_WELCOME=0,
  TOPIC_UBUNTU,

  TOPIC_SEP1,
  
  TOPIC_INTERNET,
  TOPIC_DESKTOP,
  TOPIC_FILES,
  TOPIC_MUSIC,

  TOPIC_SEP2,
  
  TOPIC_UPDATES,
  TOPIC_USERS,
  TOPIC_PRINTER,

  TOPIC_SEP3,
  
  TOPIC_YELP,
  TOPIC_COMMUNITY,

  TOPIC_SEP4,
  
  TOPIC_DELL_SUPPORT,

  N_TOPICS
};

/* Event callbacks */
static void
position_func (GtkMenu  *menu,
               gint     *x,
               gint     *y,
               gboolean *push_in,
               gpointer  data)
{
  GtkWidget *widget = GTK_WIDGET (data);
  gint xx=0, yy=0;

  *push_in = TRUE;

  gdk_window_get_origin (widget->window, &xx, &yy);
  
  *x = xx;
  *y = yy + widget->allocation.height;
}

static const gchar *
get_welcome_command (void)
{
  /* This function tries to determine what system we're on, based on lshal info */
  static const gchar *cmd = NULL;
  static gboolean cmd_set = FALSE;

  gchar *halinfo;
  char *needle;

  if (cmd_set)
    return cmd;

  if (g_spawn_command_line_sync ("lshal", &halinfo, NULL, NULL, NULL)) {
    needle = strstr (halinfo, " system.hardware.product = '");
    if (needle != NULL) {
      char *end;
      needle += strlen (" system.hardware.product = '");
      end = strchr (needle, '\'');
      if (end != NULL)
        *end = 0;
    }

    if (strcmp (needle, "Inspiron 1010") == 0)
      cmd = "yelp ghelp:tiger";
    else if (strcmp (needle, "Inspiron 1011") == 0)
      cmd = "yelp ghelp:bear";
  }

  cmd_set = TRUE;
  return cmd;
}

static void
on_yelp_activate (GtkMenuItem *item, gpointer num)
{
  gint topic = GPOINTER_TO_INT (num);
  const gchar *cmd = NULL;

  if (topic >= 0 && topic < N_TOPICS && topics[topic])
    cmd = topics[topic];

  if (topic == TOPIC_WELCOME)
    cmd = get_welcome_command ();

  if (cmd != NULL)
    g_spawn_command_line_async (cmd, NULL);
}

static gboolean
on_button_press (GtkWidget      *applet,
                 GdkEventButton *event,
                 MenuApp        *app)
{
  GtkWidget *menu;
  GtkWidget *item;
  GtkWidget *image;
  gint i;
  
  g_return_val_if_fail (app, FALSE);

  if (event->button != 1)
    return FALSE;

  menu = gtk_menu_new ();

  for (i = 0; i < N_TOPICS; i++)
  {
    if (i == TOPIC_WELCOME && get_welcome_command () == NULL)
      continue;
    if (ui[i][1] == NULL)
    {
      item = gtk_separator_menu_item_new ();
    }
    else
    {
      item = gtk_image_menu_item_new_with_label (_(ui[i][1]));
      if (ui[i][0])
      {
        if (g_strstr_len (ui[i][0], strlen (ui[i][0]), "/"))
          image = gtk_image_new_from_file (ui[i][0]);
        else
          image = gtk_image_new_from_icon_name (ui[i][0], GTK_ICON_SIZE_BUTTON);
        gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
      }
      g_signal_connect (item, "activate", G_CALLBACK (on_yelp_activate), 
                      GINT_TO_POINTER (i));
    }
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    gtk_widget_show (item); 
  }

  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, 
                  position_func, applet, 
                  event->button, event->time);
  return TRUE;
}

static void
on_panel_size_change (PanelApplet *applet, gint size, GtkImage *image)
{
  static GtkIconTheme *theme = NULL;
  GdkPixbuf           *pixbuf = NULL;

  g_return_if_fail (PANEL_IS_APPLET (applet));
  g_return_if_fail (GTK_IS_IMAGE (image));

  if (!theme)
    theme = gtk_icon_theme_get_default ();

  pixbuf = gtk_icon_theme_load_icon (theme, DEFAULT_ICON_NAME, size, 0, NULL);

  if (pixbuf)
  {
    if (gdk_pixbuf_get_width (pixbuf) != size)
    {
      GdkPixbuf *resize;

      resize = pixbuf;
      pixbuf = gdk_pixbuf_scale_simple (resize, size, 
                                        size, GDK_INTERP_BILINEAR);
      g_object_unref (resize);
    }
    gtk_image_set_from_pixbuf (image, pixbuf);
    g_object_unref (pixbuf);
  }
}

static gboolean
cw_applet_fill (PanelApplet *applet, 
                const gchar *iid, 
                gpointer     data)
{
  MenuApp *app;
   
  if (strcmp (iid, "OAFIID:GNOME_DellHelp") != 0)
		return FALSE;
  	
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  app = g_slice_new0 (MenuApp);
  
  app->applet = GTK_WIDGET (applet);
  panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);
  gtk_container_set_border_width (GTK_CONTAINER (applet), 0);
  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), 
                               _("Get help with your Dell System"));

  /* The actual icon */
  app->image = gtk_image_new ();
  on_panel_size_change (applet, panel_applet_get_size (applet), 
                        GTK_IMAGE (app->image));
  gtk_container_add (GTK_CONTAINER (applet), app->image);

	gtk_widget_show_all (GTK_WIDGET (applet));
		
  /* Signals */
	g_signal_connect (applet, "change-background",
			             G_CALLBACK (cw_panel_background_changed), NULL);
  g_signal_connect (applet, "button-press-event",
                    G_CALLBACK (on_button_press), app);
  g_signal_connect (applet, "change-size", 
                    G_CALLBACK (on_panel_size_change), app->image);
 
  panel_applet_setup_menu (PANEL_APPLET (applet),
                         	Context_menu_xml,
                         	_menu_verbs,
                         	applet);
	
	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_DellHelp_Factory",
                             PANEL_TYPE_APPLET,
                             "Dell Help",
                             "0",
                             cw_applet_fill,
                             NULL);

static void 
cw_panel_background_changed (PanelApplet               *applet,
                             PanelAppletBackgroundType  type,
                             GdkColor                  *colour,
                             GdkPixmap                 *pixmap,
                             gpointer                   user_data)
{
  GtkRcStyle *rc_style;
  GtkStyle *style;

  /* reset style */
  gtk_widget_set_style (GTK_WIDGET (applet), NULL);
  rc_style = gtk_rc_style_new ();
  gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
  gtk_rc_style_unref (rc_style);

  switch (type) 
  {
    case PANEL_NO_BACKGROUND:
      break;
    case PANEL_COLOR_BACKGROUND:
      gtk_widget_modify_bg (GTK_WIDGET (applet), GTK_STATE_NORMAL, colour);
      break;
    
    case PANEL_PIXMAP_BACKGROUND:
      style = gtk_style_copy (GTK_WIDGET (applet)->style);
      if (style->bg_pixmap[GTK_STATE_NORMAL])
        g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
      style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);
      gtk_widget_set_style (GTK_WIDGET (applet), style);
      g_object_unref (style);
      break;
  }
}

static void
display_about_dialog (BonoboUIComponent *component, 
                      gpointer           user_data, 
                      const gchar       *verb)
{
  GtkWidget *panel_about_dialog;
  	
  panel_about_dialog = gtk_about_dialog_new ();

  g_object_set (panel_about_dialog,
                "name", _("Dell Menu Applet"),
                "comments", _("Dell Menu Applet"),
                "version", VERSION,
                "authors", close_window_authors,
                "logo-icon-name", DEFAULT_ICON_NAME,
                "icon-name", DEFAULT_ICON_NAME,
                "copyright", "Copyright \xc2\xa9 2008 Dell Inc./Canonical Ltd",
                NULL);
  
  gtk_widget_show (panel_about_dialog);

  g_signal_connect (panel_about_dialog, "response",
                    G_CALLBACK (gtk_widget_destroy), NULL);
	

  gtk_window_present (GTK_WINDOW (panel_about_dialog));
}
