/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Neil Jagdish Patel <neil.patel@canonical.com>
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct
{
  DeeModel *model;

} SignalsFixture;

static void rows_setup          (SignalsFixture *fix, gconstpointer data);
static void rows_teardown       (SignalsFixture *fix, gconstpointer data);
static void proxy_rows_setup    (SignalsFixture *fix, gconstpointer data);
static void proxy_rows_teardown (SignalsFixture *fix, gconstpointer data);

static void test_signal_add     (SignalsFixture *fix, gconstpointer data);
static void test_signal_remove  (SignalsFixture *fix, gconstpointer data);
static void test_signal_changed (SignalsFixture *fix, gconstpointer data);
static void test_signal_freeze  (SignalsFixture *fix, gconstpointer data);

void
test_model_signals_create_suite (void)
{
#define SEQ_DOMAIN "/Model/Sequence/Signals"
#define PROXY_DOMAIN "/Model/Proxy/Signals"

  g_test_add (SEQ_DOMAIN"/Add", SignalsFixture, 0,
              rows_setup, test_signal_add, rows_teardown);
  g_test_add (PROXY_DOMAIN"/Add", SignalsFixture, 0,
              proxy_rows_setup, test_signal_add, proxy_rows_teardown);
  
  g_test_add (SEQ_DOMAIN"/Remove", SignalsFixture, 0,
              rows_setup, test_signal_remove, rows_teardown);
  g_test_add (PROXY_DOMAIN"/Remove", SignalsFixture, 0,
              proxy_rows_setup, test_signal_remove, proxy_rows_teardown);
              
  g_test_add (SEQ_DOMAIN"/Changed", SignalsFixture, 0,
              rows_setup, test_signal_changed, rows_teardown);
  g_test_add (PROXY_DOMAIN"/Changed", SignalsFixture, 0,
              proxy_rows_setup, test_signal_changed, proxy_rows_teardown);
              
  g_test_add (SEQ_DOMAIN"/Freeze", SignalsFixture, 0,
              rows_setup, test_signal_freeze, rows_teardown);
  g_test_add (PROXY_DOMAIN"/Freeze", SignalsFixture, 0,
              proxy_rows_setup, test_signal_freeze, proxy_rows_teardown);
}

static void
rows_setup (SignalsFixture *fix, gconstpointer data)
{
  fix->model = dee_sequence_model_new (2,
                                       G_TYPE_INT,
                                       G_TYPE_STRING);

  g_assert (DEE_IS_SEQUENCE_MODEL (fix->model));
}

static void
rows_teardown (SignalsFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
}

static void
proxy_rows_setup (SignalsFixture *fix, gconstpointer data)
{
  rows_setup (fix, data);
  fix->model = g_object_new (DEE_TYPE_PROXY_MODEL,
                             "back-end", fix->model,
                             NULL);

  g_assert (DEE_IS_PROXY_MODEL (fix->model));
}

static void
proxy_rows_teardown (SignalsFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
}

static guint n_add_signals = 0;

static void
test_signal_add_callback (DeeModel *model, DeeModelIter *iter)
{
  n_add_signals++;
}

static void
test_signal_add (SignalsFixture *fix, gconstpointer data)
{
  gint i;

  g_signal_connect (fix->model, "row-added",
                    G_CALLBACK (test_signal_add_callback), NULL);

  n_add_signals = 0;
  for (i = 0; i < 10000; i++)
    {
      dee_model_append (fix->model, 0, i, 1, "Test", -1);
    }
  g_assert_cmpint (n_add_signals, ==, 10000);
}

static guint n_remove_signals = 0;

static void
test_signal_remove_callback (DeeModel *model, DeeModelIter *iter)
{
  n_remove_signals++;
}

static void
test_signal_remove (SignalsFixture *fix, gconstpointer data)
{
  gint           i;
  DeeModelIter *iter;

  g_signal_connect (fix->model, "row-removed",
                    G_CALLBACK (test_signal_remove_callback), NULL);

  for (i = 0; i < 10000; i++)
    {
      dee_model_append (fix->model, 0, i, 1, "Test", -1);
    }

  n_remove_signals = i = 0;
  iter = dee_model_get_first_iter (fix->model);
  while (iter != NULL && !dee_model_is_last (fix->model, iter))
    {
      dee_model_remove (fix->model, iter);
      iter = dee_model_get_first_iter (fix->model);
    }
  g_assert_cmpint (n_remove_signals, ==, 10000);
}

static guint n_changed_signals = 0;

static void
test_signal_changed_callback (DeeModel *model, DeeModelIter *iter)
{
  n_changed_signals++;
}

static void
test_signal_changed (SignalsFixture *fix, gconstpointer data)
{
  DeeModelIter *iter;
  gint           i;

  g_signal_connect (fix->model, "row-changed",
                    G_CALLBACK (test_signal_changed_callback), NULL);

  for (i = 0; i < 10000; i++)
    {
      dee_model_append (fix->model, 0, i, 1, "Test", -1);
    }

  n_changed_signals = 0;
  iter = dee_model_get_first_iter (fix->model);
  while (iter != NULL && !dee_model_is_last (fix->model, iter))
    {
      gint   j = 0;
      gchar *k = "ing";

      dee_model_set (fix->model, iter, 0, j, 1, k, -1);

      iter = dee_model_next (fix->model, iter);
    }
  g_assert_cmpint (n_changed_signals, ==, 10000);

  n_changed_signals = 0;
  iter = dee_model_get_first_iter (fix->model);
  while (iter != NULL && !dee_model_is_last (fix->model, iter))
    {
      GValue value = { 0, };

      g_value_init (&value, G_TYPE_INT);
      g_value_set_int (&value, 10);

      dee_model_set_value (fix->model, iter, 0, &value);

      g_value_unset (&value);

      iter = dee_model_next (fix->model, iter);
    }
  g_assert_cmpint (n_changed_signals, ==, 10000);
}

static GSList *delayed_signals = NULL;

static void
check_delayed_signals (GSList *expected)
{
  GSList *d, *e;  
  
  g_assert_cmpint (g_slist_length (expected), ==,
                   g_slist_length (delayed_signals));
  
  d = delayed_signals;
  e = expected;
  while (e)
    {
      gpointer *e_data = (gpointer*) e->data;
      gpointer *d_data = (gpointer*) d->data;
      
      /* Test that add/change mode is the same */
      g_assert_cmpint (GPOINTER_TO_INT (e_data[0]), ==,
                       GPOINTER_TO_INT (d_data[0]));
      
      /* Test that they talk about the same iter */
      g_assert (e_data[1] == d_data[1]);
      
      e = e->next;
      d = d->next;
    }
  
}

static void
test_delayed_added_callback (DeeModel *model, DeeModelIter *iter)
{
  gpointer *data = g_new (gpointer, 2);
  data[0] = GINT_TO_POINTER (0);
  data[1] = iter;
  delayed_signals = g_slist_append (delayed_signals, data);
}

static void
test_delayed_changed_callback (DeeModel *model, DeeModelIter *iter)
{
  gpointer *data = g_new (gpointer, 2);
  data[0] = GINT_TO_POINTER (1);
  data[1] = iter;
  delayed_signals = g_slist_append (delayed_signals, data);
}

static void
test_signal_freeze (SignalsFixture *fix, gconstpointer _data)
{
  DeeModelIter *iter;
  GSList       *expected = NULL;
  gpointer     *data;
  gint           i;

  g_signal_connect (fix->model, "row-added",
                    G_CALLBACK (test_delayed_added_callback), NULL);
  g_signal_connect (fix->model, "row-changed",
                    G_CALLBACK (test_delayed_changed_callback), NULL);

  /* Double freeze the model just for fun! */
  dee_model_freeze_signals (fix->model);
  dee_model_freeze_signals (fix->model);
  
  for (i = 0; i < 10; i++)
    {
      iter = dee_model_append (fix->model, 0, i, 1, "Test", -1);
      data = g_new (gpointer, 2);
      data[0] = GINT_TO_POINTER (0);
      data[1] = iter;
      expected = g_slist_append (expected, data);
      
      dee_model_set (fix->model, iter, 0, i+1000, 1, "Test2", -1);
      data = g_new (gpointer, 2);
      data[0] = GINT_TO_POINTER (1);
      data[1] = iter;
      expected = g_slist_append (expected, data);
    }

  /* All signals should be frozen */
  g_assert_cmpint (0, ==, g_slist_length (delayed_signals));
  
  /* Thawing once, should still keep signals frozen
   * since we double-froze the model*/
  dee_model_thaw_signals (fix->model);
  g_assert_cmpint (0, ==, g_slist_length (delayed_signals));  
  
  /* Let free the signals! */
  dee_model_thaw_signals (fix->model);
  g_assert_cmpint (20, ==, g_slist_length (delayed_signals));
  check_delayed_signals (expected);
  
  g_slist_foreach (expected, (GFunc)g_free, NULL);
  g_slist_foreach (delayed_signals, (GFunc)g_free, NULL);
  g_slist_free (expected);
  g_slist_free (delayed_signals);
  delayed_signals = NULL;
}
