/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

/**
 * SECTION:dee-proxy-model
 * @short_description: A model that wraps another underlying #DeeModel
 * @include: dee.h
 *
 * #DeeProxyModel wraps another #DeeModel instance and use it as a back end
 * by proxuing all method calls down to the back end.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>

#include "dee-model.h"
#include "dee-proxy-model.h"
#include "dee-versioned-model.h"
#include "dee-marshal.h"
#include "trace-log.h"

static void dee_proxy_model_model_iface_init (DeeModelIface *iface);

G_DEFINE_TYPE_WITH_CODE (DeeProxyModel,
                         dee_proxy_model,
                         DEE_TYPE_VERSIONED_MODEL,
                         G_IMPLEMENT_INTERFACE (DEE_TYPE_MODEL,
                                                dee_proxy_model_model_iface_init));

#define DEE_PROXY_MODEL_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_PROXY_MODEL, DeeProxyModelPrivate))

enum
{
  PROP_0,
  PROP_BACK_END,
  PROP_PROXY_SIGNALS,
};

/**
 * DeeProxyModelPrivate:
 *
 * Ignore this structure.
 */
struct _DeeProxyModelPrivate
{
  /* The backend model holding the actual data */
  DeeModel  *back_end;
  
  /* Whether to use the seqnums of the backend model (if it's versioned),
   * or to use our own seqnums */
  gboolean   inherit_seqnums;
  
  /* Whether or not to automatically forward signals from the back end */
  gboolean   proxy_signals;
  
  /* Signals handlers for relaying signals from the back end */
  gulong     row_added_handler;
  gulong     row_removed_handler;
  gulong     row_changed_handler;
};

#define DEE_PROXY_MODEL_BACK_END(model) (DEE_PROXY_MODEL(model)->priv->back_end)
#define SUPER_CLASS DEE_VERSIONED_MODEL_CLASS (dee_proxy_model_parent_class)

/*
 * DeeModel forward declarations
 */
static void           dee_proxy_model_set_column_type (DeeModel *self,
                                                       guint     column,
                                                       GType     type);

static void           dee_proxy_model_set_n_columns  (DeeModel *self,
                                                      guint     n_columns);
                               
static guint          dee_proxy_model_get_n_columns  (DeeModel *self);

static guint          dee_proxy_model_get_n_rows     (DeeModel *self);

static GType          dee_proxy_model_get_column_type (DeeModel *self,
                                                       guint      column);

static void           dee_proxy_model_clear          (DeeModel *self);

static DeeModelIter*  dee_proxy_model_append_valist  (DeeModel *self,
                                                      va_list    args);

static DeeModelIter*  dee_proxy_model_prepend_valist  (DeeModel *self,
                                                       va_list    args);

static DeeModelIter*  dee_proxy_model_insert_valist  (DeeModel *self,
                                                      gint       pos,
                                                      va_list    args);

static DeeModelIter*  dee_proxy_model_insert_before_valist (DeeModel     *self,
                                                            DeeModelIter *iter,
                                                            va_list    args);

static void           dee_proxy_model_remove         (DeeModel     *self,
                                                      DeeModelIter *iter);

static void           dee_proxy_model_set_valist     (DeeModel       *self,
                                                      DeeModelIter   *iter,
                                                      va_list          args);

static void           dee_proxy_model_set_value      (DeeModel       *self,
                                                      DeeModelIter   *iter,
                                                      guint            column,
                                                      const GValue    *value);

static void           dee_proxy_model_set_value_silently (DeeModel       *self,
                                                          DeeModelIter   *iter,
                                                          guint            column,
                                                          const GValue    *value);

static void           dee_proxy_model_get_valist     (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      va_list        args);

static void           dee_proxy_model_get_value      (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column,
                                                      GValue        *value);

static DeeModelIter* dee_proxy_model_get_first_iter  (DeeModel     *self);

static DeeModelIter* dee_proxy_model_get_last_iter   (DeeModel     *self);

static DeeModelIter* dee_proxy_model_get_iter_at_row (DeeModel     *self,
                                                      guint          row);

static gboolean       dee_proxy_model_get_bool       (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static guchar         dee_proxy_model_get_uchar      (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static gint           dee_proxy_model_get_int        (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static guint          dee_proxy_model_get_uint       (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static gint64         dee_proxy_model_get_int64      (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static guint64        dee_proxy_model_get_uint64     (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static gdouble        dee_proxy_model_get_double     (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static const gchar*   dee_proxy_model_get_string     (DeeModel     *self,
                                                      DeeModelIter *iter,
                                                      guint          column);

static DeeModelIter* dee_proxy_model_next            (DeeModel     *self,
                                                      DeeModelIter *iter);

static DeeModelIter* dee_proxy_model_prev            (DeeModel     *self,
                                                      DeeModelIter *iter);

static gboolean       dee_proxy_model_is_first       (DeeModel     *self,
                                                      DeeModelIter *iter);

static gboolean       dee_proxy_model_is_last        (DeeModel     *self,
                                                      DeeModelIter *iter);

static gint           dee_proxy_model_get_position   (DeeModel     *self,
                                                      DeeModelIter *iter);

static void           dee_proxy_model_freeze_signals (DeeModel     *self);

static void           dee_proxy_model_thaw_signals (DeeModel     *self);

/*
 * Callbacks for relaying signals from the back end model
 */
static void           on_back_end_row_added          (DeeProxyModel *self,
                                                      DeeModelIter  *iter);   

static void           on_back_end_row_removed        (DeeProxyModel *self,
                                                      DeeModelIter  *iter);

static void           on_back_end_row_changed        (DeeProxyModel *self,
                                                      DeeModelIter  *iter);

/*
 * Overrides for DeeVersionedModel
 */
static DeeSeqnumState
                 dee_proxy_model_check_seqnums    (DeeModel *self,
                                                   GArray *seqnums);

static DeeSeqnumState
                 dee_proxy_model_check_seqnum     (DeeModel     *self,
                                                   guint64       seqnum);

static guint64   dee_proxy_model_inc_seqnum       (DeeModel     *self,
                                                   guint         pos);

static void      dee_proxy_model_set_seqnum       (DeeModel     *self,
                                                   guint         pos,
                                                   guint64       seqnum);

static guint64   dee_proxy_model_append_next_seqnum  (DeeModel     *self);

static guint64   dee_proxy_model_prepend_next_seqnum (DeeModel     *self);

static guint64   dee_proxy_model_insert_next_seqnum  (DeeModel     *self,
                                                      guint         pos);

static guint64   dee_proxy_model_get_seqnum       (DeeModel     *self,
                                                   guint         pos);

static guint64   dee_proxy_model_get_last_seqnum  (DeeModel     *self);

static void      dee_proxy_model_set_last_seqnum  (DeeModel     *self,
                                                   guint64       seqnum);

static void      dee_proxy_model_remove_seqnum    (DeeModel     *self,
                                                   guint         pos);

/* GObject Init */
static void
dee_proxy_model_finalize (GObject *object)
{
  DeeProxyModelPrivate *priv = DEE_PROXY_MODEL (object)->priv;
  
  if (priv->back_end)
    {
      if (priv->row_added_handler != 0)
        g_signal_handler_disconnect (priv->back_end, priv->row_added_handler);
      if (priv->row_removed_handler != 0)
        g_signal_handler_disconnect (priv->back_end, priv->row_removed_handler);
      if (priv->row_changed_handler != 0)
        g_signal_handler_disconnect (priv->back_end, priv->row_changed_handler);

      g_object_unref (priv->back_end);
    }
  
  G_OBJECT_CLASS (dee_proxy_model_parent_class)->finalize (object);
}

/* GObject Post-Init. Properties has been set */
static void
dee_proxy_model_constructed (GObject *object)
{
  DeeProxyModelPrivate *priv = DEE_PROXY_MODEL (object)->priv;
  
  if (priv->back_end == NULL)
  {
    g_critical ("You must set the 'back-end' property of "
                "the DeeProxyModel upon creation.");
    return;
  }
  
  /* Connect to signals on the back-end model so we can relay them */
  if (priv->proxy_signals)
    {
      priv->row_added_handler =
        g_signal_connect_swapped (priv->back_end, "row-added",
                                  G_CALLBACK (on_back_end_row_added), object);
      priv->row_removed_handler =
        g_signal_connect_swapped (priv->back_end, "row-removed",
                                  G_CALLBACK (on_back_end_row_removed), object);
      priv->row_changed_handler =
        g_signal_connect_swapped (priv->back_end, "row-changed",
                                  G_CALLBACK (on_back_end_row_changed), object);
    }
  
  /* GObjectClass has NULL 'constructed' member, but we add this check for
   * future robustness if we ever move to another base class */
  if (G_OBJECT_CLASS (dee_proxy_model_parent_class)->constructed != NULL)
    G_OBJECT_CLASS (dee_proxy_model_parent_class)->constructed (object);
}

static void
dee_proxy_model_set_property (GObject       *object,
                              guint          id,
                              const GValue  *value,
                              GParamSpec    *pspec)
{
  DeeProxyModelPrivate *priv = DEE_PROXY_MODEL (object)->priv;
  
  switch (id)
    {
    case PROP_BACK_END:
      /* Steal ref to the back end model */
      priv->back_end = g_value_get_object (value);
      break;
    case PROP_PROXY_SIGNALS:
      priv->proxy_signals = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_proxy_model_get_property (GObject     *object,
                              guint        id,
                              GValue      *value,
                              GParamSpec  *pspec)
{
  switch (id)
    {
    case PROP_BACK_END:
      g_value_set_object (value, DEE_PROXY_MODEL (object)->priv->back_end);
      break;
    case PROP_PROXY_SIGNALS:
      g_value_set_boolean (value, DEE_PROXY_MODEL (object)->priv->proxy_signals);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_proxy_model_class_init (DeeProxyModelClass *klass)
{
  GObjectClass           *obj_class = G_OBJECT_CLASS (klass);
  DeeVersionedModelClass *dvm_class = DEE_VERSIONED_MODEL_CLASS (klass);
  GParamSpec             *pspec;

  obj_class->finalize     = dee_proxy_model_finalize;
  obj_class->constructed  = dee_proxy_model_constructed;
  obj_class->set_property = dee_proxy_model_set_property;
  obj_class->get_property = dee_proxy_model_get_property;

  dvm_class->remove_seqnum       = dee_proxy_model_remove_seqnum;
  dvm_class->get_last_seqnum     = dee_proxy_model_get_last_seqnum;
  dvm_class->set_last_seqnum     = dee_proxy_model_set_last_seqnum;
  dvm_class->get_seqnum          = dee_proxy_model_get_seqnum;
  dvm_class->insert_next_seqnum  = dee_proxy_model_insert_next_seqnum;
  dvm_class->prepend_next_seqnum = dee_proxy_model_prepend_next_seqnum;
  dvm_class->append_next_seqnum  = dee_proxy_model_append_next_seqnum;
  dvm_class->set_seqnum          = dee_proxy_model_set_seqnum;
  dvm_class->inc_seqnum          = dee_proxy_model_inc_seqnum;
  dvm_class->check_seqnum        = dee_proxy_model_check_seqnum;
  dvm_class->check_seqnums       = dee_proxy_model_check_seqnums;  

  /**
   * DeeProxyModel:back-end:
   *
   * The backend model used by this proxy model. The ownership of the ref to
   * the back end model is transfered to the proxy model.
   **/
  pspec = g_param_spec_object ("back-end", "Back end",
                               "Back end model",
                               DEE_TYPE_MODEL,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY
                               | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_BACK_END, pspec);
  
  /**
   * DeeProxyModel:proxy-signals:
   *
   * Boolean property defining whether or not to automatically forward signals
   * from the back end model. This is especially useful for sub classes wishing
   * to do their own more advanced signal forwarding.
   **/
  pspec = g_param_spec_boolean ("proxy-signals", "Proxy signals",
                                "Whether or not to automatically forward signals from the back end",
                                TRUE,
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY
                                | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_PROXY_SIGNALS, pspec);
  
  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeeProxyModelPrivate));
}

static void
dee_proxy_model_model_iface_init (DeeModelIface *iface)
{
  iface->set_column_type      = dee_proxy_model_set_column_type;
  iface->set_n_columns        = dee_proxy_model_set_n_columns;
  iface->get_n_columns        = dee_proxy_model_get_n_columns;
  iface->get_n_rows           = dee_proxy_model_get_n_rows;
  iface->get_column_type      = dee_proxy_model_get_column_type;
  iface->clear                = dee_proxy_model_clear;
  iface->prepend_valist       = dee_proxy_model_prepend_valist;
  iface->append_valist        = dee_proxy_model_append_valist;
  iface->insert_valist        = dee_proxy_model_insert_valist;
  iface->insert_before_valist = dee_proxy_model_insert_before_valist;
  iface->remove               = dee_proxy_model_remove;
  iface->set_valist           = dee_proxy_model_set_valist;
  iface->set_value            = dee_proxy_model_set_value;
  iface->set_value_silently   = dee_proxy_model_set_value_silently;
  iface->get_valist           = dee_proxy_model_get_valist;
  iface->get_value            = dee_proxy_model_get_value;
  iface->get_first_iter       = dee_proxy_model_get_first_iter;
  iface->get_last_iter        = dee_proxy_model_get_last_iter;
  iface->get_iter_at_row      = dee_proxy_model_get_iter_at_row;
  iface->get_bool             = dee_proxy_model_get_bool;
  iface->get_uchar            = dee_proxy_model_get_uchar;
  iface->get_int              = dee_proxy_model_get_int;
  iface->get_uint             = dee_proxy_model_get_uint;
  iface->get_int64            = dee_proxy_model_get_int64;
  iface->get_uint64           = dee_proxy_model_get_uint64;
  iface->get_double           = dee_proxy_model_get_double;
  iface->get_string           = dee_proxy_model_get_string;
  iface->next                 = dee_proxy_model_next;
  iface->prev                 = dee_proxy_model_prev;
  iface->is_first             = dee_proxy_model_is_first;
  iface->is_last              = dee_proxy_model_is_last;
  iface->get_position         = dee_proxy_model_get_position;
  iface->freeze_signals       = dee_proxy_model_freeze_signals;
  iface->thaw_signals         = dee_proxy_model_thaw_signals;
}

static void
dee_proxy_model_init (DeeProxyModel *model)
{
  DeeProxyModelPrivate *priv;

  priv = model->priv = DEE_PROXY_MODEL_GET_PRIVATE (model);
  priv->back_end = NULL;
  priv->inherit_seqnums = FALSE;
  
  priv->row_added_handler = 0;
  priv->row_removed_handler = 0;
  priv->row_changed_handler = 0;
}

/*
 * DeeModel Interface Implementation
 */
static void
dee_proxy_model_set_column_type (DeeModel *self,
                                 guint     column,
                                 GType     type)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_set_column_type (DEE_PROXY_MODEL_BACK_END (self), column, type);
}

static void
dee_proxy_model_set_n_columns (DeeModel *self,
                               guint     n_columns)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_set_n_columns (DEE_PROXY_MODEL_BACK_END (self), n_columns);
}

static guint
dee_proxy_model_get_n_columns (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_n_columns (DEE_PROXY_MODEL_BACK_END (self));
}

static guint
dee_proxy_model_get_n_rows (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_n_rows (DEE_PROXY_MODEL_BACK_END (self));
}

static GType
dee_proxy_model_get_column_type (DeeModel *self,
                                 guint      column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), G_TYPE_NONE);

  return dee_model_get_column_type (DEE_PROXY_MODEL_BACK_END (self), column);
}

static void
dee_proxy_model_clear (DeeModel *self)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_clear (DEE_PROXY_MODEL_BACK_END (self));
}

static DeeModelIter*
dee_proxy_model_prepend_valist (DeeModel *self,
                                va_list    args)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_prepend_valist (DEE_PROXY_MODEL_BACK_END (self), args);
}

static DeeModelIter*
dee_proxy_model_append_valist (DeeModel *self,
                               va_list    args)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_append_valist (DEE_PROXY_MODEL_BACK_END (self), args);
}

static DeeModelIter*
dee_proxy_model_insert_valist (DeeModel *self,
                               gint       pos,
                               va_list    args)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_insert_valist (DEE_PROXY_MODEL_BACK_END (self), pos, args);
}

static DeeModelIter*
dee_proxy_model_insert_before_valist (DeeModel *self,
                                      DeeModelIter *iter,
                                      va_list args)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_insert_before_valist (DEE_PROXY_MODEL_BACK_END (self), iter, args);
}

static void
dee_proxy_model_remove (DeeModel     *self,
                        DeeModelIter *iter)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_remove (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static void
dee_proxy_model_set_valist (DeeModel       *self,
                            DeeModelIter   *iter,
                            va_list         args)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_set_valist (DEE_PROXY_MODEL_BACK_END (self), iter, args);
}

static void
dee_proxy_model_set_value (DeeModel      *self,
                           DeeModelIter  *iter,
                           guint          column,
                           const GValue  *value)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_set_value (DEE_PROXY_MODEL_BACK_END (self), iter, column, value);
}

static void
dee_proxy_model_set_value_silently (DeeModel      *self,
                                    DeeModelIter  *iter,
                                    guint          column,
                                    const GValue  *value)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_set_value_silently (DEE_PROXY_MODEL_BACK_END (self),
                                iter, column, value);
}

static void
dee_proxy_model_get_valist (DeeModel       *self,
                            DeeModelIter   *iter,
                            va_list         args)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_get_valist (DEE_PROXY_MODEL_BACK_END (self), iter, args);
}

static void
dee_proxy_model_get_value (DeeModel     *self,
                           DeeModelIter *iter,
                           guint         column,
                           GValue        *value)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_get_value (DEE_PROXY_MODEL_BACK_END (self), iter, column, value);
}

static gboolean
dee_proxy_model_get_bool (DeeModel      *self,
                          DeeModelIter  *iter,
                          guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), FALSE);

  return dee_model_get_bool (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static guchar
dee_proxy_model_get_uchar (DeeModel      *self,
                           DeeModelIter  *iter,
                           guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_uchar (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static gint
dee_proxy_model_get_int (DeeModel        *self,
                         DeeModelIter    *iter,
                         guint            column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_int (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static guint
dee_proxy_model_get_uint (DeeModel      *self,
                          DeeModelIter  *iter,
                          guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_uint (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}


static gint64
dee_proxy_model_get_int64 (DeeModel      *self,
                           DeeModelIter  *iter,
                           guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_int64 (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}


static guint64
dee_proxy_model_get_uint64 (DeeModel      *self,
                            DeeModelIter  *iter,
                            guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_uint64 (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static gdouble
dee_proxy_model_get_double (DeeModel      *self,
                            DeeModelIter  *iter,
                            guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_double (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static const gchar*
dee_proxy_model_get_string (DeeModel      *self,
                            DeeModelIter  *iter,
                            guint          column)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_get_string (DEE_PROXY_MODEL_BACK_END (self), iter, column);
}

static DeeModelIter*
dee_proxy_model_get_first_iter (DeeModel     *self)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_get_first_iter (DEE_PROXY_MODEL_BACK_END (self));
}

static DeeModelIter*
dee_proxy_model_get_last_iter (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_get_last_iter (DEE_PROXY_MODEL_BACK_END (self));
}

static DeeModelIter*
dee_proxy_model_get_iter_at_row (DeeModel *self, guint row)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_get_iter_at_row (DEE_PROXY_MODEL_BACK_END (self), row);
}

static DeeModelIter*
dee_proxy_model_next (DeeModel     *self,
                      DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_next (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static DeeModelIter*
dee_proxy_model_prev (DeeModel     *self,
                      DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), NULL);

  return dee_model_prev (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static gboolean
dee_proxy_model_is_first (DeeModel     *self,
                          DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), FALSE);

  return dee_model_is_first (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static gboolean
dee_proxy_model_is_last (DeeModel     *self,
                         DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), FALSE);

  return dee_model_is_last (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static gint
dee_proxy_model_get_position (DeeModel     *self,
                              DeeModelIter *iter)
{
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);

  return dee_model_get_position (DEE_PROXY_MODEL_BACK_END (self), iter);
}

static void
dee_proxy_model_freeze_signals (DeeModel     *self)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_freeze_signals (DEE_PROXY_MODEL_BACK_END (self));
}

static void
dee_proxy_model_thaw_signals (DeeModel     *self)
{
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));

  dee_model_thaw_signals (DEE_PROXY_MODEL_BACK_END (self));
}

/*
 * Relay signals from back end
 */

static void
on_back_end_row_added (DeeProxyModel *self,
                       DeeModelIter  *iter)
{
  g_signal_emit_by_name (self, "row-added", iter);
}

static void
on_back_end_row_removed (DeeProxyModel *self,
                         DeeModelIter  *iter)
{
  g_signal_emit_by_name (self, "row-removed", iter);
}

static void
on_back_end_row_changed (DeeProxyModel *self,
                         DeeModelIter  *iter)
{
  g_signal_emit_by_name (self, "row-changed", iter);
}

/*
 * Overrides for DeeVersionedModel
 */
static DeeSeqnumState
dee_proxy_model_check_seqnums (DeeModel *self,
                               GArray *seqnums)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_check_seqnums (priv->back_end, seqnums);
  else
    return SUPER_CLASS->check_seqnums (priv->back_end, seqnums);
}

static DeeSeqnumState
dee_proxy_model_check_seqnum (DeeModel     *self,
                              guint64       seqnum)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_check_seqnum (priv->back_end, seqnum);
  else
    return SUPER_CLASS->check_seqnum (priv->back_end, seqnum);
}

static guint64
dee_proxy_model_inc_seqnum (DeeModel     *self,
                            guint         pos)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_inc_seqnum (priv->back_end, pos);
  else
    return SUPER_CLASS->inc_seqnum (priv->back_end, pos);
}

static void
dee_proxy_model_set_seqnum (DeeModel     *self,
                            guint         pos,
                            guint64       seqnum)
{
  DeeProxyModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    dee_versioned_model_set_seqnum (priv->back_end, pos, seqnum);
  else
    SUPER_CLASS->set_seqnum (priv->back_end, pos, seqnum);
}

static guint64
dee_proxy_model_append_next_seqnum (DeeModel *self)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_append_next_seqnum (priv->back_end);
  else
    return SUPER_CLASS->append_next_seqnum (priv->back_end);
}

static guint64
dee_proxy_model_prepend_next_seqnum (DeeModel *self)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_prepend_next_seqnum (priv->back_end);
  else
    return SUPER_CLASS->prepend_next_seqnum (priv->back_end);
}

static guint64
dee_proxy_model_insert_next_seqnum  (DeeModel     *self,
                                     guint         pos)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_insert_next_seqnum (priv->back_end, pos);
  else
    return SUPER_CLASS->insert_next_seqnum (priv->back_end, pos);
}

static guint64
dee_proxy_model_get_seqnum (DeeModel     *self,
                            guint         pos)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_get_seqnum (priv->back_end, pos);
  else
    return SUPER_CLASS->get_seqnum (priv->back_end, pos);
}

static guint64
dee_proxy_model_get_last_seqnum (DeeModel     *self)
{
  DeeProxyModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_PROXY_MODEL (self), 0);
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_get_last_seqnum (priv->back_end);
  else
    return SUPER_CLASS->get_last_seqnum (priv->back_end);
}

static void
dee_proxy_model_set_last_seqnum  (DeeModel     *self,
                                  guint64       seqnum)
{
  DeeProxyModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    dee_versioned_model_set_last_seqnum (priv->back_end, seqnum);
  else
    return SUPER_CLASS->set_last_seqnum (priv->back_end, seqnum);
}

static void
dee_proxy_model_remove_seqnum (DeeModel     *self,
                               guint         pos)
{
  DeeProxyModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_PROXY_MODEL (self));
  
  priv = DEE_PROXY_MODEL (self)->priv;

  if (priv->inherit_seqnums)
    return dee_versioned_model_remove_seqnum (priv->back_end, pos);
  else
    return SUPER_CLASS->remove_seqnum (priv->back_end, pos);
}
