/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Neil Jagdish Patel <neil.patel@canonical.com>
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 * NB: Inspiration for column storage taken from GtkListStore
 *     API inspired by ClutterModel by Matthew Allumn <mallum@openedhand.com>
 *                                     Neil Patel <njp@o-hand.com>
 *                                     Emmanuele Bassi <ebassi@openedhand.com>
 */
/**
 * SECTION:dee-model
 * @short_description: A generic table model interface
 * @include: dee.h
 *
 * #DeeModel is a generic table model that can hold most GLib Types as column
 * types.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>
#include <gobject/gvaluecollector.h>
#include <dbus/dbus-gtype-specialized.h>

#include "dee-model.h"
#include "dee-marshal.h"
#include "trace-log.h"

typedef DeeModelIface DeeModelInterface;
G_DEFINE_INTERFACE (DeeModel, dee_model, G_TYPE_OBJECT)

enum
{
  /* Public signals */
  DEE_MODEL_SIGNAL_ROW_ADDED,
  DEE_MODEL_SIGNAL_ROW_REMOVED,
  DEE_MODEL_SIGNAL_ROW_CHANGED,

  DEE_MODEL_LAST_SIGNAL
};

static guint32 dee_model_signals[DEE_MODEL_LAST_SIGNAL] = { 0 };

static const GType type_list[] =
{
    G_TYPE_BOOLEAN,
    G_TYPE_UCHAR,
    G_TYPE_INT,
    G_TYPE_UINT,
    G_TYPE_INT64,
    G_TYPE_UINT64,
    G_TYPE_DOUBLE,
    G_TYPE_STRING,
    G_TYPE_INVALID
};

static const gchar type_names[] =
{
  'b',
  'y',
  'i',
  'u',
  'x',
  't',
  'd',
  's'
};

static void
dee_model_default_init (DeeModelInterface *klass)
{
  /**
   * DeeModel::row-added:
   * @self: the #DeeModel on which the signal is emitted
   * @iter: a #DeeModelIter pointing to the newly added row
   *
   * Connect to this signal to be notified when a row is added to @self.
   **/
  dee_model_signals[DEE_MODEL_SIGNAL_ROW_ADDED] =
    g_signal_new ("row-added",
                  DEE_TYPE_MODEL,
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeeModelIface,row_added),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);
  /**
   * DeeModel::row-removed:
   * @self: the #DeeModel on which the signal is emitted
   * @iter: a #DeeModelIter pointing to the removed row
   *
   * Connect to this signal to be notified when a row is removed from @self.
   *   The row is still valid while the signal is being emitted.
   **/
  dee_model_signals[DEE_MODEL_SIGNAL_ROW_REMOVED] =
    g_signal_new ("row-removed",
                  DEE_TYPE_MODEL,
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeeModelIface,row_removed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);
  /**
   * DeeModel::row-changed:
   * @self: the #DeeModel on which the signal is emitted
   * @iter: a #DeeModelIter pointing to the changed row
   *
   * Connect to this signal to be notified when a row is changed.
   **/
  dee_model_signals[DEE_MODEL_SIGNAL_ROW_CHANGED] =
    g_signal_new ("row-changed",
                  DEE_TYPE_MODEL,
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeeModelIface,row_changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);
}

/**
 * dee_model_check_type:
 * @type: The #GType to check
 *
 * Check if a given type is allowed as a storage type in a DeeModel.
 * The allowed column types for #DeeModel<!-- -->s are:
 *
 * G_TYPE_BOOLEAN,
 * G_TYPE_UCHAR,
 * G_TYPE_INT,
 * G_TYPE_UINT,
 * G_TYPE_INT64,
 * G_TYPE_UINT64,
 * G_TYPE_DOUBLE,
 * G_TYPE_STRING
 *
 * Returns: %TRUE if and only if @type can be stored in a DeeModel
 */
gboolean
dee_model_check_type (GType type)
{
  gint i = 0;

  if (! G_TYPE_IS_VALUE_TYPE (type))
    return FALSE;

  while (type_list[i] != G_TYPE_INVALID)
    {
      if (g_type_is_a (type, type_list[i]))
        return TRUE;
      i++;
    }
  return FALSE;
}

/**
 * dee_model_get_type_for_signature:
 * @chr: Character signature to find corresponding #GType for
 *
 * Find the corresponding #GType gievn a DBus type signature. The type mapping
 * is as follows:
 *
 * b G_TYPE_BOOLEAN,
 * y G_TYPE_UCHAR,
 * i G_TYPE_INT,
 * u G_TYPE_UINT,
 * x G_TYPE_INT64,
 * t G_TYPE_UINT64,
 * d G_TYPE_DOUBLE,
 * s G_TYPE_STRING
 *
 * Returns: The #Gtype for @chr or #G_TYPE_NONE in case the type of @chr is
 *          not known
 */
GType
dee_model_get_type_for_signature (gchar chr)
{
  gint i;

  for (i = 0; i < G_N_ELEMENTS (type_names); i++)
    {
      if (type_names[i] == chr)
        return type_list[i];
    }

  return G_TYPE_NONE;
}

/* Return name of type or NULL on unknown types */
static gchar
_get_string_for_type (GType type)
{
  gint i;

  for (i = 0; i < G_N_ELEMENTS (type_list); i++)
    {
      if (g_type_is_a (type, type_list[i]))
        return type_names[i];
    }
  return '\0';
}

/**
 * dee_model_build_col_spec:
 * @self:
 *
 * Returns: A newly allocated string containing the column specification
 *          encoded as a DBus type signature. Free with g_free().
 */ 
gchar*
dee_model_build_col_spec (DeeModel *self)
{
  gchar             *col_spec;
  guint              i, n_columns;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  n_columns = dee_model_get_n_columns (self);

  col_spec = g_new (gchar, n_columns + 1);
  for (i = 0; i < n_columns; i++)
    {
      col_spec[i] = _get_string_for_type (dee_model_get_column_type (self, i));
    }
  col_spec[n_columns] = '\0';

  return col_spec;
}

/**
 * dee_model_set_column_type:
 * @self:
 * @column:
 * @type:
 *
 *
 */ 
void
dee_model_set_column_type (DeeModel *self,
                           guint     column,
                           GType     type)
{
  DeeModelIface *iface;
  
  g_return_if_fail (DEE_IS_MODEL (self));
  
  iface = DEE_MODEL_GET_IFACE (self);

  (* iface->set_column_type) (self, column, type);
}

/**
 * dee_model_set_n_columns:
 * @self:
 * @n_columns:
 *
 *
 */
void
dee_model_set_n_columns (DeeModel *self,
                         guint     n_columns)
{
  DeeModelIface *iface;
  
  g_return_if_fail (DEE_IS_MODEL (self));
  
  iface = DEE_MODEL_GET_IFACE (self);

  (* iface->set_n_columns) (self, n_columns);
}

/**
 * dee_model_get_n_columns:
 * @self: a #DeeModel
 *
 * Gets the number of columns in @self
 *
 * Return value: the number of columns per row in @self
 **/
guint
dee_model_get_n_columns (DeeModel *self)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_n_columns) (self);
}

/**
 * dee_model_get_n_rows:
 * @self: a #DeeModel
 *
 * Gets the number of rows in @self
 *
 * Return value: the number of rows in @self
 **/
guint
dee_model_get_n_rows (DeeModel *self)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_n_rows) (self);
}

/**
 * dee_model_get_column_type:
 * @self: a #DeeModel
 * @column: the column to get retrieve the type of
 *
 * Get the #GType of @column
 *
 * Return value: the #GType that @column holds
 */
GType
dee_model_get_column_type (DeeModel *self,
                            guint      column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_column_type) (self, column);
}

/**
 * dee_model_clear:
 * @self: a #DeeModel object to clear
 *
 * Removes all rows in the model. Signals are emitted for each row in the model
 */
void
dee_model_clear (DeeModel *self)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->clear) (self);
}

/**
 * dee_model_append:
 * @self: a #DeeModel
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * Creates and appends a new row to the end of a #DeeModel, setting the row
 * values upon creation.
 *
 * For example, to append a new row where column 0 is %G_TYPE_INT and column 1
 * is %G_TYPE_STRING:
 *
 * <informalexample><programlisting>
 *  DeeModel *model;
 *  model = dee_model_new ("org.myapp.Results", 2, G_TYPE_INT, G_TYPE_STRING);
 *  dee_model_append (model, 0, 10, 1, "Rooney", -1);
 * </programlisting></informalexample>
 */
DeeModelIter*
dee_model_append (DeeModel *self,
                   ...)
{
  DeeModelIter     *iter;
  va_list           args;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  va_start (args, self);
  iter = dee_model_append_valist (self, args);
  va_end (args);
  
  return iter;
}

DeeModelIter*
dee_model_append_valist (DeeModel *self,
                         va_list args)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->append_valist) (self, args);
}

/**
 * dee_model_prepend:
 * @self: a #DeeModel
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * Creates and prepends a new row to the beginning of a #DeeModel, setting the
 * row values upon creation.
 *
 * For example, to prepend a new row where column 0 is %G_TYPE_INT and column 1
 * is %G_TYPE_STRING:
 *
 * <informalexample><programlisting>
 *  DeeModel *model;
 *  model = dee_model_new ("org.myapp.Results", 2, G_TYPE_INT, G_TYPE_STRING);
 *  dee_model_prepend (model, 0, 10, 1, "Rooney", -1);
 * </programlisting></informalexample>
 */
DeeModelIter*
dee_model_prepend (DeeModel *self,
                    ...)
{
  DeeModelIter     *iter;
  va_list           args;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  va_start (args, self);
  iter = dee_model_prepend_valist (self, args);
  va_end (args);
  
  return iter;
}

DeeModelIter*
dee_model_prepend_valist (DeeModel *self,
                          va_list args)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->prepend_valist) (self, args);
}

/**
 * dee_model_insert:
 * @self: a #DeeModel
 * @pos: The index to insert the row on. The existing row will be pushed down
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * Creates and inserts a new row into a #DeeModel, pushing the existing
 * rows down.
 *
 * For example, to insert a new row at position 27 where column 0 is
 * %G_TYPE_INT and column 1 is %G_TYPE_STRING:
 *
 * <informalexample><programlisting>
 *  dee_model_insert (model, 27, 0, 10, 1, "Rooney", -1);
 * </programlisting></informalexample>
 */
DeeModelIter*
dee_model_insert (DeeModel *self,
                   gint       pos,
                   ...)
{
  DeeModelIter     *iter;
  va_list           args;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  va_start (args, pos);
  iter = dee_model_insert_valist (self, pos, args);
  va_end (args);
  
  return iter;
}

/**
 * dee_model_insert_valist:
 * @self: a #DeeModel
 * @pos: The index to insert the row on. The existing row will be pushed down.
 * @args: pairs of column number and value, terminated with -1
 *
 * As dee_model_insert(), but intended for language bindings.
 */
DeeModelIter*
dee_model_insert_valist (DeeModel *self,
                          gint       pos,
                          va_list    args)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->insert_valist) (self, pos, args);
}

/**
 * dee_model_insert_before:
 * @self: a #DeeModel
 * @iter: An iter pointing to the row before which to insert the new one
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * Creates and inserts a new row into a #DeeModel just before the row pointed
 * to by @iter.
 *
 * For example, to insert a new row where column 0 is %G_TYPE_INT and column 1
 * is %G_TYPE_STRING:
 *
 * <informalexample><programlisting>
 *  DeeModelIter *iter = find_my_special_row (model);
 *  dee_model_insert_before (model, iter, 0, 10, 1, "Rooney", -1);
 * </programlisting></informalexample>
 **/
DeeModelIter*
dee_model_insert_before (DeeModel *self,
                          DeeModelIter *iter,
                          ...)
{
  va_list           args;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  va_start (args, iter);
  iter = dee_model_insert_before_valist (self, iter, args);
  va_end (args);
  
  return iter;
}

/**
 * dee_model_insert_before_valist:
 * @self: a #DeeModel
 * @iter: An iter pointing to the row before which to insert the new one
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * As dee_model_insert_before(), but intended for language bindings.
 **/
DeeModelIter*
dee_model_insert_before_valist (DeeModel *self,
                                DeeModelIter *iter,
                                va_list args)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->insert_before_valist) (self, iter, args);
}

/**
 * dee_model_remove:
 * @self: a #DeeModel
 * @iter_: a #DeeModelIter pointing to the row to remove
 *
 * Removes the row at the given position from the model.
 **/
void
dee_model_remove (DeeModel     *self,
                  DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  (* iface->remove) (self, iter);
}

/**
 * dee_model_set:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @VarArgs: a list of column/value pairs, terminated by -1
 *
 * Sets the value of one or more cells in the row references by @iter. The
 * variable argument list should contain integer column numbers, each column
 * number followed by the value to be set. The list is terminated by a -1.
 *
 * For example, to set column 0 with type %G_TYPE_STRING, use:
 * <informalexample><programlisting>
 *   dee_model_set (model, iter, 0, "foo", -1);
 * </programlisting></informalexample>
 **/
void
dee_model_set (DeeModel *self,
                DeeModelIter *iter,
                ...)
{
  va_list           args;

  g_return_if_fail (DEE_IS_MODEL (self));

  /* Update data */
  va_start (args, iter);
  dee_model_set_valist (self, iter, args);
  va_end (args);
}

/**
 * dee_model_set_valist:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @args: a list of column/value location pairs, terminated by -1
 *
 * See #dee_model_set(). This version takes a va_list for language bindings.
 **/
void
dee_model_set_valist (DeeModel       *self,
                       DeeModelIter   *iter,
                       va_list          args)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->set_valist) (self, iter, args);
}

/**
 * dee_model_set_value:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: column number to set the value
 * @value: new value for cell
 *
 * Sets the data in @column for the row @iter points to, to @value. The type
 * of @value must be convertable to the type of the column.
 *
 * When this method call completes the model will emit ::row-changed. You can
 * edit the model in place without triggering the change signals by calling
 * dee_model_set_value_silently().
 **/
void
dee_model_set_value (DeeModel      *self,
                      DeeModelIter  *iter,
                      guint           column,
                      const GValue   *value)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->set_value) (self, iter, column, value);
}

/**
 * dee_model_set_value_silently:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: column number to set the value
 * @value: new value for cell
 *
 * Sets the data in @column for the row @iter points to, to @value. The type
 * of @value must be convertable to the type of the column.
 *
 * Calling this method will <emphasise>not</emphasise> trigger a ::row-changed
 * signal. Also, if @self is a sub class of #DeeVersionedModel the seqnum of the
 * affected row will <emphasise>not</emphasise> be incremented.
 **/
void
dee_model_set_value_silently (DeeModel      *self,
                              DeeModelIter  *iter,
                              guint           column,
                              const GValue   *value)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->set_value_silently) (self, iter, column, value);
}

/**
 * dee_model_get:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @VarArgs: a list of column/return location pairs, terminated by -1
 *
 * Gets the value of one or more cells in the row references by @iter. The
 * variable argument list should contain integer column numbers, each column
 * number followed by a place to store the value being retrieved. The list is
 * terminated by a -1.
 *
 * For example, to get a value from column 0 with type %G_TYPE_STRING use:
 * <informalexample><programlisting>
 *  dee_model_get (model, iter, 0, &place_string_here, -1);
 * </programlisting></informalexample>
 *
 * where place_string_here is a gchar* to be filled with the string. If
 * appropriate, the returned values have to be freed or unreferenced.
 **/
void
dee_model_get (DeeModel *self,
                DeeModelIter *iter,
                ...)
{
  va_list args;

  g_return_if_fail (DEE_IS_MODEL (self));
  g_return_if_fail (iter);

  va_start (args, iter);
  dee_model_get_valist (self, iter, args);
  va_end (args);
}

/**
 * dee_model_get_valist:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @args: a list of column/return location pairs, terminated by -1
 *
 * See #dee_model_get(). This version takes a va_list for language bindings.
 **/
void
dee_model_get_valist (DeeModel       *self,
                       DeeModelIter   *iter,
                       va_list          args)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_valist) (self, iter, args);
}

/**
 * dee_model_get_value:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: column number to retrieve the value from
 * @value: an empty #GValue to set
 *
 * Sets an intializes @value to that at @column. When done with @value
 * g_value_unset() needs to be called to free any allocated memory.
 */
void
dee_model_get_value (DeeModel     *self,
                      DeeModelIter *iter,
                      guint          column,
                      GValue        *value)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_value) (self, iter, column, value);
}

/**
 * dee_model_get_bool:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a boolean from
 *
 * Return value: if @iter and @column are valid, the boolean stored at @column.
 *  Otherwise %FALSE
 **/
gboolean
dee_model_get_bool (DeeModel      *self,
                     DeeModelIter  *iter,
                     guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), FALSE);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_bool) (self, iter, column);
}

/**
 * dee_model_get_uchar:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a uchar from
 *
 * Return value: if @iter and @column are valid, the uchar stored at @column.
 *  Otherwise 0.
 **/
guchar
dee_model_get_uchar (DeeModel      *self,
                      DeeModelIter  *iter,
                      guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), '\0');

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_uchar) (self, iter, column);
}

/**
 * dee_model_get_int:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a int from
 *
 * Return value: if @iter and @column are valid, the int stored at @column.
 *  Otherwise 0.
 **/
gint
dee_model_get_int (DeeModel        *self,
                    DeeModelIter    *iter,
                    guint             column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_int) (self, iter, column);
}

/**
 * dee_model_get_uint:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a uint from
 *
 * Return value: if @iter and @column are valid, the uint stored at @column.
 *  Otherwise 0.
 **/
guint
dee_model_get_uint (DeeModel      *self,
                     DeeModelIter  *iter,
                     guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_uint) (self, iter, column);
}


/**
 * dee_model_get_int64:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a int64 from
 *
 * Return value: if @iter and @column are valid, the int64 stored at @column.
 *  Otherwise 0.
 **/
gint64
dee_model_get_int64 (DeeModel      *self,
                      DeeModelIter  *iter,
                      guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_int64) (self, iter, column);
}


/**
 * dee_model_get_uint64:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a uint64 from
 *
 * Return value: if @iter and @column are valid, the uint64 stored at @column.
 *  Otherwise 0.
 **/
guint64
dee_model_get_uint64 (DeeModel      *self,
                       DeeModelIter  *iter,
                       guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_uint64) (self, iter, column);
}

/**
 * dee_model_get_double:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a double from
 *
 * Return value: if @iter and @column are valid, the double stored at @column.
 *  Otherwise 0.
 **/
gdouble
dee_model_get_double (DeeModel      *self,
                       DeeModelIter  *iter,
                       guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), 0);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_double) (self, iter, column);
}

/**
 * dee_model_get_string:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 * @column: the column to retrieve a string from
 *
 * Return value: if @iter and @column are valid, the string stored at @column.
 *  Otherwise %NULL.
 **/
const gchar*
dee_model_get_string (DeeModel      *self,
                       DeeModelIter  *iter,
                       guint           column)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_string) (self, iter, column);
}

/**
 * dee_model_get_first_iter:
 * @self: a #DeeModel
 *
 * Retrieves a #DeeModelIter representing the first row in @self.
 *
 * Return value: (transfer none): A #DeeModelIter (owned by @self, do not
 *  free it)
 */
DeeModelIter*
dee_model_get_first_iter (DeeModel     *self)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_first_iter) (self);
}

/**
 * dee_model_get_last_iter:
 * @self: a #DeeModel
 *
 * Retrieves a #DeeModelIter representing the last row in @self.
 *
 * Return value: (transfer none): A #DeeModelIter (owned by @self, do not
 *  free it)
 **/
DeeModelIter*
dee_model_get_last_iter (DeeModel *self)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_last_iter) (self);
}

/**
 * dee_model_get_iter_at_row:
 * @self: a #DeeModel
 * @row: position of the row to retrieve
 *
 * Retrieves a #DeeModelIter representing the row at the given index.
 *
 * Return value: (transfer none): A new #DeeModelIter, or %NULL if @row
 *   was out of bounds. The returned iter is owned by @self, so do not free it.
 **/
DeeModelIter*
dee_model_get_iter_at_row (DeeModel *self, guint row)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_iter_at_row) (self, row);
}

/**
 * dee_model_next:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 *
 * Returns a #DeeModelIter that points to the next position in the model.
 *
 * Return value: (transfer none): A #DeeModelIter, pointing to the next row in
 *   the model. The iter is owned by @self, do not free it.
 **/
DeeModelIter*
dee_model_next (DeeModel     *self,
                 DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->next) (self, iter);
}

/**
 * dee_model_prev:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 *
 * Returns a #DeeModelIter that points to the previous position in the model.
 *
 * Return value: (transfer none): A #DeeModelIter, pointing to the previous
 *   row in the model. The iter is owned by @self, do not free it.
 **/
DeeModelIter *
dee_model_prev (DeeModel     *self,
                 DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), NULL);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->prev) (self, iter);
}

/**
 * dee_model_is_first:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 *
 * Where @iter is at the start of @self.
 *
 * Return value: #TRUE if @iter is the first iter in the model
 */
gboolean
dee_model_is_first (DeeModel     *self,
                     DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), FALSE);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->is_first) (self, iter);
}

/**
 * dee_model_is_last:
 * @self: a #DeeModel
 * @iter: a #DeeModelIter
 *
 * Whether @iter is at the end of @self.
 *
 * Return value: #TRUE if @iter is the last iter in the model
 */
gboolean
dee_model_is_last (DeeModel     *self,
                    DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), FALSE);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->is_last) (self, iter);
}

/**
 * dee_model_get_position:
 * @self:
 * @iter:
 *
 * Returns: The integer offset of @iter in @self
 */
gint
dee_model_get_position (DeeModel     *self,
                         DeeModelIter *iter)
{
  DeeModelIface *iface;

  g_return_val_if_fail (DEE_IS_MODEL (self), -1);

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->get_position) (self, iter);
}

/**
 * dee_model_freeze_signals:
 * @self: The model for which to freeze the change notification signals
 *
 * Increase the freeze count of @self by one. When the freeze count of a model
 * is greater than 0 it will not emit any of the signals ::row-added
 * or ::row-changed, but simply queue them up.
 * Once the freeze count hits zero (by calling dee_model_thaw_signals()) all
 * signals will be played back in the order they where originally triggered.
 *
 * Note that the ::row-removed signal will be emitted disregarding the freeze
 * count. It has to be this way because otherwise the row iter would
 * be invalid once the signal was emitted.
 */
void
dee_model_freeze_signals (DeeModel     *self)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->freeze_signals) (self);
}

/**
 * dee_model_thaw_signals:
 * @self: The model for which to freeze the 
 *
 * Decrease the freeze count of @self by one. When the freeze count of a model
 * is greater than 0 it will not emit any of the signals ::row-added
 * or ::row-changed, but simply queue them up.
 * Once the freeze count hits zero (by calling this method) all
 * signals will be played back in the order they where originally triggered.
 *
 * Note that the ::row-removed signal will be emitted disregarding the freeze
 * count. It has to be this way because otherwise the row iter would
 * be invalid once the signal was emitted.
 */
void
dee_model_thaw_signals (DeeModel     *self)
{
  DeeModelIface *iface;

  g_return_if_fail (DEE_IS_MODEL (self));

  iface = DEE_MODEL_GET_IFACE (self);

  return (* iface->thaw_signals) (self);
}
