/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dbusmodel/dbusmodel.h>

typedef struct 
{
  DbusModel *model;

} ColumnFixture;

static void column_setup    (ColumnFixture *fix, gconstpointer data);
static void column_teardown (ColumnFixture *fix, gconstpointer data);

static void test_column_allocation   (ColumnFixture *fix, gconstpointer data);
static void test_column_types        (ColumnFixture *fix, gconstpointer data);
static void test_column_modification (ColumnFixture *fix, gconstpointer data);
static void test_column_bad_types    (void);

void
test_model_column_create_suite (void)
{
#define DOMAIN "/Model/Column"

  g_test_add (DOMAIN"/Allocation", ColumnFixture, 0,
              column_setup, test_column_allocation, column_teardown);
  g_test_add (DOMAIN"/Types", ColumnFixture, 0,
              column_setup, test_column_types, column_teardown);
  g_test_add (DOMAIN"/Modification", ColumnFixture, 0,
              column_setup, test_column_modification, column_teardown);
  g_test_add_func (DOMAIN"/BadTypes", test_column_bad_types);
}

static void
column_setup (ColumnFixture *fix, gconstpointer data)
{
  fix->model = dbus_model_new ("com.canonical.DbusModel.Tests.Columns",
                               8,
                               G_TYPE_BOOLEAN,
                               G_TYPE_UCHAR,
                               G_TYPE_INT,
                               G_TYPE_UINT,
                               G_TYPE_INT64,
                               G_TYPE_UINT64,
                               G_TYPE_DOUBLE,
                               G_TYPE_STRING);

  g_assert (DBUS_IS_MODEL (fix->model));
}

static void
column_teardown (ColumnFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
}

static void
test_column_allocation (ColumnFixture *fix, gconstpointer data)
{
  g_assert (DBUS_IS_MODEL (fix->model));
}

static void
test_column_types (ColumnFixture *fix, gconstpointer data)
{
  DbusModel     *model = fix->model;
  DbusModelIter *iter;
  GValue         value = { 0, };

  dbus_model_add (fix->model,
                  0, TRUE,
                  1, '1',
                  2, G_MININT,
                  3, G_MAXUINT,
                  4, G_MAXINT64,
                  5, G_MAXUINT64,
                  6, G_MAXDOUBLE,
                  7, "Hello World",
                  -1);

  g_assert (dbus_model_get_n_rows (model) == 1);

  iter = dbus_model_get_first_iter (model);

  dbus_model_get_value (model, iter, 0, &value);
  g_assert (g_value_get_boolean (&value));
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 1, &value);
  g_assert (g_value_get_uchar (&value) == '1');
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 2, &value);
  g_assert (g_value_get_int (&value) == G_MININT);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 3, &value);
  g_assert (g_value_get_uint (&value) == G_MAXUINT);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 4, &value);
  g_assert (g_value_get_int64 (&value) == G_MAXINT64);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 5, &value);
  g_assert (g_value_get_uint64 (&value) == G_MAXUINT64);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 6, &value);
  g_assert (g_value_get_double (&value) == G_MAXDOUBLE);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 7, &value);
  g_assert_cmpstr (g_value_get_string (&value), ==, "Hello World");
  g_value_unset (&value);
}

static void
test_column_modification (ColumnFixture *fix, gconstpointer data)
{
  DbusModel     *model = fix->model;
  DbusModelIter *iter;
  GValue         value = { 0, };

  dbus_model_add (fix->model,
                  0, TRUE,
                  1, '1',
                  2, G_MININT,
                  3, G_MAXUINT,
                  4, G_MAXINT64,
                  5, G_MAXUINT64,
                  6, G_MAXDOUBLE,
                  7, "Hello World",
                  -1);

  g_assert (dbus_model_get_n_rows (model) == 1);

  iter = dbus_model_get_first_iter (model);

  dbus_model_set (model, iter, 
                  0, FALSE,
                  1, '2',
                  2, G_MININT+5,
                  3, G_MAXUINT-5,
                  4, G_MAXINT64-5,
                  5, G_MAXUINT64-5,
                  6, G_MAXDOUBLE-5.0,
                  7, "World Hello",
                  -1);

  dbus_model_get_value (model, iter, 0, &value);
  g_assert (!g_value_get_boolean (&value));
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 1, &value);
  g_assert (g_value_get_uchar (&value) == '2');
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 2, &value);
  g_assert (g_value_get_int (&value) == G_MININT+5);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 3, &value);
  g_assert (g_value_get_uint (&value) == G_MAXUINT-5);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 4, &value);
  g_assert (g_value_get_int64 (&value) == G_MAXINT64-5);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 5, &value);
  g_assert (g_value_get_uint64 (&value) == G_MAXUINT64-5);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 6, &value);
  g_assert (g_value_get_double (&value) == G_MAXDOUBLE-5.0);
  g_value_unset (&value);

  dbus_model_get_value (model, iter, 7, &value);
  g_assert_cmpstr (g_value_get_string (&value), ==, "World Hello");
  g_value_unset (&value);
}

static void
test_column_bad_types (void)
{
  gint           n_types, i;
  /* Some random types which we shouldn't support */
  static const GType types[] =
    {
      G_TYPE_BOXED,
      G_TYPE_CHAR,
      G_TYPE_ENUM,
      G_TYPE_FLAGS,
      G_TYPE_FLOAT,
      G_TYPE_OBJECT,
      G_TYPE_LONG,
    };

  n_types = G_N_ELEMENTS (types);

  for (i = 0; i < n_types; i++)
    {
      /* Make sure we can't create a model with GTypes which we don't support */
      if (g_test_trap_fork (0, G_TEST_TRAP_SILENCE_STDOUT
                               | G_TEST_TRAP_SILENCE_STDERR))
        {
          DbusModel *model = NULL;

          model = dbus_model_new ("com.canonical.DbusMenu.Test",
                                  1,
                                  types[i]);
          g_assert (model == NULL);
        }
      g_test_trap_assert_failed ();
    }
}
