/***
 *  This file is part of Clutter-Gesture.
 *
 *  Copyright 2009 (c) Intel Corp.
 *  Author: Long Bu    (long.bu@intel.com)
 *
 *  Clutter-Gesture is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published
 *  by the Free Software Foundation; either version 2.1 of the License,
 *  or (at your option) any later version.
 *
 *  Clutter-Gesture is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with ClutterGesture; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 *  USA.
 ***/
#include <clutter/clutter.h>
#include "math.h"
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <cairo/cairo.h>
#include <pango/pangocairo.h>
#include <GL/gl.h>
#include <glib-object.h>

#include "clutter-gesture.h"
#include "engine.h"


G_DEFINE_TYPE (ClutterGesture, clutter_gesture, G_TYPE_OBJECT);

#define CLUTTER_GESTURE_GET_PRIVATE(obj)          \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), CLUTTER_TYPE_GESTURE, ClutterGesturePrivate))


enum
{
  PROP_0,
};

struct _ClutterGesturePrivate
{
  ClutterActor *root_actor;
  handle_t gesture_handle;
  GSList *actors;  /* interesting list of actores */
};


static GMutex *lock = NULL;
static GSList *root_actors = NULL;

static void
clutter_gesture_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec)
{
  ClutterGesture *gesture;
  ClutterGesturePrivate *priv;

  gesture = CLUTTER_GESTURE (object);
  priv = gesture->priv;

  switch (prop_id)
  {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
clutter_gesture_get_property (GObject      *object,
                              guint         prop_id,
                              GValue       *value,
                              GParamSpec   *pspec)
{
  ClutterGesture *gesture;
  ClutterGesturePrivate *priv;

  gesture = CLUTTER_GESTURE (object);
  priv = gesture->priv;


  switch (prop_id)
  {

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

enum
{
  GEST_SLIDE,
  GEST_PINCH,
  GEST_ROTATE,
  GEST_WINDOW,
  GEST_ANY,

  LAST_SIGNAL
};

static guint  gesture_signals[LAST_SIGNAL] = { 0, };
static void
clutter_gesture_finalize (GObject *object)
{
  ClutterGestureClass *klass;
  ClutterGesture *gesture;

  gesture = CLUTTER_GESTURE(object);
  klass = CLUTTEER_GESTURE_GET_CLASS(gesture);
  if (klass) {
    ClutterGesturePrivate *priv;
    GSList *dummy;

    priv = gesture->priv;

    g_mutex_lock (lock);
    dummy = g_slist_remove (root_actors, priv->root_actor);
    g_mutex_unlock (lock);
  }
  
  G_OBJECT_CLASS (clutter_gesture_parent_class)->finalize (object);
}

#if 0
  gboolean
_clutter_boolean_handled_accumulator (GSignalInvocationHint *ihint,
    GValue                *return_accu,
    const GValue          *handler_return,
    gpointer               dummy)
{
  gboolean continue_emission;
  gboolean signal_handled;

  signal_handled = g_value_get_boolean (handler_return);
  g_value_set_boolean (return_accu, signal_handled);
  continue_emission = !signal_handled;

  return continue_emission;
}
#endif

static void
clutter_gesture_class_init (ClutterGestureClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->finalize     = clutter_gesture_finalize;
  object_class->set_property = clutter_gesture_set_property;
  object_class->get_property = clutter_gesture_get_property;

  if (!g_thread_supported ())
    g_thread_init(NULL);
  lock = g_mutex_new();

  g_type_class_add_private (klass, sizeof (ClutterGesturePrivate));

  gesture_signals[GEST_SLIDE] = 
  g_signal_new ("gesture-slide-event",
                G_TYPE_FROM_CLASS (object_class),
                G_SIGNAL_RUN_LAST,
                G_STRUCT_OFFSET (ClutterGestureClass, gesture_slide_event),
                NULL,
                NULL,
                g_cclosure_marshal_VOID__POINTER,
                G_TYPE_NONE, 1,
                G_TYPE_UINT);

  printf("slide signal:%d\n", gesture_signals[GEST_SLIDE]);

  gesture_signals[GEST_PINCH] = 
    g_signal_new ("gesture-pinch-event",
        G_TYPE_FROM_CLASS (object_class),
        G_SIGNAL_RUN_LAST,
        G_STRUCT_OFFSET (ClutterGestureClass, gesture_pinch_event),
        NULL, NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE, 1,
        G_TYPE_UINT);
  /*
        clutter_marshal_BOOLEAN__BOXED,
        G_TYPE_BOOLEAN, 1,
        CLUTTER_TYPE_GESTURE);  */

  printf("pinch signal:%d\n", gesture_signals[GEST_PINCH]);
  gesture_signals[GEST_ROTATE] = 
    g_signal_new ("gesture-rotate-event",
        G_TYPE_FROM_CLASS (object_class),
        G_SIGNAL_RUN_LAST,
        G_STRUCT_OFFSET (ClutterGestureClass, gesture_rotate_event),
        NULL, NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE, 1,
        G_TYPE_UINT);

  printf("rotate signal:%d\n", gesture_signals[GEST_ROTATE]);
  gesture_signals[GEST_WINDOW] = 
    g_signal_new ("gesture-window-event",
        G_TYPE_FROM_CLASS (object_class),
        G_SIGNAL_RUN_LAST,
        G_STRUCT_OFFSET (ClutterGestureClass, gesture_window_event),
        NULL, NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE, 1,
        G_TYPE_UINT);

  printf("window signal:%d\n", gesture_signals[GEST_WINDOW]);

  gesture_signals[GEST_ANY] = 
    g_signal_new ("gesture-any-event",
        G_TYPE_FROM_CLASS (object_class),
        G_SIGNAL_RUN_LAST,
        G_STRUCT_OFFSET (ClutterGestureClass, gesture_any_event),
        NULL, NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE, 1,
        G_TYPE_UINT);

  printf("any signal:%d\n", gesture_signals[GEST_ANY]);
}

static void
clutter_gesture_init (ClutterGesture *gesture)
{
  gesture->priv = G_TYPE_INSTANCE_GET_PRIVATE (gesture, CLUTTER_TYPE_GESTURE, ClutterGesturePrivate);
}

static gboolean
is_a_child(ClutterActor *parent, ClutterActor *child)
{
  ClutterActor *t;
  if (!parent || !child)
    return FALSE;
  
  for (t = child; t; t = clutter_actor_get_parent(t))
  {
    if (t == parent)
      return TRUE;
  }

  return FALSE;
}

static gboolean
captured_event_cb (ClutterActor *root_actor, ClutterEvent *event, gpointer data)
{
  ClutterGesturePrivate *priv;
  ClutterGesture *gesture;
  GSList *node;
  gboolean ret;
  g_print("root_captured_event_cb\n");

  gesture = (ClutterGesture *)data;
  priv = gesture->priv;

  {
    ClutterActor *stage = CLUTTER_ACTOR(event->any.stage);
    stage = NULL;
  }
  ret = FALSE;
  if (event->any.flags & (1<<31))
    return ret;  /* this event has been through the engine */

  /* Check if the event->any.source is a child of any actors in the list
   * of interesting actors */
  for (node = priv->actors; node; node = g_slist_next(node)) {
    ClutterActor *interesting_actor;
    if (!node->data)
      continue;

    interesting_actor = (ClutterActor *)(node->data);
    if (is_a_child(interesting_actor, event->any.source)) {
      ret = fill_event(priv->gesture_handle, interesting_actor, event);
      break;
    }
  }

  if (!ret) {
    /* The event is not eaten by the engine.
     * Pass it alone*/
    return FALSE;
  } else {
    /* Before we return true. We need to pass
     * the event to other instance of ClutterGesture.
     * Otherwise, the captured_event_cb of other ClutterGesture instance
     * may not have a change to be called */
    ClutterGestureClass *klass;
    gboolean dummy;
    klass = CLUTTEER_GESTURE_GET_CLASS(gesture);
    g_mutex_lock(lock);
    for (node = root_actors; node; node = g_slist_next(node)) {
      ClutterActor *root_actor = (ClutterActor *)node->data;
      if (root_actor != priv->root_actor) {
        /*don't need to send this event to myself */
        g_signal_emit_by_name (root_actor, "catpured-event", event, &dummy);
      }
    }
    g_mutex_unlock(lock);

    return TRUE;
  }
}

/* which application is interested in which gestures*/
gboolean
clutter_gesture_set_gesture_mask (ClutterGesture *gesture,
	                               ClutterActor   *actor,
                                 gint mask)
{
  ClutterGesturePrivate *priv;

  priv = gesture->priv;

  if (!actor)
    return FALSE;

  /* The actor must be a child of the root_actor.
   * Otherwise, no event can be captured for the actor */
  if (!is_a_child(priv->root_actor, actor))
    return FALSE;

  priv->actors = g_slist_append(priv->actors, actor);

  return set_gesture_mask(priv->gesture_handle, actor, mask);
}

static void
event_list_free (GSList *l)
{
  GSList *node;
  for (node = l; node; node = g_slist_next(node)) {
    ClutterEvent *event = (ClutterEvent *)(node->data);
    clutter_event_free(event);
  }

  g_slist_free(l);
}

static inline void
emit_event (ClutterEvent *event,
            gboolean      is_key_event)
{
  static gboolean      lock = FALSE;

  GPtrArray *event_tree = NULL;
  ClutterActor *actor;
  gint i = 0;

  if (!event->any.source)
  {
    g_warning("No source set, discarding event");
    return;
  }

  /* reentrancy check */
  if (lock != FALSE)
  {
    g_warning ("Tried emitting event during event delivery, bailing out.n");
    return;
  }

  lock = TRUE;

  event_tree = g_ptr_array_sized_new (64);

  actor = event->any.source;

  /* Build 'tree' of emitters for the event */
  while (actor)
  {
    ClutterActor *parent;

    parent = clutter_actor_get_parent (actor);
    if (clutter_actor_get_reactive (actor) ||
        parent == NULL ||         /* stage gets all events */
        is_key_event)             /* keyboard events are always emitted */
    {
      g_ptr_array_add (event_tree, g_object_ref (actor));
    }

    actor = parent;
  }

  /* Capture */
  for (i = event_tree->len - 1; i >= 0; i--)
    if (clutter_actor_event (g_ptr_array_index (event_tree, i), event, TRUE))
      goto done;

  /* Bubble */
  for (i = 0; i < event_tree->len; i++)
    if (clutter_actor_event (g_ptr_array_index (event_tree, i), event, FALSE))
      goto done;

done:
  for (i = 0; i < event_tree->len; i++)
    g_object_unref (g_ptr_array_index (event_tree, i));

  g_ptr_array_free (event_tree, TRUE);

  lock = FALSE;
}

static gboolean
process_clutter_event (gpointer data)
{
  GSList *node;
  GSList *event_list = (GSList *)data;
  g_print("process_clutter_event: type: CLUTTER_EVENT\n");
  for (node = event_list; node; node = g_slist_next(node)) {
    ClutterEvent *event = (ClutterEvent *)(node->data);

    event->any.flags |= (1<<31);  /* indicates that this event has been processed by the engine */
    g_print("process_clutter_event: do_event\n");
    g_assert(event->type == CLUTTER_MOTION ||
             event->type == CLUTTER_BUTTON_PRESS ||
             event->type == CLUTTER_BUTTON_RELEASE ||
             event->type == CLUTTER_SCROLL);
    emit_event (event, FALSE);
//    clutter_do_event (event);
  }

  event_list_free(event_list);

  return FALSE;
}

static GSList *
event_list_copy (GSList *l)
{
  GSList *node;
  GSList *ret = NULL;
  for (node = l; node; node = g_slist_next(node)) {
    ClutterEvent *event = clutter_event_copy(node->data);
    ret = g_slist_append(ret, event);
  }
  return ret;
}

static void
recognize_cb (ClutterActor *target_actor,
              event_type_t  event_type,
              GSList       *event_list,
              gint          number,
              void         *data)
{
  GSList *node;
  ClutterGesture *gesture = (ClutterGesture *)data;
  g_print("recognized_cb gets called, type:%d, gesture:%p\n", event_type, data);
  if (event_type == GESTURE_EVENT) {
    g_print("recognized_cb: type: GESTURE_EVENT, number:%d\n", number);
    for (node = event_list; node; node = g_slist_next(node)) {
      ClutterGestureEvent *event = (ClutterGestureEvent *)(node->data);
      gboolean dummy;

      switch (event->type) {
        case GESTURE_PINCH:
          g_signal_emit (gesture,
                         gesture_signals[GEST_PINCH],
                         0,
                         event,
                         &dummy);
          break;
        case GESTURE_SLIDE:
          g_signal_emit (gesture,
                         gesture_signals[GEST_SLIDE],
                         0,
                         event,
                         &dummy);
          break;

        case GESTURE_ROTATE:
          g_signal_emit (gesture,
                         gesture_signals[GEST_ROTATE],
                         0,
                         event,
                         &dummy);
          break;

        case GESTURE_WINDOW:
          g_signal_emit (gesture,
                         gesture_signals[GEST_WINDOW],
                         0,
                         event,
                         &dummy);
          break;
        default:
          break;
      }

      if (event->type == GESTURE_PINCH || 
          event->type == GESTURE_SLIDE || event->type ==GESTURE_ROTATE) {
        g_signal_emit (gesture, gesture_signals[GEST_ANY], 0,
                       event, &dummy); 
      }

    }
  } else if (event_type == CLUTTER_EVENT) {
    /* The event_list will be free before the timeout_cb gets called */
    g_timeout_add(0, process_clutter_event, event_list_copy(event_list));
  }

}

ClutterGesture *
clutter_gesture_new (ClutterActor *actor)
{
  ClutterGesture *gesture;
  ClutterGesturePrivate *priv;
  handle_t gesture_handle;
  ClutterGestureClass *klass;

  if (!actor)
    return NULL;

  gesture = g_object_new (CLUTTER_TYPE_GESTURE, NULL);
  if (!gesture)
    return NULL;
  priv = gesture->priv;
  priv->root_actor = actor;

  g_signal_connect (G_OBJECT (actor), "captured-event",
                    G_CALLBACK (captured_event_cb), gesture);

  /*By default no gestures should be interesting */
  gesture_handle = engine_init (recognize_cb, gesture);
  if (!gesture_handle) {
    g_object_unref (gesture);
    return NULL;
  }

  priv->gesture_handle = gesture_handle;

  klass = CLUTTEER_GESTURE_GET_CLASS (gesture);
  if (!klass) {
    g_object_unref (gesture);
    engine_shutdown (gesture_handle);
    return NULL;
  }

  g_mutex_lock (lock);
  root_actors = g_slist_prepend (root_actors, actor);
  g_mutex_unlock (lock);

  return gesture;
}
