# -*- coding: ascii -*-

###########################################################################
# clive, video extraction utility
# Copyright (C) 2007-2008 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

## The "nomad" core classes.

import sys
import os
import gzip
import time
import math
import subprocess
import string
import re

try:
	from cStringIO import StringIO
except ImportError:
	import StringIO

from urlgrabber.grabber import URLGrabber, URLGrabError

import clive as _clive
from clive.console import Console
from clive.parse import PageParser
from clive.opts import Options
from clive.update import Update
from clive.progress import Progress, ProgressVerify
from clive.error import CliveError

__all__ = ['Nomad']


## The "nomad" engine and core class of clive.
class Nomad:

	## Entry point for the engine.
	# \param self The object pointer
	# \param opts The parsed runtime options returned by optparse.OptionParser
	# \param args The parsed command line arguments (optparse.OptionParser)
	# \param say The pointer to a stdout method used throughout life-time
	def run(self, opts, args, say):
		self._opts = opts
		self._args = args
		self._say = say

		self._found_urls = [] # [(video_url, video_filename, video_size),]
		self._error_urls = [] # [(raw_url, errmsg),]
		self._dld_videos = [] # [(video_filename),]
		self._encoded_videos = [] # [(video_filename, encoded_video_filename),]

		if opts.check_updates:
			self._check_updates()

		if opts.configure:
			self._configure()

		if opts.clear_logs:
			self._clear_logs()

		if opts.history:
			raw_urls = self._browse_history()
		else:
			raw_urls = self._get_raw_urls(args)

		self._check_raw_urls(raw_urls)

		if not opts.emit:
			self._dl()
			if opts.re_encode != 'off':
				self._re_encode()
			if opts.play != 'off':
				self._play()
		else:
			self._emit()

	def _check_updates(self):
		Update(self._say, self._opts, self._get_proxy()).check()
		raise SystemExit # !

	def _browse_history(self):
		if not self._opts._tkinter_avail:
			raise SystemExit('error: tkinter module not found.')

		from clive.history import History

		sel = History().main()

		if len(sel) > 0:
			return sel
		else:
			raise SystemExit # !

	def _configure(self):
		if not self._opts._tkinter_avail:
			raise SystemExit('error: tkinter module not found.')

		from clive.configure import Configure

		Configure().main()
		raise SystemExit # !

	def _clear_logs(self):
		Options().clear_logs()
		self._say('status: logs cleared.')
		raise SystemExit # !

	def _emit(self):
		for video in self._found_urls:
			(url, vurl, hlen, length, fn, reget, exists) = video
			self._say('video: "%s" "%s" "%s"\n' % (
				vurl, os.path.basename(fn), hlen),
				show_always=1
			)

	def _dl(self):
		if len(self._found_urls) == 0:
			raise SystemExit('error: nothing to extract.')

		for video in self._found_urls:
			(url, vurl, hlen, length, fn, reget, exists) = video
			try:
				if not exists:
					self._dl_video(vurl, fn, reget)

				self._dld_videos.append((vurl, fn))

			except URLGrabError, err:
				self._say('%s [%s]' % (err.strerror,url), is_error=1)
				self._error_urls.append((url, err.strerror))
	
	def _re_encode(self):
		try:
			for (url, file) in self._dld_videos:

				if file.lower().endswith('.%s' % self._opts.re_encode):
					s = 're-encode: source (%s) is already %s. skipping.' % (
						os.path.basename(file), self._opts.re_encode)
					self._say(s)
					continue

				output_file = '%s.%s' % (file, self._opts.re_encode)
				output_file_stderr = '%s.stderr' % output_file

				self._say('re-encode: %s -> %s' % (
					os.path.basename(file), os.path.basename(output_file)))

				cmd = self._opts.ffmpeg.replace('%','$')
				d = {'i':'"%s"'%file, 'o':'"%s"'%output_file}
				cmd = string.Template(cmd).substitute(d)
				
				if sys.platform != 'win32':
					cmd += ' >/dev/null'
				else:
					cmd += ' >nul'
				
				cmd += ' 2>"%s"' % output_file_stderr

				rc = subprocess.call(cmd, shell=1)

				if rc != 0:
					self._say('warn: ffmpeg returned an error, ' +
						'dumping stderr (%s).' % output_file_stderr)
				else:
					try:
						os.remove('%s' % output_file_stderr)
					except:
						pass
					self._encoded_videos.append((file, output_file))

		except OSError, e:
			self._say(str(e), is_error=1)
	
	def _play(self):
		play_videos = self._dld_videos

		if self._opts.play != 'src':
			play_videos = self._encoded_videos

		for (ignore, file) in play_videos:
			self._say('play: %s' % os.path.basename(file))

			cmd = self._opts.player.replace('%','$')
			cmd = string.Template(cmd).substitute({'i':'"%s"'%file})

			if sys.platform != 'win32':
				cmd += ' >/dev/null'
			else:
				cmd += ' >nul'
			
			cmd += ' 2>&1'

			rc = subprocess.call(cmd, shell=1)

			if rc != 0:
				self._say('warn: player returned an error (%d).' % rc)

	def _get_raw_urls(self, args):
		if not args:
			c = Console(self._opts, self._say)
			(urls, self._opts) = c.read()
			return urls
		else:
			return args
	
	def _check_raw_urls(self, raw_urls):
		for (index, url) in enumerate(raw_urls):
			self._check_url(url, (index,len(raw_urls)))
		self._show_queue(raw_urls)

	def _check_host(self, url):
		found = 0
		for host in PageParser()._supported_hosts:
			if url.lower().find(host[0]) != -1:
				found = 1
				break

		if not found:
			raise CliveError('error: unsupported host (%s)' % url)

	def _check_url(self, url, index):
		if not url.startswith('http://'):
			url = 'http://' + url

		url = url.replace('/v/','/watch?v=') # ytube

		url = url.replace('/googleplayer.swf?docId=',
			'/videoplay?docid=') # vgoogle

		try:
			self._check_host(url)

			http_headers = ()
			if self._opts.gzip:
				http_headers += ('accept-encoding', 'gzip'),

			g = URLGrabber(
				user_agent = self._opts.user_agent,
				http_headers = http_headers,
				progress_obj = ProgressVerify(index),
				throttle = self._opts.throttle,
				proxies = self._get_proxy()
			)

			o = g.urlopen(url)
			data = o.read()

			self._say("\n", show_always=1)

			if o.hdr.get('content-encoding') == 'gzip':
				data = gzip.GzipFile(fileobj=StringIO(data)).read()

			a = PageParser().parse(data, url, self._opts,
				self._callb_check_file_len, self._say)

			self._found_urls.append(a)

		except URLGrabError, err:
			self._say('%s [%s]' % (err.strerror,url), is_error=1)
			self._error_urls.append((url, err.strerror))

		except CliveError, err:
			self._say(str(err), is_error=1)
			self._error_urls.append((url, str(err)))

	def _callb_check_file_len(self, video_url):
		self._say('status: checking file length... ', newline=0)

		g = URLGrabber(
			user_agent = self._opts.user_agent,
			throttle = self._opts.throttle,
			proxies = self._get_proxy()
		)

		o = g.urlopen(video_url)
		o.close()

		bytes = long(o.hdr.get('content-length'))
		mbytes = '%.3fMB' % (float(bytes) / (1024*1024))

		self._say('%s' % mbytes)

		return (mbytes, bytes)

	def _show_queue(self, raw_urls):
		opts = Options()

		total_mb = 0
		total_count = 0
		total_skipped = 0

		for (index, video) in enumerate(self._found_urls):
			(url, vurl, hlen, length, fn, reget, exists) = video
			if not exists:
				total_count += 1
				total_mb += float(hlen.strip("MB"))
				if self._opts.log: # update ~/.clive/history.log
					opts.write_history(self._found_urls[index])
			else:
				total_skipped += 1

		self._say('queue: %d (total: %.3fMB), failed: %d, skipped: %d.' % (
			total_count, total_mb, len(self._error_urls), total_skipped))

		if self._opts.log: # update ~/.clive/last.log
			if total_count > 0:
				opts.write_last_log(raw_urls)
			else:
				if total_skipped > 0:
					opts.write_last_log(raw_urls)

	def _dl_video(self, url, file, reget, omit_say=0):
		if not omit_say:
			self._say('write: %s' % file)

		try:
			g = URLGrabber(
				user_agent = self._opts.user_agent,
				throttle = self._opts.throttle,
				progress_obj = Progress(),
				proxies = self._get_proxy(),
				reget = reget
			)

			file = g.urlgrab(url, filename=file)

		except URLGrabError, err:
			try:
				# 403: Fail-safe for resuming dl, overwrite file
				# if server returns 403 (forbidden)
				if getattr(err, 'code') == 403:
					if reget == 'simple':
						self._say('server forbids resuming; ' + 
							'forcing `--overwrite`.')
						self._say('hit ctrl-c within the next 5s ' +
							'to cancel.')
						time.sleep(5)
						self._dl_video(url, file, reget=None, omit_say=1)

			except AttributeError:
				pass

			raise err # Unhandled, pass the error on

	def _get_proxy(self):
		# If self._opts.proxy == None:
		#	proxy was not set and http_proxy env. var. was not used,
		#	see also clive/opts.py which checks for the http_proxy
		#
		# NOTE: Returning 'None' from this method disables the use
		#	of proxy altogether.
		proxy = self._opts.proxy
		if not proxy:
			return None
		else:
			if not proxy.startswith('http://'):
				proxy = 'http://' + proxy
		return {'http':proxy}


