/* Convention: 
 * We are handling bidirectional communications with dbus in more than one way:
 * - we can register on the bus and other dbus objects will call us through they
 *   bridge. we will reply to them.
 * - we can call other objects on the bus and they will reply to us.
 * 
 * All methods and messages have a prefix, js2dbus or dbus2js which identifies
 * the direction of the *original* communication. For example if the JS client
 * makes a call to something on the dbus it uses js2dbus_call, but the return
 * value is sent in a js2dbus_return message: the message is flowing backwards
 * but it's part of a conversation initiated by JS, so it stil has prefix js2dbus
 */

function BridgeClient(serverAddress)
{
   this.serverAddress = serverAddress;
   this.socket = null;
   this.interfaces = {};
   this.on_connected = null;
   this.on_disconnected = null;
   this.js2dbus_last_call_id = 0;
   this.js2dbus_call_queue = {};
}

BridgeClient.prototype.connect = function()
{
   if (this.socket !== null) this.disconnect();
   this.socket = new WebSocket(this.serverAddress);
   this.socket.bridge = this;

   this.socket.onopen = function(event)
   {
      console.log("Connected");
      if (this.bridge.on_connected instanceof Function)
         this.bridge.on_connected();
   }
   
   this.socket.onclose = function(e) {
      console.log("Disconnected");
      if (this.bridge.on_disconnected instanceof Function)
         this.on_disconnected();
      this.methods = {};
   }
   
   this.socket.onmessage = function(event)
   {
      console.log("Got message: " + event.data);
      var packet = JSON.parse(event.data);
      var cmd = packet['cmd'];
      switch (cmd) {
         case 'dbus2js_call': // they are calling, we handle it
            this.bridge.dbus2js_call(packet['call_id'], packet['interface'],
                                     packet['method'], packet['args']);
            break;
         case 'js2dbus_return': // we called, they are replying
            this.bridge.js2dbus_return(packet['call_id'], packet['result']);
            break;
         case 'js2dbus_error': // we called, they are replying with an error
            this.bridge.js2dbus_error(packet['call_id'], packet['error']);
            break;
         default:
            console.log("Unknown packet. Discarding.");
            break;
      }
   }
}

BridgeClient.prototype.disconnect = function()
{
   if (this.socket !== null) this.socket.close();
   this.socket = null;
}

BridgeClient.prototype.register_on_bus = function(bus_name, object_name)
{
   if (this.socket === null) {
      console.log("Can't register on bus while disconnected.");
      return;
   }  

   this.socket.send(JSON.stringify({
      cmd: 'register_on_bus',
      bus_name: bus_name,
      object_name: object_name
   }))
}

BridgeClient.prototype.register_method = function(target_function, interface,
                                                  method_name, in_sig, out_sig)
{
   if (this.socket === null) {
      console.log("Can't register methods while disconnected.");
      return;
   }
   
   iface = this.interfaces[interface];
   if (iface === undefined) iface = this.interfaces[interface] = {};
   iface[method_name] = target_function;

   this.socket.send(JSON.stringify({
      cmd: 'register_method',
      interface: interface,
      name: method_name,
      in_sig: in_sig,
      out_sig: out_sig
   }))
}

/* The called function can return its result asyncronously.
 * Essentially when the function is called "this" will be an object with a "defer"
 * property, which is a function acception one argument.
 * This function can be called at any time to have the bridge return whatever value
 * is passed to it to the other side.
 * 
 * Be careful to return either the defer function itself or null or undefined from
 * your registered function if you want the async behaviour to work (otherwise the
 * value you return will be passed throught the bridge twice. This isn't usually
 * a problem since the bridge discards the duplicate, but it's a waste of cycles).
 */
BridgeClient.prototype.dbus2js_call = function(call_id, interface, method_name, args)
{
   var iface = this.interfaces[interface]
   if (iface) {
      var method = iface[method_name]
      if (method) {
         /* We need to have this (== the brige) in a local variable so that the anonymous
          * function we create for defer will have it as part of its closure and can use it.
          * If we use "this" _inside_ the anon function, it may not work since it will be bound
          * at call time to whatever the caller decides to (e.g. window.setTimeout will set this
          * to the window itself). */
         var bridge = this;
         var defer = function(return_value) { bridge.dbus2js_return(call_id, return_value); }
         var result = method.apply({ defer: defer }, args);
         if (result == defer) return;
         else this.dbus2js_return(call_id, result);
      } else console.log("Method not found:" + method_name);
   } else console.log("Interface not found:" + interface);
}

BridgeClient.prototype.dbus2js_return = function(call_id, return_value)
{
   if (return_value === undefined || return_value === null) return;
   this.socket.send(JSON.stringify({
      cmd: 'dbus2js_return',
      call_id: call_id,
      result: return_value
   }));
}

BridgeClient.prototype.js2dbus_call = function(bus_name, object_name, interface,
                                               method, args,
                                               return_handler, error_handler)
{
   // FIXME: need to allow args to specify custom js to dbus type mappings if
   // the user wants/needs to.
   var id = this.js2dbus_last_call_id++;
   this.js2dbus_call_queue[id] = {
      success: (typeof(return_handler) !== "function") ? null : return_handler,
      error: (typeof(error_handler) !== "function") ? null : error_handler
   };
   console.log("Calling to dbus (id " + id + ") :" + bus_name + " " +
               object_name + " " + interface + "." + method + " with args:");
   console.log(args);
   this.socket.send(JSON.stringify({
      cmd: 'js2dbus_call',
      call_id: id,
      bus_name: bus_name,
      object_name: object_name,
      interface: interface,
      method: method,
      args: args
   }));
}

BridgeClient.prototype.js2dbus_return = function(call_id, return_value)
{
   var handler = this.js2dbus_call_queue[call_id].success;
   if (typeof(handler) === "function") {
      console.log("Received return for call id:" + call_id);
      handler(return_value);
   } else if (handler === undefined) {
      console.log("Received return for unknown call id:" + call_id);
      return;
   }
   delete this.js2dbus_call_queue[call_id]
}

BridgeClient.prototype.js2dbus_error = function(call_id, error)
{
   var handler = this.js2dbus_call_queue[call_id].error;
   if (typeof(handler) === "function") {
      handler(error);
   } else if (handler === null) {
      console.log("Call with id " + call_id + " returned error: " + error);
   } else if (handler === undefined) {
      console.log("Received error for unknown call id:" + call_id);
      return;
   }
   delete this.js2dbus_call_queue[call_id]
}