// Copyright (c) 2009 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "webkit/plugins/npapi/gtk_plugin_container.h"

#include <gtk/gtk.h>

#include "base/basictypes.h"

namespace webkit {
namespace npapi {

namespace {

// NOTE: This class doesn't have constructors/destructors, it is created
// through GLib's object management.
class GtkPluginContainer : public GtkSocket {
 public:
  // Sets the requested size of the widget.
  void set_size(int width, int height) {
    width_ = width;
    height_ = height;
  }

  // Casts a widget into a GtkPluginContainer, after checking the type.
  template <class T>
  static GtkPluginContainer *CastChecked(T *instance) {
    return G_TYPE_CHECK_INSTANCE_CAST(instance, GetType(), GtkPluginContainer);
  }

  // Create and register our custom container type with GTK.
  static GType GetType() {
    static GType type = 0;  // We only want to register our type once.
    if (!type) {
      static const GTypeInfo info = {
        sizeof(GtkSocketClass),
        NULL, NULL,
        static_cast<GClassInitFunc>(&ClassInit),
        NULL, NULL,
        sizeof(GtkPluginContainer),
        0, &InstanceInit,
      };
      type = g_type_register_static(GTK_TYPE_SOCKET,
                                    "GtkPluginContainer",
                                    &info,
                                    static_cast<GTypeFlags>(0));
    }
    return type;
  }

  // Implementation of the class initializer.
  static void ClassInit(gpointer klass, gpointer class_data_unusued) {
    GtkWidgetClass* widget_class = reinterpret_cast<GtkWidgetClass*>(klass);
    widget_class->size_request = &HandleSizeRequest;
  }

  // Implementation of the instance initializer (constructor).
  static void InstanceInit(GTypeInstance *instance, gpointer klass) {
    GtkPluginContainer *container = CastChecked(instance);
    container->set_size(0, 0);
  }

  // Report our allocation size during size requisition.
  static void HandleSizeRequest(GtkWidget* widget,
                                GtkRequisition* requisition) {
    GtkPluginContainer *container = CastChecked(widget);
    requisition->width = container->width_;
    requisition->height = container->height_;
  }

  int width_;
  int height_;
  DISALLOW_IMPLICIT_CONSTRUCTORS(GtkPluginContainer);
};

}  // namespace

// Create a new instance of our GTK widget object.
GtkWidget* gtk_plugin_container_new() {
  return GTK_WIDGET(g_object_new(GtkPluginContainer::GetType(), NULL));
}

void gtk_plugin_container_set_size(GtkWidget *widget, int width, int height) {
  GtkPluginContainer::CastChecked(widget)->set_size(width, height);
  // Signal the parent that the size request has changed.
  gtk_widget_queue_resize_no_redraw(widget);
}

}  // namespace npapi
}  // namespace webkit
